/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import com.google.common.base.Supplier;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeoutException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSClientAdapter;
import org.apache.hadoop.hdfs.DFSOutputStream;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.datatransfer.TrustedChannelResolver;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.DataTransferSaslUtil;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.SaslDataTransferServer;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenSecretManager;
import org.apache.hadoop.hdfs.security.token.block.DataEncryptionKey;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

@RunWith(value=Parameterized.class)
public class TestEncryptedTransfer {
    private static final Log LOG = LogFactory.getLog(TestEncryptedTransfer.class);
    private static final String PLAIN_TEXT = "this is very secret plain text";
    private static final Path TEST_PATH = new Path("/non-encrypted-file");
    private MiniDFSCluster cluster;
    private Configuration conf;
    private FileSystem fs;
    String resolverClazz;

    @Parameterized.Parameters
    public static Collection<Object[]> data() {
        ArrayList<Object[]> params = new ArrayList<Object[]>();
        params.add(new Object[]{null});
        params.add(new Object[]{"org.apache.hadoop.hdfs.TestEncryptedTransfer$TestTrustedChannelResolver"});
        return params;
    }

    private void setEncryptionConfigKeys() {
        this.conf.setBoolean("dfs.encrypt.data.transfer", true);
        this.conf.setBoolean("dfs.block.access.token.enable", true);
        if (this.resolverClazz != null) {
            this.conf.set("dfs.trustedchannel.resolver.class", this.resolverClazz);
        }
    }

    private static FileSystem getFileSystem(Configuration conf) throws IOException {
        Configuration localConf = new Configuration(conf);
        localConf.setBoolean("dfs.encrypt.data.transfer", false);
        localConf.unset("dfs.encrypt.data.transfer.algorithm");
        return FileSystem.get((Configuration)localConf);
    }

    public TestEncryptedTransfer(String resolverClazz) {
        LogManager.getLogger(SaslDataTransferServer.class).setLevel(Level.DEBUG);
        LogManager.getLogger(DataTransferSaslUtil.class).setLevel(Level.DEBUG);
        this.cluster = null;
        this.conf = null;
        this.fs = null;
        this.resolverClazz = resolverClazz;
    }

    @Before
    public void setup() throws IOException {
        this.conf = new Configuration();
    }

    @After
    public void teardown() throws IOException {
        if (this.fs != null) {
            this.fs.close();
        }
        if (this.cluster != null) {
            this.cluster.shutdown();
        }
    }

    private FileChecksum writeUnencryptedAndThenRestartEncryptedCluster() throws IOException {
        this.cluster = new MiniDFSCluster.Builder(this.conf).build();
        this.fs = TestEncryptedTransfer.getFileSystem(this.conf);
        TestEncryptedTransfer.writeTestDataToFile(this.fs);
        Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
        FileChecksum checksum = this.fs.getFileChecksum(TEST_PATH);
        this.fs.close();
        this.cluster.shutdown();
        this.setEncryptionConfigKeys();
        this.cluster = new MiniDFSCluster.Builder(this.conf).manageDataDfsDirs(false).manageNameDfsDirs(false).format(false).startupOption(HdfsServerConstants.StartupOption.REGULAR).build();
        this.fs = TestEncryptedTransfer.getFileSystem(this.conf);
        return checksum;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testEncryptedRead(String algorithm, String cipherSuite, boolean matchLog, boolean readAfterRestart) throws IOException {
        this.conf.set("dfs.encrypt.data.transfer.algorithm", algorithm);
        this.conf.set("dfs.encrypt.data.transfer.cipher.suites", cipherSuite);
        FileChecksum checksum = this.writeUnencryptedAndThenRestartEncryptedCluster();
        GenericTestUtils.LogCapturer logs = GenericTestUtils.LogCapturer.captureLogs((Log)LogFactory.getLog(SaslDataTransferServer.class));
        GenericTestUtils.LogCapturer logs1 = GenericTestUtils.LogCapturer.captureLogs((Log)LogFactory.getLog(DataTransferSaslUtil.class));
        try {
            Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
            Assert.assertEquals((Object)checksum, (Object)this.fs.getFileChecksum(TEST_PATH));
        }
        finally {
            logs.stopCapturing();
            logs1.stopCapturing();
        }
        if (this.resolverClazz == null) {
            if (matchLog) {
                GenericTestUtils.assertMatches((String)logs.getOutput(), (String)"Server using cipher suite");
                GenericTestUtils.assertMatches((String)logs1.getOutput(), (String)"Creating IOStreamPair of CryptoInputStream and CryptoOutputStream.");
            } else {
                GenericTestUtils.assertDoesNotMatch((String)logs.getOutput(), (String)"Server using cipher suite");
                GenericTestUtils.assertDoesNotMatch((String)logs1.getOutput(), (String)"Creating IOStreamPair of CryptoInputStream and CryptoOutputStream.");
            }
        }
        if (readAfterRestart) {
            this.cluster.restartNameNode(new String[0]);
            this.fs = TestEncryptedTransfer.getFileSystem(this.conf);
            Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
            Assert.assertEquals((Object)checksum, (Object)this.fs.getFileChecksum(TEST_PATH));
        }
    }

    @Test
    public void testEncryptedReadDefaultAlgorithmCipherSuite() throws IOException {
        this.testEncryptedRead("", "", false, false);
    }

    @Test
    public void testEncryptedReadWithRC4() throws IOException {
        this.testEncryptedRead("rc4", "", false, false);
    }

    @Test
    public void testEncryptedReadWithAES() throws IOException {
        this.testEncryptedRead("", "AES/CTR/NoPadding", true, false);
    }

    @Test
    public void testEncryptedReadAfterNameNodeRestart() throws IOException {
        this.testEncryptedRead("", "", false, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testClientThatDoesNotSupportEncryption() throws IOException {
        this.conf.setInt("dfs.client.retry.window.base", 10);
        this.writeUnencryptedAndThenRestartEncryptedCluster();
        DFSClient client = DFSClientAdapter.getDFSClient((DistributedFileSystem)this.fs);
        DFSClient spyClient = (DFSClient)Mockito.spy((Object)client);
        ((DFSClient)Mockito.doReturn((Object)false).when((Object)spyClient)).shouldEncryptData();
        DFSClientAdapter.setDFSClient((DistributedFileSystem)this.fs, spyClient);
        GenericTestUtils.LogCapturer logs = GenericTestUtils.LogCapturer.captureLogs((Log)LogFactory.getLog(DataNode.class));
        try {
            Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
            if (this.resolverClazz != null && !this.resolverClazz.endsWith("TestTrustedChannelResolver")) {
                Assert.fail((String)"Should not have been able to read without encryption enabled.");
            }
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains((String)"Could not obtain block:", (Throwable)ioe);
        }
        finally {
            logs.stopCapturing();
        }
        if (this.resolverClazz == null) {
            GenericTestUtils.assertMatches((String)logs.getOutput(), (String)"Failed to read expected encryption handshake from client at");
        }
    }

    @Test
    public void testLongLivedReadClientAfterRestart() throws IOException {
        FileChecksum checksum = this.writeUnencryptedAndThenRestartEncryptedCluster();
        Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
        Assert.assertEquals((Object)checksum, (Object)this.fs.getFileChecksum(TEST_PATH));
        this.cluster.restartNameNode(new String[0]);
        Assert.assertTrue((boolean)this.cluster.restartDataNode(0));
        Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
        Assert.assertEquals((Object)checksum, (Object)this.fs.getFileChecksum(TEST_PATH));
    }

    @Test
    public void testLongLivedWriteClientAfterRestart() throws IOException {
        this.setEncryptionConfigKeys();
        this.cluster = new MiniDFSCluster.Builder(this.conf).build();
        this.fs = TestEncryptedTransfer.getFileSystem(this.conf);
        TestEncryptedTransfer.writeTestDataToFile(this.fs);
        Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
        this.cluster.restartNameNode(new String[0]);
        Assert.assertTrue((boolean)this.cluster.restartDataNodes());
        this.cluster.waitActive();
        TestEncryptedTransfer.writeTestDataToFile(this.fs);
        Assert.assertEquals((Object)"this is very secret plain textthis is very secret plain text", (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
    }

    @Test
    public void testLongLivedClient() throws IOException, InterruptedException {
        FileChecksum checksum = this.writeUnencryptedAndThenRestartEncryptedCluster();
        BlockTokenSecretManager btsm = this.cluster.getNamesystem().getBlockManager().getBlockTokenSecretManager();
        btsm.setKeyUpdateIntervalForTesting(2000L);
        btsm.setTokenLifetime(2000L);
        btsm.clearAllKeysForTesting();
        Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
        Assert.assertEquals((Object)checksum, (Object)this.fs.getFileChecksum(TEST_PATH));
        LOG.info((Object)"Sleeping so that encryption keys expire...");
        Thread.sleep(15000L);
        LOG.info((Object)"Done sleeping.");
        Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
        Assert.assertEquals((Object)checksum, (Object)this.fs.getFileChecksum(TEST_PATH));
    }

    @Test
    public void testLongLivedClientPipelineRecovery() throws IOException, InterruptedException, TimeoutException {
        if (this.resolverClazz != null) {
            return;
        }
        int numDataNodes = 4;
        this.conf.setBoolean("dfs.namenode.replication.considerLoad", false);
        this.setEncryptionConfigKeys();
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(numDataNodes).build();
        this.fs = TestEncryptedTransfer.getFileSystem(this.conf);
        DFSClient client = DFSClientAdapter.getDFSClient((DistributedFileSystem)this.fs);
        DFSClient spyClient = (DFSClient)Mockito.spy((Object)client);
        DFSClientAdapter.setDFSClient((DistributedFileSystem)this.fs, spyClient);
        TestEncryptedTransfer.writeTestDataToFile(this.fs);
        BlockTokenSecretManager btsm = this.cluster.getNamesystem().getBlockManager().getBlockTokenSecretManager();
        btsm.setKeyUpdateIntervalForTesting(2000L);
        btsm.setTokenLifetime(2000L);
        btsm.clearAllKeysForTesting();
        LOG.info((Object)"Wait until encryption keys become invalid...");
        final DataEncryptionKey encryptionKey = spyClient.getEncryptionKey();
        ArrayList<DataNode> dataNodes = this.cluster.getDataNodes();
        for (final DataNode dn : dataNodes) {
            GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

                public Boolean get() {
                    return !dn.getBlockPoolTokenSecretManager().get(encryptionKey.blockPoolId).hasKey(encryptionKey.keyId);
                }
            }, (int)100, (int)30000);
        }
        LOG.info((Object)"The encryption key is invalid on all nodes now.");
        try (FSDataOutputStream out = this.fs.append(TEST_PATH);){
            DFSOutputStream dfstream = (DFSOutputStream)out.getWrappedStream();
            DatanodeInfo[] targets = dfstream.getPipeline();
            this.cluster.stopDataNode(targets[0].getXferAddr());
            out.write(PLAIN_TEXT.getBytes());
            out.hflush();
            Assert.assertFalse((String)"The first datanode in the pipeline was not replaced.", (boolean)Arrays.asList(dfstream.getPipeline()).contains(targets[0]));
        }
        ((DFSClient)Mockito.verify((Object)spyClient, (VerificationMode)Mockito.times((int)1))).clearDataEncryptionKey();
    }

    @Test
    public void testEncryptedWriteWithOneDn() throws IOException {
        this.testEncryptedWrite(1);
    }

    @Test
    public void testEncryptedWriteWithTwoDns() throws IOException {
        this.testEncryptedWrite(2);
    }

    @Test
    public void testEncryptedWriteWithMultipleDns() throws IOException {
        this.testEncryptedWrite(10);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testEncryptedWrite(int numDns) throws IOException {
        this.setEncryptionConfigKeys();
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(numDns).build();
        this.fs = TestEncryptedTransfer.getFileSystem(this.conf);
        GenericTestUtils.LogCapturer logs = GenericTestUtils.LogCapturer.captureLogs((Log)LogFactory.getLog(SaslDataTransferServer.class));
        GenericTestUtils.LogCapturer logs1 = GenericTestUtils.LogCapturer.captureLogs((Log)LogFactory.getLog(DataTransferSaslUtil.class));
        try {
            TestEncryptedTransfer.writeTestDataToFile(this.fs);
        }
        finally {
            logs.stopCapturing();
            logs1.stopCapturing();
        }
        Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
        if (this.resolverClazz == null) {
            GenericTestUtils.assertDoesNotMatch((String)logs.getOutput(), (String)"Server using cipher suite");
            GenericTestUtils.assertDoesNotMatch((String)logs1.getOutput(), (String)"Creating IOStreamPair of CryptoInputStream and CryptoOutputStream.");
        }
    }

    @Test
    public void testEncryptedAppend() throws IOException {
        this.setEncryptionConfigKeys();
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(3).build();
        this.fs = TestEncryptedTransfer.getFileSystem(this.conf);
        TestEncryptedTransfer.writeTestDataToFile(this.fs);
        Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
        TestEncryptedTransfer.writeTestDataToFile(this.fs);
        Assert.assertEquals((Object)"this is very secret plain textthis is very secret plain text", (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
    }

    @Test
    public void testEncryptedAppendRequiringBlockTransfer() throws IOException {
        this.setEncryptionConfigKeys();
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(4).build();
        this.fs = TestEncryptedTransfer.getFileSystem(this.conf);
        TestEncryptedTransfer.writeTestDataToFile(this.fs);
        Assert.assertEquals((Object)PLAIN_TEXT, (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
        FSDataInputStream in = this.fs.open(TEST_PATH);
        List<LocatedBlock> locatedBlocks = DFSTestUtil.getAllBlocks(in);
        in.close();
        Assert.assertEquals((long)1L, (long)locatedBlocks.size());
        Assert.assertEquals((long)3L, (long)locatedBlocks.get(0).getLocations().length);
        DataNode dn = this.cluster.getDataNode(locatedBlocks.get(0).getLocations()[0].getIpcPort());
        dn.shutdown();
        TestEncryptedTransfer.writeTestDataToFile(this.fs);
        Assert.assertEquals((Object)"this is very secret plain textthis is very secret plain text", (Object)DFSTestUtil.readFile(this.fs, TEST_PATH));
    }

    private static void writeTestDataToFile(FileSystem fs) throws IOException {
        FSDataOutputStream out = null;
        out = !fs.exists(TEST_PATH) ? fs.create(TEST_PATH) : fs.append(TEST_PATH);
        out.write(PLAIN_TEXT.getBytes());
        out.close();
    }

    static class TestTrustedChannelResolver
    extends TrustedChannelResolver {
        TestTrustedChannelResolver() {
        }

        public boolean isTrusted() {
            return true;
        }

        public boolean isTrusted(InetAddress peerAddress) {
            return true;
        }
    }
}

