/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.services.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.netflix.genie.common.dto.JobStatus;
import com.netflix.genie.common.exceptions.GenieNotFoundException;
import com.netflix.genie.common.internal.dto.JobDirectoryManifest;
import com.netflix.genie.common.internal.exceptions.unchecked.GenieJobNotFoundException;
import com.netflix.genie.common.util.GenieObjectMapper;
import com.netflix.genie.web.resources.writers.DefaultDirectoryWriter;
import com.netflix.genie.web.services.AgentFileStreamService;
import com.netflix.genie.web.services.JobDirectoryServerService;
import com.netflix.genie.web.services.JobFileService;
import com.netflix.genie.web.services.JobPersistenceService;
import io.micrometer.core.instrument.MeterRegistry;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Optional;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.converter.ResourceHttpMessageConverter;
import org.springframework.http.converter.ResourceRegionHttpMessageConverter;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;

public class JobDirectoryServerServiceImpl
implements JobDirectoryServerService {
    private static final Logger log = LoggerFactory.getLogger(JobDirectoryServerServiceImpl.class);
    private static final String SLASH = "/";
    private final ResourceLoader resourceLoader;
    private final JobPersistenceService jobPersistenceService;
    private final JobFileService jobFileService;
    private final AgentFileStreamService agentFileStreamService;
    private final MeterRegistry meterRegistry;
    private final LoadingCache<String, ManifestCacheValue> manifestCache;
    private final GenieResourceHandler.Factory genieResourceHandlerFactory;

    public JobDirectoryServerServiceImpl(ResourceLoader resourceLoader, JobPersistenceService jobPersistenceService, JobFileService jobFileService, AgentFileStreamService agentFileStreamService, MeterRegistry meterRegistry) {
        this(resourceLoader, jobPersistenceService, jobFileService, agentFileStreamService, meterRegistry, new GenieResourceHandler.Factory());
    }

    @VisibleForTesting
    JobDirectoryServerServiceImpl(final ResourceLoader resourceLoader, final JobPersistenceService jobPersistenceService, JobFileService jobFileService, AgentFileStreamService agentFileStreamService, MeterRegistry meterRegistry, GenieResourceHandler.Factory genieResourceHandlerFactory) {
        this.resourceLoader = resourceLoader;
        this.jobPersistenceService = jobPersistenceService;
        this.jobFileService = jobFileService;
        this.agentFileStreamService = agentFileStreamService;
        this.meterRegistry = meterRegistry;
        this.genieResourceHandlerFactory = genieResourceHandlerFactory;
        this.manifestCache = CacheBuilder.newBuilder().maximumSize(50L).recordStats().build((CacheLoader)new CacheLoader<String, ManifestCacheValue>(){

            public ManifestCacheValue load(String key) throws Exception {
                String archiveLocation = jobPersistenceService.getJobArchiveLocation(key).orElseThrow(() -> new JobNotArchivedException("Job " + key + " wasn't archived"));
                URI jobDirectoryRoot = new URI(archiveLocation + JobDirectoryServerServiceImpl.SLASH).normalize();
                URI manifestLocation = StringUtils.isBlank((CharSequence)"genie") ? jobDirectoryRoot.resolve("manifest.json").normalize() : jobDirectoryRoot.resolve("genie/").resolve("manifest.json").normalize();
                Resource manifestResource = resourceLoader.getResource(manifestLocation.toString());
                if (!manifestResource.exists()) {
                    throw new GenieNotFoundException("No job manifest exists at " + manifestLocation);
                }
                JobDirectoryManifest manifest = (JobDirectoryManifest)GenieObjectMapper.getMapper().readValue(manifestResource.getInputStream(), JobDirectoryManifest.class);
                return new ManifestCacheValue(manifest, jobDirectoryRoot);
            }
        });
    }

    @Override
    public void serveResource(String jobId, URL baseUrl, String relativePath, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
        URI baseUri;
        boolean isV4;
        JobStatus jobStatus;
        try {
            jobStatus = this.jobPersistenceService.getJobStatus(jobId);
        }
        catch (GenieNotFoundException e) {
            log.error(e.getMessage(), (Throwable)e);
            response.sendError(HttpStatus.NOT_FOUND.value(), e.getMessage());
            return;
        }
        try {
            isV4 = this.jobPersistenceService.isV4(jobId);
        }
        catch (GenieJobNotFoundException nfe) {
            log.error(nfe.getMessage(), (Throwable)nfe);
            response.sendError(HttpStatus.NOT_FOUND.value(), nfe.getMessage());
            return;
        }
        try {
            baseUri = new URI(baseUrl.toString() + SLASH).normalize();
        }
        catch (URISyntaxException e) {
            log.error(e.getMessage(), (Throwable)e);
            response.sendError(HttpStatus.INTERNAL_SERVER_ERROR.value(), "Unable to convert " + baseUrl + " to valid URI");
            return;
        }
        if (jobStatus.isActive() && isV4) {
            URI jobDirRoot;
            Optional<JobDirectoryManifest> manifest = this.agentFileStreamService.getManifest(jobId);
            if (!manifest.isPresent()) {
                log.error("Manifest not found for active job: {}", (Object)jobId);
                response.sendError(HttpStatus.SERVICE_UNAVAILABLE.value(), "Could not load manifest for job: " + jobId);
                return;
            }
            try {
                jobDirRoot = new URI("agent", jobId, SLASH, null);
            }
            catch (URISyntaxException e) {
                log.error(e.getMessage(), (Throwable)e);
                response.sendError(HttpStatus.INTERNAL_SERVER_ERROR.value(), e.getMessage());
                return;
            }
            this.handleRequest(baseUri, relativePath, request, response, manifest.get(), jobDirRoot);
        } else if (jobStatus.isActive()) {
            URI jobDirRoot;
            Resource jobDir = this.jobFileService.getJobFileAsResource(jobId, "");
            if (!jobDir.exists()) {
                log.error("Job directory {} doesn't exist. Unable to serve job contents.", (Object)jobDir);
                response.sendError(HttpStatus.NOT_FOUND.value());
                return;
            }
            try {
                jobDirRoot = new URI(jobDir.getURI().toString() + SLASH).normalize();
            }
            catch (URISyntaxException e) {
                log.error(e.getMessage(), (Throwable)e);
                response.sendError(HttpStatus.INTERNAL_SERVER_ERROR.value(), e.getMessage());
                return;
            }
            Path jobDirPath = Paths.get(jobDirRoot);
            JobDirectoryManifest manifest = new JobDirectoryManifest(jobDirPath, false);
            this.handleRequest(baseUri, relativePath, request, response, manifest, jobDirRoot);
        } else {
            URI jobDirRoot;
            JobDirectoryManifest manifest;
            try {
                ManifestCacheValue cacheValue = (ManifestCacheValue)this.manifestCache.get((Object)jobId);
                manifest = cacheValue.getManifest();
                jobDirRoot = cacheValue.getJobDirectoryRoot();
            }
            catch (Exception e) {
                if (e.getCause() instanceof JobNotArchivedException) {
                    log.error(e.getCause().getMessage(), e.getCause());
                    response.sendError(HttpStatus.NOT_FOUND.value(), e.getCause().getMessage());
                } else {
                    log.error(e.getMessage(), (Throwable)e);
                    response.sendError(HttpStatus.INTERNAL_SERVER_ERROR.value(), e.getMessage());
                }
                return;
            }
            this.handleRequest(baseUri, relativePath, request, response, manifest, jobDirRoot);
        }
    }

    private void handleRequest(URI baseUri, String relativePath, HttpServletRequest request, HttpServletResponse response, JobDirectoryManifest manifest, URI jobDirectoryRoot) throws IOException, ServletException {
        log.debug("Handle request, baseUri: '{}', relpath: '{}', jobRootUri: '{}'", new Object[]{baseUri, relativePath, jobDirectoryRoot});
        Optional entryOptional = manifest.getEntry(relativePath);
        if (!entryOptional.isPresent()) {
            log.error("No such entry in job manifest: {}", (Object)relativePath);
            response.sendError(HttpStatus.NOT_FOUND.value(), "Not found: " + relativePath);
            return;
        }
        JobDirectoryManifest.ManifestEntry entry = (JobDirectoryManifest.ManifestEntry)entryOptional.get();
        if (entry.isDirectory()) {
            DefaultDirectoryWriter.Directory directory = new DefaultDirectoryWriter.Directory();
            ArrayList files = Lists.newArrayList();
            ArrayList directories = Lists.newArrayList();
            try {
                entry.getParent().ifPresent(parentPath -> {
                    JobDirectoryManifest.ManifestEntry parentEntry = (JobDirectoryManifest.ManifestEntry)manifest.getEntry(parentPath).orElseThrow(IllegalArgumentException::new);
                    directory.setParent(this.createEntry(parentEntry, baseUri));
                });
                for (String childPath : entry.getChildren()) {
                    JobDirectoryManifest.ManifestEntry childEntry = (JobDirectoryManifest.ManifestEntry)manifest.getEntry(childPath).orElseThrow(IllegalArgumentException::new);
                    if (childEntry.isDirectory()) {
                        directories.add(this.createEntry(childEntry, baseUri));
                        continue;
                    }
                    files.add(this.createEntry(childEntry, baseUri));
                }
            }
            catch (IllegalArgumentException iae) {
                log.error("Encountered unexpected problem traversing the manifest for directory entry {}", (Object)entry, (Object)iae);
                response.sendError(HttpStatus.INTERNAL_SERVER_ERROR.value());
                return;
            }
            directories.sort(Comparator.comparing(DefaultDirectoryWriter.Entry::getName));
            files.sort(Comparator.comparing(DefaultDirectoryWriter.Entry::getName));
            directory.setDirectories(directories);
            directory.setFiles(files);
            String accept = request.getHeader("Accept");
            if (accept != null && accept.contains("text/html")) {
                response.setContentType("text/html");
                response.getOutputStream().write(DefaultDirectoryWriter.directoryToHTML(entry.getName(), directory).getBytes(StandardCharsets.UTF_8));
            } else {
                response.setContentType("application/json");
                GenieObjectMapper.getMapper().writeValue((OutputStream)response.getOutputStream(), (Object)directory);
            }
        } else {
            URI location = jobDirectoryRoot.resolve(entry.getPath());
            log.debug("Get resource: {}", (Object)location);
            Resource jobResource = this.resourceLoader.getResource(location.toString());
            String mediaType = entry.getMimeType().orElse("text/plain");
            ResourceHttpRequestHandler handler = this.genieResourceHandlerFactory.get(mediaType, jobResource);
            handler.handleRequest(request, response);
        }
    }

    private DefaultDirectoryWriter.Entry createEntry(JobDirectoryManifest.ManifestEntry manifestEntry, URI baseUri) {
        DefaultDirectoryWriter.Entry entry = new DefaultDirectoryWriter.Entry();
        if (manifestEntry.isDirectory()) {
            entry.setName(manifestEntry.getName().endsWith(SLASH) ? manifestEntry.getName() : manifestEntry.getName() + SLASH);
        } else {
            entry.setName(manifestEntry.getName());
        }
        entry.setUrl(baseUri.resolve(manifestEntry.getPath()).toString());
        entry.setSize(manifestEntry.getSize());
        entry.setLastModified(manifestEntry.getLastModifiedTime());
        return entry;
    }

    private static class JobNotArchivedException
    extends RuntimeException {
        JobNotArchivedException(String message) {
            super(message);
        }
    }

    private static class ManifestCacheValue {
        private final JobDirectoryManifest manifest;
        private final URI jobDirectoryRoot;

        public JobDirectoryManifest getManifest() {
            return this.manifest;
        }

        public URI getJobDirectoryRoot() {
            return this.jobDirectoryRoot;
        }

        public ManifestCacheValue(JobDirectoryManifest manifest, URI jobDirectoryRoot) {
            this.manifest = manifest;
            this.jobDirectoryRoot = jobDirectoryRoot;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof ManifestCacheValue)) {
                return false;
            }
            ManifestCacheValue other = (ManifestCacheValue)o;
            if (!other.canEqual(this)) {
                return false;
            }
            JobDirectoryManifest this$manifest = this.manifest;
            JobDirectoryManifest other$manifest = other.manifest;
            if (this$manifest == null ? other$manifest != null : !this$manifest.equals(other$manifest)) {
                return false;
            }
            URI this$jobDirectoryRoot = this.jobDirectoryRoot;
            URI other$jobDirectoryRoot = other.jobDirectoryRoot;
            return !(this$jobDirectoryRoot == null ? other$jobDirectoryRoot != null : !((Object)this$jobDirectoryRoot).equals(other$jobDirectoryRoot));
        }

        protected boolean canEqual(Object other) {
            return other instanceof ManifestCacheValue;
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            JobDirectoryManifest $manifest = this.manifest;
            result = result * 59 + ($manifest == null ? 43 : $manifest.hashCode());
            URI $jobDirectoryRoot = this.jobDirectoryRoot;
            result = result * 59 + ($jobDirectoryRoot == null ? 43 : ((Object)$jobDirectoryRoot).hashCode());
            return result;
        }

        public String toString() {
            return "JobDirectoryServerServiceImpl.ManifestCacheValue(manifest=" + this.manifest + ", jobDirectoryRoot=" + this.jobDirectoryRoot + ")";
        }
    }

    private static class GenieResourceHandler
    extends ResourceHttpRequestHandler {
        private static final ResourceHttpMessageConverter RESOURCE_HTTP_MESSAGE_CONVERTER = new ResourceHttpMessageConverter();
        private static final ResourceRegionHttpMessageConverter RESOURCE_REGION_HTTP_MESSAGE_CONVERTER = new ResourceRegionHttpMessageConverter();
        private final MediaType mediaType;
        private final Resource jobResource;

        GenieResourceHandler(String mediaType, Resource jobResource) {
            this.mediaType = MediaType.parseMediaType((String)mediaType);
            this.jobResource = jobResource;
            this.setResourceHttpMessageConverter(RESOURCE_HTTP_MESSAGE_CONVERTER);
            this.setResourceRegionHttpMessageConverter(RESOURCE_REGION_HTTP_MESSAGE_CONVERTER);
        }

        protected Resource getResource(HttpServletRequest request) throws IOException {
            return this.jobResource;
        }

        protected MediaType getMediaType(HttpServletRequest request, Resource resource) {
            return this.mediaType;
        }

        private static class Factory {
            private Factory() {
            }

            ResourceHttpRequestHandler get(String mediaType, Resource jobResource) {
                return new GenieResourceHandler(mediaType, jobResource);
            }
        }
    }
}

