/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.services.impl;

import com.amazonaws.services.s3.AmazonS3URI;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import com.netflix.genie.common.exceptions.GenieBadRequestException;
import com.netflix.genie.common.exceptions.GenieException;
import com.netflix.genie.common.exceptions.GenieServerException;
import com.netflix.genie.common.internal.aws.s3.S3ClientFactory;
import com.netflix.genie.web.properties.S3FileTransferProperties;
import com.netflix.genie.web.services.FileTransfer;
import com.netflix.genie.web.util.MetricsUtils;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.MeterRegistry;
import java.io.File;
import java.util.HashSet;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3FileTransferImpl
implements FileTransfer {
    private static final Logger log = LoggerFactory.getLogger(S3FileTransferImpl.class);
    static final String DOWNLOAD_TIMER_NAME = "genie.files.s3.download.timer";
    static final String UPLOAD_TIMER_NAME = "genie.files.s3.upload.timer";
    static final String STRICT_VALIDATION_COUNTER_NAME = "genie.files.s3.failStrictValidation.counter";
    private static final String GET_METADATA_TIMER_NAME = "genie.files.s3.getObjectMetadata.timer";
    private static final Pattern S3_PREFIX_PATTERN = Pattern.compile("^s3[n]?://.*$");
    private static final Pattern S3_BUCKET_PATTERN = Pattern.compile("^[a-z0-9][a-z0-9.\\-]{1,61}[a-z0-9]$");
    private static final Pattern S3_KEY_PATTERN = Pattern.compile("^[0-9a-zA-Z!\\-_.*'()]+(?:/[0-9a-zA-Z!\\-_.*'()]+)*$");
    private final MeterRegistry registry;
    private final S3ClientFactory s3ClientFactory;
    private final S3FileTransferProperties s3FileTransferProperties;
    private final Counter urlFailingStrictValidationCounter;

    public S3FileTransferImpl(@NotNull S3ClientFactory s3ClientFactory, @NotNull MeterRegistry registry, @NotNull S3FileTransferProperties s3FileTransferProperties) {
        this.s3ClientFactory = s3ClientFactory;
        this.registry = registry;
        this.urlFailingStrictValidationCounter = registry.counter(STRICT_VALIDATION_COUNTER_NAME, new String[0]);
        this.s3FileTransferProperties = s3FileTransferProperties;
    }

    @Override
    public boolean isValid(String fileName) throws GenieException {
        log.debug("Called with file name {}", (Object)fileName);
        try {
            this.getS3Uri(fileName);
            return true;
        }
        catch (GenieBadRequestException e) {
            log.error("Invalid S3 path {} ({})", (Object)fileName, (Object)e.getMessage());
            return false;
        }
    }

    @Override
    public void getFile(@NotBlank(message="Source file path cannot be empty.") @NotBlank(message="Source file path cannot be empty.") String srcRemotePath, @NotBlank(message="Destination local path cannot be empty") @NotBlank(message="Destination local path cannot be empty") String dstLocalPath) throws GenieException {
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet();
        try {
            log.debug("Called with src path {} and destination path {}", (Object)srcRemotePath, (Object)dstLocalPath);
            AmazonS3URI s3Uri = this.getS3Uri(srcRemotePath);
            try {
                this.s3ClientFactory.getClient(s3Uri).getObject(new GetObjectRequest(s3Uri.getBucket(), s3Uri.getKey()), new File(dstLocalPath));
            }
            catch (AmazonS3Exception ase) {
                log.error("Error fetching file {} from s3 due to exception {}", (Object)srcRemotePath, (Object)ase.toString());
                throw new GenieServerException("Error downloading file from s3. Filename: " + srcRemotePath, (Throwable)ase);
            }
            MetricsUtils.addSuccessTags(tags);
        }
        catch (Throwable t) {
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw t;
        }
        finally {
            this.registry.timer(DOWNLOAD_TIMER_NAME, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    @Override
    public void putFile(@NotBlank(message="Source local path cannot be empty.") @NotBlank(message="Source local path cannot be empty.") String srcLocalPath, @NotBlank(message="Destination remote path cannot be empty") @NotBlank(message="Destination remote path cannot be empty") String dstRemotePath) throws GenieException {
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet();
        try {
            log.debug("Called with src path {} and destination path {}", (Object)srcLocalPath, (Object)dstRemotePath);
            AmazonS3URI s3Uri = this.getS3Uri(dstRemotePath);
            try {
                this.s3ClientFactory.getClient(s3Uri).putObject(s3Uri.getBucket(), s3Uri.getKey(), new File(srcLocalPath));
            }
            catch (AmazonS3Exception ase) {
                log.error("Error posting file {} to s3 due to exception {}", (Object)dstRemotePath, (Object)ase.toString());
                throw new GenieServerException("Error uploading file to s3. Filename: " + dstRemotePath, (Throwable)ase);
            }
            MetricsUtils.addSuccessTags(tags);
        }
        catch (Throwable t) {
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw t;
        }
        finally {
            this.registry.timer(UPLOAD_TIMER_NAME, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    @Override
    public long getLastModifiedTime(String path) throws GenieException {
        long lastModTime;
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet();
        try {
            AmazonS3URI s3Uri = this.getS3Uri(path);
            try {
                ObjectMetadata o = this.s3ClientFactory.getClient(s3Uri).getObjectMetadata(s3Uri.getBucket(), s3Uri.getKey());
                lastModTime = o.getLastModified().getTime();
            }
            catch (Exception ase) {
                String message = String.format("Failed getting the metadata of the s3 file %s", path);
                log.error(message);
                throw new GenieServerException(message, (Throwable)ase);
            }
            MetricsUtils.addSuccessTags(tags);
        }
        catch (Throwable t) {
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw t;
        }
        finally {
            this.registry.timer(GET_METADATA_TIMER_NAME, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
        return lastModTime;
    }

    @VisibleForTesting
    AmazonS3URI getS3Uri(String path) throws GenieBadRequestException {
        AmazonS3URI uri;
        if (!S3_PREFIX_PATTERN.matcher(path).matches()) {
            throw new GenieBadRequestException(String.format("Invalid prefix in path for s3 file %s", path));
        }
        String adjustedPath = path.replaceFirst("^s3n://", "s3://");
        try {
            uri = new AmazonS3URI(adjustedPath, false);
        }
        catch (IllegalArgumentException e) {
            throw new GenieBadRequestException(String.format("Invalid path for s3 file %s", path), (Throwable)e);
        }
        if (StringUtils.isBlank((CharSequence)uri.getBucket()) || StringUtils.isBlank((CharSequence)uri.getKey())) {
            throw new GenieBadRequestException(String.format("Invalid blank components in path for s3 file %s", path));
        }
        boolean bucketPassesStrictValidation = S3_BUCKET_PATTERN.matcher(uri.getBucket()).matches();
        boolean keyPassesStrictValidation = S3_KEY_PATTERN.matcher(uri.getKey()).matches();
        if (!bucketPassesStrictValidation || !keyPassesStrictValidation) {
            if (this.s3FileTransferProperties.isStrictUrlCheckEnabled()) {
                throw new GenieBadRequestException(String.format("Invalid bucket %s in path for s3 file %s", uri.getBucket(), path));
            }
            log.warn("S3 URL fails strict validation: \"{}\"", (Object)path);
            this.urlFailingStrictValidationCounter.increment();
        }
        return uri;
    }
}

