/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.tasks.node;

import com.netflix.genie.common.dto.Job;
import com.netflix.genie.common.exceptions.GenieException;
import com.netflix.genie.common.internal.jobs.JobConstants;
import com.netflix.genie.web.data.services.JobSearchService;
import com.netflix.genie.web.properties.DiskCleanupProperties;
import com.netflix.genie.web.properties.JobsProperties;
import com.netflix.genie.web.tasks.TaskUtils;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.MeterRegistry;
import java.io.File;
import java.io.IOException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import javax.validation.constraints.NotNull;
import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.Executor;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.support.CronTrigger;

public class DiskCleanupTask
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(DiskCleanupTask.class);
    private final DiskCleanupProperties properties;
    private final File jobsDir;
    private final JobSearchService jobSearchService;
    private final boolean runAsUser;
    private final Executor processExecutor;
    private final AtomicLong numberOfDeletedJobDirs;
    private final AtomicLong numberOfDirsUnableToDelete;
    private final Counter unableToGetJobCounter;
    private final Counter unableToDeleteJobDirCounter;

    public DiskCleanupTask(@NotNull DiskCleanupProperties properties, @NotNull TaskScheduler scheduler, @NotNull Resource jobsDir, @NotNull JobSearchService jobSearchService, @NotNull JobsProperties jobsProperties, @NotNull Executor processExecutor, @NotNull MeterRegistry registry) throws IOException {
        if (!jobsDir.exists()) {
            throw new IOException("Jobs dir " + jobsDir + " doesn't exist. Unable to create task to cleanup.");
        }
        this.properties = properties;
        this.jobsDir = jobsDir.getFile();
        this.jobSearchService = jobSearchService;
        this.runAsUser = jobsProperties.getUsers().isRunAsUserEnabled();
        this.processExecutor = processExecutor;
        this.numberOfDeletedJobDirs = (AtomicLong)registry.gauge("genie.tasks.diskCleanup.numberDeletedJobDirs.gauge", (Number)new AtomicLong());
        this.numberOfDirsUnableToDelete = (AtomicLong)registry.gauge("genie.tasks.diskCleanup.numberDirsUnableToDelete.gauge", (Number)new AtomicLong());
        this.unableToGetJobCounter = registry.counter("genie.tasks.diskCleanup.unableToGetJobs.rate", new String[0]);
        this.unableToDeleteJobDirCounter = registry.counter("genie.tasks.diskCleanup.unableToDeleteJobsDir.rate", new String[0]);
        if (this.runAsUser && !SystemUtils.IS_OS_UNIX) {
            log.error("System is not UNIX like. Unable to schedule disk cleanup due to needing Unix commands");
        } else {
            CronTrigger trigger = new CronTrigger(properties.getExpression(), JobConstants.UTC);
            scheduler.schedule((Runnable)this, (Trigger)trigger);
        }
    }

    @Override
    public void run() {
        log.info("Running disk cleanup task...");
        File[] jobDirs = this.jobsDir.listFiles();
        if (jobDirs == null) {
            log.warn("No job dirs found. Returning.");
            this.numberOfDeletedJobDirs.set(0L);
            this.numberOfDirsUnableToDelete.set(0L);
            return;
        }
        long deletedCount = 0L;
        long unableToDeleteCount = 0L;
        for (File dir : jobDirs) {
            if (!dir.isDirectory()) {
                log.info("File {} isn't a directory. Skipping.", (Object)dir.getName());
                continue;
            }
            String id = dir.getName();
            try {
                Job job = this.jobSearchService.getJob(id);
                if (job.getStatus().isActive()) continue;
                Instant midnightUTC = TaskUtils.getMidnightUTC();
                Instant retentionThreshold = midnightUTC.minus(this.properties.getRetention(), ChronoUnit.DAYS);
                Optional finished = job.getFinished();
                if (!finished.isPresent() || !((Instant)finished.get()).isBefore(retentionThreshold)) continue;
                log.info("Attempting to delete job directory for job {}", (Object)id);
                if (this.runAsUser) {
                    CommandLine commandLine = new CommandLine("sudo");
                    commandLine.addArgument("rm");
                    commandLine.addArgument("-rf");
                    commandLine.addArgument(dir.getAbsolutePath());
                    this.processExecutor.execute(commandLine);
                } else {
                    FileUtils.deleteDirectory((File)dir);
                }
                ++deletedCount;
                log.info("Successfully deleted job directory for job {}", (Object)id);
            }
            catch (GenieException ge) {
                log.error("Unable to get job {}. Continuing.", (Object)id, (Object)ge);
                this.unableToGetJobCounter.increment();
                ++unableToDeleteCount;
            }
            catch (IOException ioe) {
                log.error("Unable to delete job directory for job with id: {}", (Object)id, (Object)ioe);
                this.unableToDeleteJobDirCounter.increment();
                ++unableToDeleteCount;
            }
        }
        this.numberOfDeletedJobDirs.set(deletedCount);
        this.numberOfDirsUnableToDelete.set(unableToDeleteCount);
    }
}

