/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.scripts;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.netflix.genie.web.exceptions.checked.ScriptExecutionException;
import com.netflix.genie.web.exceptions.checked.ScriptLoadingException;
import com.netflix.genie.web.exceptions.checked.ScriptNotConfiguredException;
import com.netflix.genie.web.properties.ScriptManagerProperties;
import com.netflix.genie.web.util.MetricsUtils;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.HashSet;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.concurrent.ThreadSafe;
import javax.script.Bindings;
import javax.script.Compilable;
import javax.script.CompiledScript;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.scheduling.TaskScheduler;

@ThreadSafe
public class ScriptManager {
    private static final Logger log = LoggerFactory.getLogger(ScriptManager.class);
    private static final String SCRIPT_LOAD_TIMER_NAME = "genie.scripts.load.timer";
    private static final String SCRIPT_EVALUATE_TIMER_NAME = "genie.scripts.evaluate.timer";
    private final ConcurrentMap<URI, AtomicReference<CompiledScript>> scriptsMap = Maps.newConcurrentMap();
    private final ScriptManagerProperties properties;
    private final TaskScheduler taskScheduler;
    private final ExecutorService executorService;
    private final ScriptEngineManager scriptEngineManager;
    private final ResourceLoader resourceLoader;
    private final MeterRegistry meterRegistry;

    public ScriptManager(ScriptManagerProperties properties, TaskScheduler taskScheduler, ExecutorService executorService, ScriptEngineManager scriptEngineManager, ResourceLoader resourceLoader, MeterRegistry meterRegistry) {
        this.properties = properties;
        this.taskScheduler = taskScheduler;
        this.executorService = executorService;
        this.scriptEngineManager = scriptEngineManager;
        this.resourceLoader = resourceLoader;
        this.meterRegistry = meterRegistry;
    }

    public void manageScript(URI scriptUri) {
        AtomicBoolean newKey = new AtomicBoolean(false);
        AtomicReference compiledScriptReference = this.scriptsMap.computeIfAbsent(scriptUri, key -> {
            newKey.set(true);
            return new AtomicReference();
        });
        if (newKey.get()) {
            this.taskScheduler.scheduleAtFixedRate((Runnable)new LoadScriptTask(scriptUri, compiledScriptReference), Instant.now(), Duration.ofMillis(this.properties.getRefreshInterval()));
            log.debug("Scheduled periodic refresh task for script: {}", (Object)scriptUri);
        }
    }

    protected Object evaluateScript(URI scriptUri, Bindings bindings, long timeout) throws ScriptNotConfiguredException, ScriptExecutionException {
        CompiledScript compiledScript;
        HashSet tags = Sets.newHashSet();
        tags.add(Tag.of((String)"scriptUri", (String)scriptUri.toString()));
        long start = System.nanoTime();
        try {
            compiledScript = this.getCompiledScript(scriptUri);
        }
        catch (ScriptNotConfiguredException e) {
            long durationNano = System.nanoTime() - start;
            MetricsUtils.addFailureTagsWithException(tags, (Throwable)((Object)e));
            this.meterRegistry.timer(SCRIPT_EVALUATE_TIMER_NAME, (Iterable)tags).record(durationNano, TimeUnit.NANOSECONDS);
            throw e;
        }
        Future<Object> taskFuture = this.executorService.submit(() -> compiledScript.eval(bindings));
        try {
            Object evaluationResult = taskFuture.get(timeout, TimeUnit.MILLISECONDS);
            MetricsUtils.addSuccessTags(tags);
            Object object = evaluationResult;
            return object;
        }
        catch (InterruptedException | ExecutionException | TimeoutException e) {
            taskFuture.cancel(true);
            MetricsUtils.addFailureTagsWithException(tags, e);
            throw new ScriptExecutionException("Script evaluation failed: " + scriptUri + ": " + e.getMessage(), e);
        }
        finally {
            long durationNano = System.nanoTime() - start;
            this.meterRegistry.timer(SCRIPT_EVALUATE_TIMER_NAME, (Iterable)tags).record(durationNano, TimeUnit.NANOSECONDS);
        }
    }

    private CompiledScript getCompiledScript(URI scriptUri) throws ScriptNotConfiguredException {
        AtomicReference compiledScriptReference = (AtomicReference)this.scriptsMap.get(scriptUri);
        if (compiledScriptReference == null) {
            throw new ScriptNotConfiguredException("Unknown script: " + scriptUri);
        }
        CompiledScript compiledScript = (CompiledScript)compiledScriptReference.get();
        if (compiledScript == null) {
            throw new ScriptNotConfiguredException("Script not loaded/compiled: " + scriptUri);
        }
        return compiledScript;
    }

    boolean isLoaded(URI scriptUri) {
        try {
            this.getCompiledScript(scriptUri);
            return true;
        }
        catch (ScriptNotConfiguredException e) {
            return false;
        }
    }

    private class LoadScriptTask
    implements Runnable {
        private final URI scriptUri;
        private final AtomicReference<CompiledScript> compiledScriptReference;

        LoadScriptTask(URI scriptUri, AtomicReference<CompiledScript> compiledScriptReference) {
            this.scriptUri = scriptUri;
            this.compiledScriptReference = compiledScriptReference;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            HashSet tags = Sets.newHashSet();
            tags.add(Tag.of((String)"scriptUri", (String)this.scriptUri.toString()));
            long start = System.nanoTime();
            try {
                CompiledScript compiledScript = this.loadScript();
                this.compiledScriptReference.set(compiledScript);
                MetricsUtils.addSuccessTags(tags);
            }
            catch (ScriptLoadingException e) {
                log.error("Failed to load script: " + this.scriptUri, (Throwable)((Object)e));
                MetricsUtils.addFailureTagsWithException(tags, (Throwable)((Object)e));
            }
            catch (Exception e) {
                log.error("Error loading script: " + this.scriptUri, (Throwable)e);
                MetricsUtils.addFailureTagsWithException(tags, e);
            }
            finally {
                long durationNano = System.nanoTime() - start;
                ScriptManager.this.meterRegistry.timer(ScriptManager.SCRIPT_LOAD_TIMER_NAME, (Iterable)tags).record(durationNano, TimeUnit.NANOSECONDS);
            }
        }

        private CompiledScript loadScript() throws ScriptLoadingException {
            CompiledScript compiledScript;
            InputStream scriptInputStream;
            String scriptUriString = this.scriptUri.toString();
            String scriptExtension = StringUtils.substringAfterLast((String)this.scriptUri.getPath(), (String)".");
            if (StringUtils.isBlank((CharSequence)scriptExtension)) {
                throw new ScriptLoadingException("Failed to determine file extension: " + scriptUriString);
            }
            Resource scriptResource = ScriptManager.this.resourceLoader.getResource(scriptUriString);
            if (!scriptResource.exists()) {
                throw new ScriptLoadingException("Script not found: " + scriptUriString);
            }
            ScriptEngine engine = ScriptManager.this.scriptEngineManager.getEngineByExtension(scriptExtension);
            if (engine == null) {
                throw new ScriptLoadingException("Script engine for file extension: " + scriptExtension + " not available");
            }
            if (!(engine instanceof Compilable)) {
                throw new ScriptLoadingException("Script engine for file extension: " + scriptExtension + " is not " + Compilable.class.getName());
            }
            Compilable compilable = (Compilable)((Object)engine);
            try {
                scriptInputStream = scriptResource.getInputStream();
            }
            catch (IOException e) {
                throw new ScriptLoadingException("Failed to read script", e);
            }
            InputStreamReader reader = new InputStreamReader(scriptInputStream, StandardCharsets.UTF_8);
            try {
                compiledScript = compilable.compile(reader);
            }
            catch (ScriptException e) {
                throw new ScriptLoadingException("Failed to compile script: " + scriptUriString, e);
            }
            log.info("Successfully compiled: " + scriptUriString);
            return compiledScript;
        }
    }
}

