/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.tasks.job;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.netflix.genie.common.dto.Application;
import com.netflix.genie.common.dto.Job;
import com.netflix.genie.common.dto.JobRequest;
import com.netflix.genie.common.dto.JobStatus;
import com.netflix.genie.common.exceptions.GenieException;
import com.netflix.genie.common.exceptions.GenieServerException;
import com.netflix.genie.common.external.util.GenieObjectMapper;
import com.netflix.genie.common.internal.exceptions.checked.JobArchiveException;
import com.netflix.genie.common.internal.jobs.JobConstants;
import com.netflix.genie.common.internal.services.JobArchiveService;
import com.netflix.genie.web.data.services.JobPersistenceService;
import com.netflix.genie.web.data.services.JobSearchService;
import com.netflix.genie.web.events.JobFinishedEvent;
import com.netflix.genie.web.events.JobFinishedReason;
import com.netflix.genie.web.jobs.JobDoneFile;
import com.netflix.genie.web.jobs.JobKillReasonFile;
import com.netflix.genie.web.properties.JobsProperties;
import com.netflix.genie.web.services.MailService;
import com.netflix.genie.web.util.MetricsUtils;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.validation.constraints.NotNull;
import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteStreamHandler;
import org.apache.commons.exec.Executor;
import org.apache.commons.exec.PumpStreamHandler;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.retry.support.RetryTemplate;

public class JobCompletionService {
    private static final Logger log = LoggerFactory.getLogger(JobCompletionService.class);
    static final String JOB_COMPLETION_TIMER_NAME = "genie.jobs.completion.timer";
    static final String JOB_COMPLETION_ERROR_COUNTER_NAME = "genie.jobs.errors.count";
    static final String ERROR_SOURCE_TAG = "error";
    static final String JOB_FINAL_STATE = "jobFinalState";
    private final JobPersistenceService jobPersistenceService;
    private final JobSearchService jobSearchService;
    private final JobArchiveService jobArchiveService;
    private final File baseWorkingDir;
    private final MailService mailServiceImpl;
    private final Executor executor;
    private final boolean deleteDependencies;
    private final boolean runAsUserEnabled;
    private final MeterRegistry registry;
    private final RetryTemplate retryTemplate;

    public JobCompletionService(JobPersistenceService jobPersistenceService, JobSearchService jobSearchService, JobArchiveService jobArchiveService, Resource genieWorkingDir, MailService mailServiceImpl, MeterRegistry registry, JobsProperties jobsProperties, @NotNull RetryTemplate retryTemplate) throws GenieException {
        this.jobPersistenceService = jobPersistenceService;
        this.jobSearchService = jobSearchService;
        this.jobArchiveService = jobArchiveService;
        this.mailServiceImpl = mailServiceImpl;
        this.deleteDependencies = jobsProperties.getCleanup().isDeleteDependencies();
        this.runAsUserEnabled = jobsProperties.getUsers().isRunAsUserEnabled();
        this.executor = new DefaultExecutor();
        this.executor.setStreamHandler((ExecuteStreamHandler)new PumpStreamHandler(null, null));
        try {
            this.baseWorkingDir = genieWorkingDir.getFile();
        }
        catch (IOException gse) {
            throw new GenieServerException("Could not load the base path from resource", (Throwable)gse);
        }
        this.registry = registry;
        this.retryTemplate = retryTemplate;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void handleJobCompletion(JobFinishedEvent event) {
        long start = System.nanoTime();
        String jobId = event.getId();
        HashSet tags = Sets.newHashSet();
        try {
            Job job = (Job)this.retryTemplate.execute(context -> this.getJob(jobId));
            JobStatus status = job.getStatus();
            if (status.isActive()) {
                try {
                    this.retryTemplate.execute(context -> this.processJobDir(job));
                }
                catch (Exception e) {
                    log.error("Failed archiving directory for job: {}", (Object)jobId, (Object)e);
                    this.incrementErrorCounter("JOB_DIRECTORY_FAILURE", e);
                }
                try {
                    this.retryTemplate.execute(context -> this.updateJob(job, event, tags));
                }
                catch (Exception e) {
                    log.error("Failed updating for job: {}", (Object)jobId, (Object)e);
                }
                try {
                    this.retryTemplate.execute(context -> this.sendEmail(jobId));
                }
                catch (Exception e) {
                    log.error("Failed sending email for job: {}", (Object)jobId, (Object)e);
                    this.incrementErrorCounter("JOB_UPDATE_FAILURE", e);
                }
            }
            MetricsUtils.addSuccessTags(tags);
        }
        catch (Exception e) {
            log.error("Failed getting job with id: {}", (Object)jobId, (Object)e);
            MetricsUtils.addFailureTagsWithException(tags, e);
        }
        finally {
            this.registry.timer(JOB_COMPLETION_TIMER_NAME, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    private Job getJob(String jobId) throws GenieException {
        return this.jobSearchService.getJob(jobId);
    }

    private Void updateJob(Job job, JobFinishedEvent event, Set<Tag> tags) throws GenieException {
        try {
            String jobId = event.getId();
            JobStatus status = job.getStatus();
            JobStatus eventStatus = null;
            if (status == JobStatus.INIT) {
                switch (event.getReason()) {
                    case KILLED: {
                        eventStatus = JobStatus.KILLED;
                        break;
                    }
                    case INVALID: {
                        eventStatus = JobStatus.INVALID;
                        break;
                    }
                    case FAILED_TO_INIT: {
                        eventStatus = JobStatus.FAILED;
                        break;
                    }
                    case PROCESS_COMPLETED: {
                        eventStatus = JobStatus.SUCCEEDED;
                        break;
                    }
                    case SYSTEM_CRASH: {
                        eventStatus = JobStatus.FAILED;
                        break;
                    }
                    default: {
                        eventStatus = JobStatus.INVALID;
                        log.warn("Unknown event status for job: {}", (Object)jobId);
                        break;
                    }
                }
            } else if (event.getReason() != JobFinishedReason.SYSTEM_CRASH) {
                try {
                    String finalStatus = (String)this.retryTemplate.execute(context -> this.updateFinalStatusForJob(jobId).toString());
                    tags.add(Tag.of((String)JOB_FINAL_STATE, (String)finalStatus));
                    this.cleanupProcesses(jobId);
                }
                catch (Exception e) {
                    log.error("Failed updating the exit code and status for job: {}", (Object)jobId, (Object)e);
                }
            } else {
                tags.add(Tag.of((String)JOB_FINAL_STATE, (String)JobStatus.FAILED.toString()));
                eventStatus = JobStatus.FAILED;
            }
            if (eventStatus != null) {
                tags.add(Tag.of((String)JOB_FINAL_STATE, (String)status.toString()));
                this.jobPersistenceService.updateJobStatus(jobId, eventStatus, event.getMessage());
            }
        }
        catch (Throwable t) {
            this.incrementErrorCounter("JOB_UPDATE_FAILURE", t);
            throw t;
        }
        return null;
    }

    private void cleanupProcesses(String jobId) {
        try {
            if (!this.jobSearchService.getJobStatus(jobId).equals((Object)JobStatus.INVALID)) {
                this.jobSearchService.getJobExecution(jobId).getProcessId().ifPresent(pid -> {
                    try {
                        CommandLine commandLine = new CommandLine("pkill");
                        commandLine.addArgument(JobConstants.getKillFlag());
                        commandLine.addArgument(Integer.toString(pid));
                        this.executor.execute(commandLine);
                        this.incrementErrorCounter("JOB_PROCESS_CLEANUP_NOT_THROWING_FAILURE", new RuntimeException());
                    }
                    catch (Exception e) {
                        log.debug("Received expected exception. Ignoring.");
                    }
                });
            }
        }
        catch (GenieException ge) {
            log.error("Unable to cleanup process for job {} due to exception.", (Object)jobId, (Object)ge);
            this.incrementErrorCounter("JOB_CLEANUP_FAILURE", ge);
        }
        catch (Throwable t) {
            this.incrementErrorCounter("JOB_PROCESS_CLEANUP_FAILURE", t);
            throw t;
        }
    }

    private JobStatus updateFinalStatusForJob(String id) throws GenieException {
        log.debug("Updating the status of the job.");
        try {
            JobStatus finalStatus;
            File jobDir = new File(this.baseWorkingDir, id);
            JobDoneFile jobDoneFile = (JobDoneFile)GenieObjectMapper.getMapper().readValue(new File(this.baseWorkingDir + "/" + id + "/" + "genie/genie.done"), JobDoneFile.class);
            File killReasonFile = new File(this.baseWorkingDir + "/" + id + "/" + "genie/kill-reason");
            String killedStatusMessages = killReasonFile.exists() ? ((JobKillReasonFile)GenieObjectMapper.getMapper().readValue(killReasonFile, JobKillReasonFile.class)).getKillReason() : "Job was killed by user.";
            int exitCode = jobDoneFile.getExitCode();
            File stdOut = new File(jobDir, "stdout");
            Long stdOutSize = stdOut.exists() && stdOut.isFile() ? Long.valueOf(stdOut.length()) : null;
            File stdErr = new File(jobDir, "stderr");
            Long stdErrSize = stdErr.exists() && stdErr.isFile() ? Long.valueOf(stdErr.length()) : null;
            switch (exitCode) {
                case 999: {
                    this.jobPersistenceService.setJobCompletionInformation(id, exitCode, JobStatus.KILLED, killedStatusMessages, stdOutSize, stdErrSize);
                    finalStatus = JobStatus.KILLED;
                    break;
                }
                case 0: {
                    this.jobPersistenceService.setJobCompletionInformation(id, exitCode, JobStatus.SUCCEEDED, "Job finished successfully.", stdOutSize, stdErrSize);
                    finalStatus = JobStatus.SUCCEEDED;
                    break;
                }
                default: {
                    this.jobPersistenceService.setJobCompletionInformation(id, exitCode, JobStatus.FAILED, "Job failed.", stdOutSize, stdErrSize);
                    finalStatus = JobStatus.FAILED;
                }
            }
            return finalStatus;
        }
        catch (IOException ioe) {
            this.incrementErrorCounter("JOB_FINAL_UPDATE_FAILURE", ioe);
            log.error("Could not load the done file for job {}. Marking it as failed.", (Object)id, (Object)ioe);
            this.jobPersistenceService.updateJobStatus(id, JobStatus.FAILED, "Failed to load done file.");
            return JobStatus.FAILED;
        }
        catch (Throwable t) {
            this.incrementErrorCounter("JOB_FINAL_UPDATE_FAILURE", t);
            throw t;
        }
    }

    private void deleteDependenciesDirectories(String jobId, File jobDir) {
        log.debug("Deleting dependencies.");
        if (jobDir.exists()) {
            HashSet dependencyDirectories = Sets.newHashSet();
            try {
                for (Application application : this.jobSearchService.getJobApplications(jobId)) {
                    application.getId().ifPresent(appId -> dependencyDirectories.add(new File(jobDir, "genie/applications/" + appId + "/" + "dependencies")));
                }
            }
            catch (GenieException e) {
                log.error("Error collecting application dependencies for job: {} due to error {}", (Object)jobId, (Object)e.toString());
                this.incrementErrorCounter("DELETE_APPLICATION_DEPENDENCIES_FAILURE", e);
            }
            try {
                this.jobSearchService.getJobCluster(jobId).getId().ifPresent(clusterId -> dependencyDirectories.add(new File(jobDir, "genie/cluster/" + clusterId + "/" + "dependencies")));
            }
            catch (GenieException e) {
                log.error("Error collecting cluster dependency for job: {} due to error {}", (Object)jobId, (Object)e.toString());
                this.incrementErrorCounter("DELETE_CLUSTER_DEPENDENCIES_FAILURE", e);
            }
            try {
                this.jobSearchService.getJobCommand(jobId).getId().ifPresent(commandId -> dependencyDirectories.add(new File(jobDir, "genie/command/" + commandId + "/" + "dependencies")));
            }
            catch (GenieException e) {
                log.error("Error collecting command dependency for job: {} due to error {}", (Object)jobId, (Object)e.toString());
                this.incrementErrorCounter("DELETE_COMMAND_DEPENDENCIES_FAILURE", e);
            }
            for (File dependencyDirectory : dependencyDirectories) {
                if (!dependencyDirectory.exists()) continue;
                try {
                    if (this.runAsUserEnabled) {
                        CommandLine deleteCommand = new CommandLine("sudo");
                        deleteCommand.addArgument("rm");
                        deleteCommand.addArgument("-rf");
                        deleteCommand.addArgument(dependencyDirectory.getCanonicalPath());
                        log.debug("Delete command is {}", (Object)deleteCommand);
                        this.executor.execute(deleteCommand);
                        continue;
                    }
                    FileUtils.deleteDirectory((File)dependencyDirectory);
                }
                catch (IOException e) {
                    this.incrementErrorCounter("DELETE_DEPENDENCIES_FAILURE");
                    log.error("Error deleting dependency directory: {}: {}", (Object)dependencyDirectory.getAbsolutePath(), (Object)e.toString());
                }
                catch (Throwable t) {
                    this.incrementErrorCounter("DELETE_DEPENDENCIES_FAILURE");
                    throw t;
                }
            }
        }
    }

    private boolean processJobDir(Job job) throws GenieException {
        String jobId;
        File jobDir;
        log.debug("Got a job finished event. Will process job directory.");
        Optional oJobId = job.getId();
        if (oJobId.isPresent() && !this.jobSearchService.getJobStatus((String)oJobId.get()).equals((Object)JobStatus.INVALID) && (jobDir = new File(this.baseWorkingDir, jobId = (String)oJobId.get())).exists()) {
            Optional archiveLocation;
            if (this.deleteDependencies) {
                this.deleteDependenciesDirectories(jobId, jobDir);
            }
            if ((archiveLocation = job.getArchiveLocation()).isPresent() && !Strings.isNullOrEmpty((String)((String)archiveLocation.get()))) {
                log.debug("Archiving job directory");
                try {
                    this.jobArchiveService.archiveDirectory(jobDir.toPath(), new URI((String)archiveLocation.get()));
                }
                catch (JobArchiveException | URISyntaxException e) {
                    log.warn("Failed to archive job files for job {} due to {}", new Object[]{jobId, e.getMessage(), e});
                    this.incrementErrorCounter("JOB_ARCHIVAL_FAILURE", e);
                    return false;
                }
            }
        }
        return true;
    }

    private boolean sendEmail(String jobId) throws GenieException {
        JobRequest jobRequest = this.jobSearchService.getJobRequest(jobId);
        boolean result = false;
        Optional email = jobRequest.getEmail();
        if (email.isPresent() && !Strings.isNullOrEmpty((String)((String)email.get()))) {
            log.debug("Got a job finished event. Sending email: {}", email.get());
            JobStatus status = this.jobSearchService.getJobStatus(jobId);
            StringBuilder subject = new StringBuilder().append("Genie Job Finished. Id: [").append(jobId).append("], Name: [").append(jobRequest.getName()).append("], Status: [").append(status).append("].");
            StringBuilder body = new StringBuilder().append("Id: [").append(jobId).append("]\n").append("Name: [").append(jobRequest.getName()).append("]\n").append("Status: [").append(status).append("]\n").append("User: [").append(jobRequest.getUser()).append("]\n").append("Tags: ").append(jobRequest.getTags()).append("\n");
            jobRequest.getDescription().ifPresent(description -> body.append("[").append((String)description).append("]"));
            try {
                this.mailServiceImpl.sendEmail((String)email.get(), subject.toString(), body.toString());
            }
            catch (Throwable t) {
                this.incrementErrorCounter("JOB_EMAIL_FAILURE", t);
                throw t;
            }
            result = true;
        }
        return result;
    }

    private void incrementErrorCounter(String errorTagValue, Throwable throwable) {
        this.incrementErrorCounter((Set<Tag>)ImmutableSet.of((Object)Tag.of((String)ERROR_SOURCE_TAG, (String)errorTagValue), (Object)Tag.of((String)"exceptionClass", (String)throwable.getClass().getCanonicalName())));
    }

    private void incrementErrorCounter(String errorTagValue) {
        this.incrementErrorCounter((Set<Tag>)ImmutableSet.of((Object)Tag.of((String)ERROR_SOURCE_TAG, (String)errorTagValue)));
    }

    private void incrementErrorCounter(Set<Tag> tags) {
        this.registry.counter(JOB_COMPLETION_ERROR_COUNTER_NAME, tags).increment();
    }
}

