/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.services.impl;

import com.google.common.collect.Sets;
import com.netflix.genie.common.internal.dtos.v4.files.JobFileState;
import com.netflix.genie.web.services.JobFileService;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;
import javax.annotation.Nullable;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

public class DiskJobFileServiceImpl
implements JobFileService {
    private static final Logger log = LoggerFactory.getLogger(DiskJobFileServiceImpl.class);
    private final Path jobsDirRoot;

    public DiskJobFileServiceImpl(Resource jobsDir) throws IOException {
        this.jobsDirRoot = jobsDir.getFile().toPath();
        this.createOrCheckDirectory(this.jobsDirRoot);
    }

    @Override
    public void createJobDirectory(String jobId) throws IOException {
        Path jobDirectory = this.jobsDirRoot.resolve(jobId);
        this.createOrCheckDirectory(jobDirectory);
    }

    @Override
    public Set<JobFileState> getJobDirectoryFileState(String jobId, boolean calculateMd5) throws IOException {
        log.debug("Getting job directory state for job {} {} MD5", (Object)jobId, (Object)(calculateMd5 ? "with" : "without"));
        Path jobDirectory = this.jobsDirRoot.resolve(jobId);
        this.createOrCheckDirectory(jobDirectory);
        HashSet jobDirectoryFiles = Sets.newHashSet();
        Files.walkFileTree(jobDirectory, new FileStateVisitor(jobDirectory, calculateMd5, jobDirectoryFiles));
        return jobDirectoryFiles;
    }

    @Override
    public void updateFile(String jobId, String relativePath, long startByte, byte[] data) throws IOException {
        log.debug("Attempting to write {} bytes from position {} into log file {} for job {}", new Object[]{data.length, startByte, relativePath, jobId});
        Path jobFile = this.jobsDirRoot.resolve(jobId).resolve(relativePath);
        if (Files.notExists(jobFile, new LinkOption[0])) {
            Path logFileParent = jobFile.getParent();
            if (logFileParent != null) {
                this.createOrCheckDirectory(logFileParent);
            }
        } else if (Files.isDirectory(jobFile, new LinkOption[0])) {
            throw new IllegalArgumentException(relativePath + " is a directory not a file. Unable to update");
        }
        try (FileChannel fileChannel = FileChannel.open(jobFile, EnumSet.of(StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.SPARSE), new FileAttribute[0]);){
            fileChannel.position(startByte);
            ByteBuffer byteBuffer = ByteBuffer.wrap(data);
            while (byteBuffer.hasRemaining()) {
                fileChannel.write(byteBuffer);
            }
        }
    }

    @Override
    public Resource getJobFileAsResource(String jobId, @Nullable String relativePath) {
        log.debug("Attempting to get resource for job {} with relative path {}", (Object)jobId, (Object)relativePath);
        Path jobDirectoryLocation = StringUtils.isBlank((CharSequence)relativePath) ? this.jobsDirRoot.resolve(jobId) : this.jobsDirRoot.resolve(jobId).resolve(relativePath);
        return new FileSystemResource(jobDirectoryLocation);
    }

    @Override
    public void deleteJobFile(String jobId, String relativePath) throws IOException {
        log.debug("Requested to delete file {} for job {}", (Object)relativePath, (Object)jobId);
        Path jobFile = this.jobsDirRoot.resolve(jobId).resolve(relativePath);
        if (Files.deleteIfExists(jobFile)) {
            log.debug("Deleted file {} for job {}", (Object)relativePath, (Object)jobId);
        } else {
            log.debug("No file {} exists for job {}. Ignoring", (Object)relativePath, (Object)jobId);
        }
    }

    private void createOrCheckDirectory(Path dir) throws IOException {
        if (Files.notExists(dir, new LinkOption[0])) {
            Files.createDirectories(dir, new FileAttribute[0]);
        } else if (!Files.isDirectory(dir, new LinkOption[0])) {
            throw new IOException(dir + " exists but is not a directory and must be");
        }
    }

    static class FileStateVisitor
    implements FileVisitor<Path> {
        private static final Logger log = LoggerFactory.getLogger(FileStateVisitor.class);
        private final Path jobDirectoryRoot;
        private final boolean calculateMd5;
        private final Set<JobFileState> files;

        FileStateVisitor(Path jobDirectoryRoot, boolean calculateMd5, Set<JobFileState> files) {
            this.jobDirectoryRoot = jobDirectoryRoot;
            this.calculateMd5 = calculateMd5;
            this.files = files;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            String md5;
            log.debug("Visiting file {}", (Object)file);
            String relativePath = this.jobDirectoryRoot.relativize(file).toString();
            long size = Files.size(file);
            if (this.calculateMd5) {
                try (InputStream fileInputStream = Files.newInputStream(file, StandardOpenOption.READ);){
                    md5 = DigestUtils.md5Hex((InputStream)fileInputStream);
                }
            } else {
                md5 = null;
            }
            JobFileState jobFileState = new JobFileState(relativePath, size, md5);
            log.debug("Visited file has state {}", (Object)jobFileState);
            this.files.add(jobFileState);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            return FileVisitResult.TERMINATE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            return FileVisitResult.CONTINUE;
        }
    }
}

