/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.agent.services.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.netflix.genie.web.agent.services.AgentConnectionTrackingService;
import com.netflix.genie.web.agent.services.AgentRoutingService;
import java.time.Instant;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.actuate.info.Info;
import org.springframework.boot.actuate.info.InfoContributor;
import org.springframework.scheduling.TaskScheduler;

public class AgentConnectionTrackingServiceImpl
implements AgentConnectionTrackingService,
InfoContributor {
    private static final Logger log = LoggerFactory.getLogger(AgentConnectionTrackingServiceImpl.class);
    protected static final int STREAM_EXPIRATION_PERIOD = 10000;
    protected static final int CLEANUP_TASK_PERIOD = 2000;
    private final AgentRoutingService agentRoutingService;
    private final TaskScheduler taskScheduler;
    private final ConcurrentMap<String, JobStreamsRecord> jobStreamRecordsMap = Maps.newConcurrentMap();
    private final Supplier<Instant> timeSupplier;

    public AgentConnectionTrackingServiceImpl(AgentRoutingService agentRoutingService, TaskScheduler taskScheduler) {
        this(agentRoutingService, taskScheduler, Instant::now);
    }

    @VisibleForTesting
    AgentConnectionTrackingServiceImpl(AgentRoutingService agentRoutingService, TaskScheduler taskScheduler, Supplier<Instant> timeSupplier) {
        this.agentRoutingService = agentRoutingService;
        this.taskScheduler = taskScheduler;
        this.timeSupplier = timeSupplier;
        this.taskScheduler.scheduleAtFixedRate(this::cleanupTask, 2000L);
    }

    @Override
    public void notifyHeartbeat(String streamId, String claimedJobId) {
        AtomicBoolean atomicIsNew = new AtomicBoolean(false);
        JobStreamsRecord jobStreamsRecord = this.jobStreamRecordsMap.computeIfAbsent(claimedJobId, jobId -> {
            atomicIsNew.set(true);
            return new JobStreamsRecord((String)jobId);
        });
        boolean isNew = atomicIsNew.get();
        log.debug("Received heartbeat for {} job {} using stream {}", new Object[]{isNew ? "new" : "existing", claimedJobId, streamId});
        jobStreamsRecord.updateActiveStream(streamId, this.timeSupplier.get());
        if (isNew) {
            log.debug("Notifying new agent connected executing job: {}", (Object)claimedJobId);
            this.agentRoutingService.handleClientConnected(claimedJobId);
        }
    }

    @Override
    public void notifyDisconnected(String streamId, String claimedJobId) {
        JobStreamsRecord jobStreamsRecord = (JobStreamsRecord)this.jobStreamRecordsMap.get(claimedJobId);
        log.debug("Received disconnection for {} job {} using stream {}", new Object[]{jobStreamsRecord == null ? "unknown" : "existing", claimedJobId, streamId});
        if (jobStreamsRecord != null) {
            jobStreamsRecord.removeActiveStream(streamId);
            if (!jobStreamsRecord.hasActiveStreams()) {
                log.debug("Job {} last stream disconnected, notifying routing service", (Object)claimedJobId);
                this.jobStreamRecordsMap.remove(claimedJobId);
                this.agentRoutingService.handleClientDisconnected(claimedJobId);
            }
        }
    }

    @Override
    public long getConnectedAgentsCount() {
        return this.jobStreamRecordsMap.size();
    }

    private void cleanupTask() {
        try {
            this.removeExpiredStreams();
            this.removeExpiredAgents();
        }
        catch (Exception e) {
            log.error("Exception in cleanup task", (Throwable)e);
        }
    }

    private void removeExpiredAgents() {
        HashSet removedJobIds = Sets.newHashSet();
        this.jobStreamRecordsMap.entrySet().removeIf(entry -> {
            if (!((JobStreamsRecord)entry.getValue()).hasActiveStreams()) {
                removedJobIds.add(entry.getKey());
                return true;
            }
            return false;
        });
        for (String jobId : removedJobIds) {
            log.debug("Job {} last stream expired, notifying routing service", (Object)jobId);
            this.agentRoutingService.handleClientDisconnected(jobId);
        }
    }

    private void removeExpiredStreams() {
        Instant cutoff = this.timeSupplier.get().minusMillis(10000L);
        this.jobStreamRecordsMap.forEach((jobId, record) -> ((JobStreamsRecord)record).expungeExpiredStreams(cutoff));
    }

    public void contribute(Info.Builder builder) {
        ImmutableList jobIds = ImmutableList.copyOf(this.jobStreamRecordsMap.keySet());
        builder.withDetail("connectedAgents", (Object)jobIds);
    }

    private static final class JobStreamsRecord {
        private final String jobId;
        private final Map<String, Instant> streamsLastHeartbeatMap = Maps.newHashMap();

        private JobStreamsRecord(String jobId) {
            this.jobId = jobId;
        }

        private synchronized void updateActiveStream(String streamId, Instant currentTime) {
            Instant previousHeartbeat = this.streamsLastHeartbeatMap.put(streamId, currentTime);
            log.debug("{} heartbeat for job {} stream {}", new Object[]{previousHeartbeat == null ? "Created" : "Updated", this.jobId, streamId});
        }

        private synchronized void removeActiveStream(String streamId) {
            Instant previousHeartbeat = this.streamsLastHeartbeatMap.remove(streamId);
            if (previousHeartbeat != null) {
                log.debug("Removed job {} stream {}", (Object)this.jobId, (Object)streamId);
            }
        }

        private synchronized boolean hasActiveStreams() {
            return !this.streamsLastHeartbeatMap.isEmpty();
        }

        private synchronized void expungeExpiredStreams(Instant cutoff) {
            boolean removed = this.streamsLastHeartbeatMap.entrySet().removeIf(entry -> ((Instant)entry.getValue()).isBefore(cutoff));
            if (removed) {
                log.debug("Removed expired streams for job {}", (Object)this.jobId);
            }
        }
    }
}

