/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.services.impl;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.netflix.genie.common.external.dtos.v4.Application;
import com.netflix.genie.common.external.dtos.v4.Cluster;
import com.netflix.genie.common.external.dtos.v4.ClusterMetadata;
import com.netflix.genie.common.external.dtos.v4.Command;
import com.netflix.genie.common.external.dtos.v4.CommonResource;
import com.netflix.genie.common.external.dtos.v4.Criterion;
import com.netflix.genie.common.external.dtos.v4.JobEnvironment;
import com.netflix.genie.common.external.dtos.v4.JobRequest;
import com.netflix.genie.common.external.dtos.v4.JobSpecification;
import com.netflix.genie.common.external.dtos.v4.JobStatus;
import com.netflix.genie.common.internal.exceptions.checked.GenieJobResolutionException;
import com.netflix.genie.web.data.services.DataServices;
import com.netflix.genie.web.data.services.PersistenceService;
import com.netflix.genie.web.dtos.ResolvedJob;
import com.netflix.genie.web.dtos.ResourceSelectionResult;
import com.netflix.genie.web.properties.JobsProperties;
import com.netflix.genie.web.selectors.ClusterSelectionContext;
import com.netflix.genie.web.selectors.ClusterSelector;
import com.netflix.genie.web.selectors.CommandSelectionContext;
import com.netflix.genie.web.selectors.CommandSelector;
import com.netflix.genie.web.services.JobResolverService;
import com.netflix.genie.web.util.MetricsUtils;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import java.io.File;
import java.net.URI;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.validation.Valid;
import javax.validation.constraints.NotEmpty;
import org.apache.commons.lang3.RegExUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.TargetClassAware;
import org.springframework.core.env.Environment;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.validation.annotation.Validated;

@Validated
public class JobResolverServiceImpl
implements JobResolverService {
    private static final Logger log = LoggerFactory.getLogger(JobResolverServiceImpl.class);
    private static final String RESOLVE_JOB_TIMER = "genie.services.jobResolver.resolve.timer";
    private static final String RESOLVE_COMMAND_TIMER = "genie.services.jobResolver.resolveCommand.timer";
    private static final String RESOLVE_CLUSTER_TIMER = "genie.services.jobResolver.resolveCluster.timer";
    private static final String RESOLVE_APPLICATIONS_TIMER = "genie.services.jobResolver.resolveApplications.timer";
    private static final String GENERATE_CRITERIA_PERMUTATIONS_TIMER = "genie.services.jobResolver.generateClusterCriteriaPermutations.timer";
    private static final String CLUSTER_SELECTOR_COUNTER = "genie.services.jobResolver.resolveCluster.clusterSelector.counter";
    private static final String NO_RATIONALE = "No rationale provided";
    private static final String NO_ID_FOUND = "No id found";
    private static final String VERSION_4 = "4";
    private static final Tag SAVED_TAG = Tag.of((String)"saved", (String)"true");
    private static final Tag NOT_SAVED_TAG = Tag.of((String)"saved", (String)"false");
    private static final Tag NO_CLUSTER_RESOLVED_ID = Tag.of((String)"clusterId", (String)"None Resolved");
    private static final Tag NO_CLUSTER_RESOLVED_NAME = Tag.of((String)"clusterName", (String)"None Resolved");
    private static final Tag NO_COMMAND_RESOLVED_ID = Tag.of((String)"commandId", (String)"None Resolved");
    private static final Tag NO_COMMAND_RESOLVED_NAME = Tag.of((String)"commandName", (String)"None Resolved");
    private static final String ID_FIELD = "id";
    private static final String NAME_FIELD = "name";
    private static final String STATUS_FIELD = "status";
    private static final String VERSION_FIELD = "version";
    private static final String CLUSTER_SELECTOR_STATUS_SUCCESS = "success";
    private static final String CLUSTER_SELECTOR_STATUS_NO_PREFERENCE = "no preference";
    private final PersistenceService persistenceService;
    private final List<ClusterSelector> clusterSelectors;
    private final CommandSelector commandSelector;
    private final MeterRegistry registry;
    private final int defaultMemory;
    private final File defaultJobDirectory;
    private final String defaultArchiveLocation;

    public JobResolverServiceImpl(DataServices dataServices, @NotEmpty List<ClusterSelector> clusterSelectors, CommandSelector commandSelector, MeterRegistry registry, JobsProperties jobsProperties, Environment environment) {
        this.persistenceService = dataServices.getPersistenceService();
        this.clusterSelectors = clusterSelectors;
        this.commandSelector = commandSelector;
        this.defaultMemory = jobsProperties.getMemory().getDefaultJobMemory();
        URI jobDirProperty = jobsProperties.getLocations().getJobs();
        this.defaultJobDirectory = Paths.get(jobDirProperty).toFile();
        String archiveLocation = jobsProperties.getLocations().getArchives().toString();
        this.defaultArchiveLocation = archiveLocation.endsWith(File.separator) ? archiveLocation : archiveLocation + File.separator;
        this.registry = registry;
    }

    @Override
    @Nonnull
    @Transactional
    public ResolvedJob resolveJob(String id) throws GenieJobResolutionException {
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet((Object[])new Tag[]{SAVED_TAG});
        try {
            log.info("Received request to resolve a job with id {}", (Object)id);
            JobStatus jobStatus = this.persistenceService.getJobStatus(id);
            if (!jobStatus.isResolvable()) {
                throw new IllegalArgumentException("Job " + id + " is already resolved: " + jobStatus);
            }
            JobRequest jobRequest = this.persistenceService.getJobRequest(id);
            boolean apiJob = this.persistenceService.isApiJob(id);
            JobResolutionContext context = new JobResolutionContext(id, jobRequest, apiJob);
            ResolvedJob resolvedJob = this.resolve(context);
            this.persistenceService.saveResolvedJob(id, resolvedJob);
            MetricsUtils.addSuccessTags(tags);
            ResolvedJob resolvedJob2 = resolvedJob;
            return resolvedJob2;
        }
        catch (GenieJobResolutionException e) {
            MetricsUtils.addFailureTagsWithException(tags, e);
            throw e;
        }
        catch (Throwable t) {
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw new GenieJobResolutionException(t);
        }
        finally {
            this.registry.timer(RESOLVE_JOB_TIMER, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    @Override
    @Nonnull
    public ResolvedJob resolveJob(String id, @Valid JobRequest jobRequest, boolean apiJob) throws GenieJobResolutionException {
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet((Object[])new Tag[]{NOT_SAVED_TAG});
        try {
            log.info("Received request to resolve a job for id {} and request {}", (Object)id, (Object)jobRequest);
            JobResolutionContext context = new JobResolutionContext(id, jobRequest, apiJob);
            ResolvedJob resolvedJob = this.resolve(context);
            MetricsUtils.addSuccessTags(tags);
            ResolvedJob resolvedJob2 = resolvedJob;
            return resolvedJob2;
        }
        catch (GenieJobResolutionException e) {
            MetricsUtils.addFailureTagsWithException(tags, e);
            throw e;
        }
        catch (Throwable t) {
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw new GenieJobResolutionException(t);
        }
        finally {
            this.registry.timer(RESOLVE_JOB_TIMER, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    private ResolvedJob resolve(JobResolutionContext context) throws GenieJobResolutionException {
        this.resolveCommand(context);
        this.resolveCluster(context);
        this.resolveApplications(context);
        this.resolveJobMemory(context);
        this.resolveEnvironmentVariables(context);
        this.resolveTimeout(context);
        this.resolveArchiveLocation(context);
        this.resolveJobDirectory(context);
        return context.build();
    }

    private void resolveCommand(JobResolutionContext context) throws GenieJobResolutionException {
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet();
        try {
            JobRequest jobRequest = context.getJobRequest();
            Criterion criterion = jobRequest.getCriteria().getCommandCriterion();
            Set<Command> commands = this.persistenceService.findCommandsMatchingCriterion(criterion, true);
            if (commands.isEmpty()) {
                throw new GenieJobResolutionException("No command matching command criterion found");
            }
            Map<Command, List<Criterion>> commandClusterCriterions = this.generateClusterCriteriaPermutations(commands, jobRequest);
            Set<Criterion> uniqueCriteria = this.flattenClusterCriteriaPermutations(commandClusterCriterions);
            Set<Cluster> allCandidateClusters = this.persistenceService.findClustersMatchingAnyCriterion(uniqueCriteria, true);
            if (allCandidateClusters.isEmpty()) {
                throw new GenieJobResolutionException("No clusters available to run any candidate command on");
            }
            Map<Command, Set<Cluster>> commandClusters = this.generateCommandClustersMap(commandClusterCriterions, allCandidateClusters);
            if (commandClusters.isEmpty()) {
                throw new GenieJobResolutionException("No clusters available to run any candidate command on");
            }
            context.setCommandClusters(commandClusters);
            ResourceSelectionResult result = this.commandSelector.select(new CommandSelectionContext(context.getJobId(), jobRequest, context.isApiJob(), commandClusters));
            Command command = (Command)result.getSelectedResource().orElseThrow(() -> new GenieJobResolutionException("Expected a command but " + result.getSelectorClass().getSimpleName() + " didn't select anything. Rationale: " + result.getSelectionRationale().orElse(NO_RATIONALE)));
            log.debug("Selected command {} for criterion {} using {} due to {}", new Object[]{command.getId(), criterion, result.getSelectorClass().getName(), result.getSelectionRationale().orElse(NO_RATIONALE)});
            MetricsUtils.addSuccessTags(tags);
            tags.add(Tag.of((String)"commandId", (String)command.getId()));
            tags.add(Tag.of((String)"commandName", (String)command.getMetadata().getName()));
            context.setCommand(command);
        }
        catch (GenieJobResolutionException e) {
            tags.add(NO_COMMAND_RESOLVED_ID);
            tags.add(NO_COMMAND_RESOLVED_NAME);
            MetricsUtils.addFailureTagsWithException(tags, e);
            throw e;
        }
        catch (Throwable t) {
            tags.add(NO_COMMAND_RESOLVED_ID);
            tags.add(NO_COMMAND_RESOLVED_NAME);
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw new GenieJobResolutionException(t);
        }
        finally {
            this.registry.timer(RESOLVE_COMMAND_TIMER, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void resolveCluster(JobResolutionContext context) throws GenieJobResolutionException {
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet();
        String jobId = context.getJobId();
        try {
            Command command = context.getCommand().orElseThrow(() -> new IllegalStateException("Command not resolved before attempting to resolve a cluster for job " + jobId));
            Set<Cluster> candidateClusters = context.getCommandClusters().orElseThrow(() -> new IllegalStateException("Command to candidate cluster map not available for job " + jobId)).get(command);
            if (candidateClusters == null || candidateClusters.isEmpty()) {
                throw new IllegalStateException("Command " + command.getId() + " had no candidate clusters for job " + jobId);
            }
            Cluster cluster = null;
            for (ClusterSelector clusterSelector : this.clusterSelectors) {
                HashSet selectorTags = Sets.newHashSet((Iterable)tags);
                String clusterSelectorClass = this.getProxyObjectClassName(clusterSelector);
                selectorTags.add(Tag.of((String)"class", (String)clusterSelectorClass));
                try {
                    ResourceSelectionResult result = clusterSelector.select(new ClusterSelectionContext(jobId, context.getJobRequest(), context.isApiJob(), command, candidateClusters));
                    Optional selectedClusterOptional = result.getSelectedResource();
                    if (selectedClusterOptional.isPresent()) {
                        cluster = (Cluster)selectedClusterOptional.get();
                        log.debug("Successfully selected cluster {} using selector {} for job {} with rationale: {}", new Object[]{cluster.getId(), clusterSelectorClass, jobId, result.getSelectionRationale().orElse(NO_RATIONALE)});
                        selectorTags.add(Tag.of((String)STATUS_FIELD, (String)CLUSTER_SELECTOR_STATUS_SUCCESS));
                        selectorTags.add(Tag.of((String)"clusterId", (String)cluster.getId()));
                        selectorTags.add(Tag.of((String)"clusterName", (String)cluster.getMetadata().getName()));
                        break;
                    }
                    selectorTags.add(Tag.of((String)STATUS_FIELD, (String)CLUSTER_SELECTOR_STATUS_NO_PREFERENCE));
                    selectorTags.add(NO_CLUSTER_RESOLVED_ID);
                    selectorTags.add(NO_CLUSTER_RESOLVED_NAME);
                    log.debug("Selector {} returned no preference with rationale: {}", (Object)clusterSelectorClass, (Object)result.getSelectionRationale().orElse(NO_RATIONALE));
                }
                catch (Exception e) {
                    selectorTags.add(NO_CLUSTER_RESOLVED_ID);
                    selectorTags.add(NO_CLUSTER_RESOLVED_NAME);
                    MetricsUtils.addFailureTagsWithException(selectorTags, e);
                    log.warn("Cluster selector {} evaluation threw exception for job {}", new Object[]{clusterSelectorClass, jobId, e});
                }
                finally {
                    this.registry.counter(CLUSTER_SELECTOR_COUNTER, (Iterable)selectorTags).increment();
                }
            }
            if (cluster == null) {
                throw new GenieJobResolutionException("No cluster resolved for job " + jobId);
            }
            log.debug("Resolved cluster {} for job {}", (Object)cluster.getId(), (Object)jobId);
            context.setCluster(cluster);
            MetricsUtils.addSuccessTags(tags);
            tags.add(Tag.of((String)"clusterId", (String)cluster.getId()));
            tags.add(Tag.of((String)"clusterName", (String)cluster.getMetadata().getName()));
        }
        catch (GenieJobResolutionException e) {
            tags.add(NO_CLUSTER_RESOLVED_ID);
            tags.add(NO_CLUSTER_RESOLVED_NAME);
            MetricsUtils.addFailureTagsWithException(tags, e);
            throw e;
        }
        catch (Throwable t) {
            tags.add(NO_CLUSTER_RESOLVED_ID);
            tags.add(NO_CLUSTER_RESOLVED_NAME);
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw new GenieJobResolutionException(t);
        }
        finally {
            this.registry.timer(RESOLVE_CLUSTER_TIMER, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    private void resolveApplications(JobResolutionContext context) throws GenieJobResolutionException {
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet();
        String id = context.getJobId();
        JobRequest jobRequest = context.getJobRequest();
        try {
            String commandId = context.getCommand().orElseThrow(() -> new IllegalStateException("Command hasn't been resolved before applications")).getId();
            log.debug("Selecting applications for job {} and command {}", (Object)id, (Object)commandId);
            ArrayList applications = Lists.newArrayList();
            if (jobRequest.getCriteria().getApplicationIds().isEmpty()) {
                applications.addAll(this.persistenceService.getApplicationsForCommand(commandId));
            } else {
                for (String applicationId : jobRequest.getCriteria().getApplicationIds()) {
                    applications.add(this.persistenceService.getApplication(applicationId));
                }
            }
            log.debug("Resolved applications {} for job {}", (Object)applications.stream().map(CommonResource::getId).reduce((one, two) -> one + "," + two).orElse(NO_ID_FOUND), (Object)id);
            MetricsUtils.addSuccessTags(tags);
            context.setApplications(applications);
        }
        catch (Throwable t) {
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw new GenieJobResolutionException(t);
        }
        finally {
            this.registry.timer(RESOLVE_APPLICATIONS_TIMER, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    private void resolveJobMemory(JobResolutionContext context) {
        context.setJobMemory(context.getJobRequest().getRequestedJobEnvironment().getRequestedJobMemory().orElse(context.getCommand().orElseThrow(() -> new IllegalStateException("Command not resolved before attempting to resolve job memory")).getMemory().orElse(this.defaultMemory)));
    }

    private void resolveEnvironmentVariables(JobResolutionContext context) {
        Command command = context.getCommand().orElseThrow(() -> new IllegalStateException("Command not resolved before attempting to resolve env variables"));
        Cluster cluster = context.getCluster().orElseThrow(() -> new IllegalStateException("Cluster not resolved before attempting to resolve env variables"));
        String id = context.getJobId();
        JobRequest jobRequest = context.getJobRequest();
        int jobMemory = context.getJobMemory().orElseThrow(() -> new IllegalStateException("Job memory not resolved before attempting to resolve env variables"));
        ImmutableMap.Builder envVariables = ImmutableMap.builder();
        envVariables.put((Object)"GENIE_VERSION", (Object)VERSION_4);
        envVariables.put((Object)"GENIE_CLUSTER_ID", (Object)cluster.getId());
        envVariables.put((Object)"GENIE_CLUSTER_NAME", (Object)cluster.getMetadata().getName());
        envVariables.put((Object)"GENIE_CLUSTER_TAGS", (Object)this.tagsToString(cluster.getMetadata().getTags()));
        envVariables.put((Object)"GENIE_COMMAND_ID", (Object)command.getId());
        envVariables.put((Object)"GENIE_COMMAND_NAME", (Object)command.getMetadata().getName());
        envVariables.put((Object)"GENIE_COMMAND_TAGS", (Object)this.tagsToString(command.getMetadata().getTags()));
        envVariables.put((Object)"GENIE_JOB_ID", (Object)id);
        envVariables.put((Object)"GENIE_JOB_NAME", (Object)jobRequest.getMetadata().getName());
        envVariables.put((Object)"GENIE_JOB_MEMORY", (Object)String.valueOf(jobMemory));
        envVariables.put((Object)"GENIE_JOB_TAGS", (Object)this.tagsToString(jobRequest.getMetadata().getTags()));
        envVariables.put((Object)"GENIE_JOB_GROUPING", (Object)jobRequest.getMetadata().getGrouping().orElse(""));
        envVariables.put((Object)"GENIE_JOB_GROUPING_INSTANCE", (Object)jobRequest.getMetadata().getGroupingInstance().orElse(""));
        envVariables.put((Object)"GENIE_REQUESTED_COMMAND_TAGS", (Object)this.tagsToString(jobRequest.getCriteria().getCommandCriterion().getTags()));
        List clusterCriteria = jobRequest.getCriteria().getClusterCriteria();
        ArrayList clusterCriteriaTags = Lists.newArrayListWithExpectedSize((int)clusterCriteria.size());
        for (int i = 0; i < clusterCriteria.size(); ++i) {
            Criterion criterion = (Criterion)clusterCriteria.get(i);
            String criteriaTagsString = this.tagsToString(criterion.getTags());
            envVariables.put((Object)("GENIE_REQUESTED_CLUSTER_TAGS_" + i), (Object)criteriaTagsString);
            clusterCriteriaTags.add("[" + criteriaTagsString + "]");
        }
        envVariables.put((Object)"GENIE_REQUESTED_CLUSTER_TAGS", (Object)("[" + StringUtils.join((Iterable)clusterCriteriaTags, (char)',') + "]"));
        envVariables.put((Object)"GENIE_USER", (Object)jobRequest.getMetadata().getUser());
        envVariables.put((Object)"GENIE_USER_GROUP", (Object)jobRequest.getMetadata().getGroup().orElse(""));
        context.setEnvironmentVariables((Map<String, String>)envVariables.build());
    }

    private void resolveTimeout(JobResolutionContext context) {
        JobRequest jobRequest = context.getJobRequest();
        if (jobRequest.getRequestedAgentConfig().getTimeoutRequested().isPresent()) {
            context.setTimeout((Integer)jobRequest.getRequestedAgentConfig().getTimeoutRequested().get());
        } else if (context.isApiJob()) {
            context.setTimeout(604800);
        }
    }

    private void resolveArchiveLocation(JobResolutionContext context) {
        context.setArchiveLocation(this.defaultArchiveLocation + context.getJobId());
    }

    private void resolveJobDirectory(JobResolutionContext context) {
        context.setJobDirectory(context.getJobRequest().getRequestedAgentConfig().getRequestedJobDirectoryLocation().orElse(this.defaultJobDirectory));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Map<Command, List<Criterion>> generateClusterCriteriaPermutations(Set<Command> commands, JobRequest jobRequest) {
        long start = System.nanoTime();
        try {
            ImmutableMap.Builder mapBuilder = ImmutableMap.builder();
            for (Command command : commands) {
                ImmutableList.Builder listBuilder = ImmutableList.builder();
                for (Criterion commandClusterCriterion : command.getClusterCriteria()) {
                    for (Criterion jobClusterCriterion : jobRequest.getCriteria().getClusterCriteria()) {
                        try {
                            listBuilder.add((Object)this.mergeCriteria(commandClusterCriterion, jobClusterCriterion));
                        }
                        catch (IllegalArgumentException e) {
                            log.debug("Unable to merge command cluster criterion {} and job cluster criterion {}. Skipping.", new Object[]{commandClusterCriterion, jobClusterCriterion, e});
                        }
                    }
                }
                mapBuilder.put((Object)command, (Object)listBuilder.build());
            }
            ImmutableMap immutableMap = mapBuilder.build();
            return immutableMap;
        }
        finally {
            this.registry.timer(GENERATE_CRITERIA_PERMUTATIONS_TIMER, new String[0]).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    private Set<Criterion> flattenClusterCriteriaPermutations(Map<Command, List<Criterion>> commandCriteriaMap) {
        return commandCriteriaMap.values().stream().flatMap(Collection::stream).collect(Collectors.toSet());
    }

    private boolean clusterMatchesCriterion(Cluster cluster, Criterion criterion) {
        ClusterMetadata metadata = cluster.getMetadata();
        return criterion.getId().map(id -> cluster.getId().equals(id)).orElse(true) != false && criterion.getName().map(name -> metadata.getName().equals(name)).orElse(true) != false && criterion.getVersion().map(version -> metadata.getVersion().equals(version)).orElse(true) != false && criterion.getStatus().map(status -> metadata.getStatus().name().equals(status)).orElse(true) != false && metadata.getTags().containsAll(criterion.getTags());
    }

    private Map<Command, Set<Cluster>> generateCommandClustersMap(Map<Command, List<Criterion>> commandClusterCriteria, Set<Cluster> candidateClusters) {
        ImmutableMap.Builder matrixBuilder = ImmutableMap.builder();
        block0: for (Map.Entry<Command, List<Criterion>> entry : commandClusterCriteria.entrySet()) {
            Command command = entry.getKey();
            ImmutableSet.Builder matchedClustersBuilder = ImmutableSet.builder();
            for (Criterion criterion : entry.getValue()) {
                for (Cluster candidateCluster : candidateClusters) {
                    if (!this.clusterMatchesCriterion(candidateCluster, criterion)) continue;
                    log.debug("Cluster {} matched criterion {} for command {}", new Object[]{candidateCluster.getId(), criterion, command.getId()});
                    matchedClustersBuilder.add((Object)candidateCluster);
                }
                ImmutableSet matchedClusters = matchedClustersBuilder.build();
                if (matchedClusters.isEmpty()) continue;
                matrixBuilder.put((Object)command, (Object)matchedClusters);
                log.debug("For command {} matched clusters {}", (Object)command, (Object)matchedClusters);
                continue block0;
            }
        }
        ImmutableMap matrix = matrixBuilder.build();
        log.debug("Complete command -> clusters matrix: {}", (Object)matrix);
        return matrix;
    }

    private Criterion mergeCriteria(Criterion one, Criterion two) throws IllegalArgumentException {
        Criterion.Builder builder = new Criterion.Builder();
        builder.withId(this.mergeCriteriaStrings(one.getId().orElse(null), two.getId().orElse(null), ID_FIELD));
        builder.withName(this.mergeCriteriaStrings(one.getName().orElse(null), two.getName().orElse(null), NAME_FIELD));
        builder.withStatus(this.mergeCriteriaStrings(one.getStatus().orElse(null), two.getStatus().orElse(null), STATUS_FIELD));
        builder.withVersion(this.mergeCriteriaStrings(one.getVersion().orElse(null), two.getVersion().orElse(null), VERSION_FIELD));
        HashSet tags = Sets.newHashSet((Iterable)one.getTags());
        tags.addAll(two.getTags());
        builder.withTags((Set)tags);
        return builder.build();
    }

    private String mergeCriteriaStrings(@Nullable String one, @Nullable String two, String fieldName) throws IllegalArgumentException {
        if (StringUtils.equals((CharSequence)one, (CharSequence)two)) {
            return one;
        }
        if (one == null) {
            return two;
        }
        if (two == null) {
            return one;
        }
        throw new IllegalArgumentException(fieldName + "'s were both present but not equal");
    }

    private String tagsToString(Set<String> tags) {
        ArrayList sortedTags = Lists.newArrayList(tags);
        sortedTags.sort(Comparator.naturalOrder());
        String joinedString = StringUtils.join((Iterable)sortedTags, (char)',');
        return RegExUtils.replaceAll((String)RegExUtils.replaceAll((String)joinedString, (String)"'", (String)"\\'"), (String)"\"", (String)"\\\"");
    }

    private String getProxyObjectClassName(Object possibleProxyObject) {
        Class targetClass;
        String className = possibleProxyObject instanceof TargetClassAware ? ((targetClass = ((TargetClassAware)possibleProxyObject).getTargetClass()) != null ? targetClass.getCanonicalName() : possibleProxyObject.getClass().getCanonicalName()) : possibleProxyObject.getClass().getCanonicalName();
        return className;
    }

    static class JobResolutionContext {
        private final String jobId;
        private final JobRequest jobRequest;
        private final boolean apiJob;
        private Command command;
        private Cluster cluster;
        private List<Application> applications;
        private Integer jobMemory;
        private Map<String, String> environmentVariables;
        private Integer timeout;
        private String archiveLocation;
        private File jobDirectory;
        private Map<Command, Set<Cluster>> commandClusters;

        Optional<Command> getCommand() {
            return Optional.ofNullable(this.command);
        }

        Optional<Cluster> getCluster() {
            return Optional.ofNullable(this.cluster);
        }

        Optional<List<Application>> getApplications() {
            return Optional.ofNullable(this.applications);
        }

        Optional<Integer> getJobMemory() {
            return Optional.ofNullable(this.jobMemory);
        }

        Optional<Map<String, String>> getEnvironmentVariables() {
            return Optional.ofNullable(this.environmentVariables);
        }

        Optional<Integer> getTimeout() {
            return Optional.ofNullable(this.timeout);
        }

        Optional<String> getArchiveLocation() {
            return Optional.ofNullable(this.archiveLocation);
        }

        Optional<File> getJobDirectory() {
            return Optional.ofNullable(this.jobDirectory);
        }

        Optional<Map<Command, Set<Cluster>>> getCommandClusters() {
            return Optional.ofNullable(this.commandClusters);
        }

        ResolvedJob build() {
            if (this.command == null) {
                throw new IllegalStateException("Command was never resolved for job " + this.jobId);
            }
            if (this.cluster == null) {
                throw new IllegalStateException("Cluster was never resolved for job " + this.jobId);
            }
            if (this.applications == null) {
                throw new IllegalStateException("Applications were never resolved for job " + this.jobId);
            }
            if (this.jobMemory == null) {
                throw new IllegalStateException("Job memory was never resolved for job " + this.jobId);
            }
            if (this.environmentVariables == null) {
                throw new IllegalStateException("Environment variables were never resolved for job " + this.jobId);
            }
            if (this.archiveLocation == null) {
                throw new IllegalStateException("Archive location was never resolved for job " + this.jobId);
            }
            if (this.jobDirectory == null) {
                throw new IllegalStateException("Job directory was never resolved for job " + this.jobId);
            }
            JobSpecification jobSpecification = new JobSpecification(this.command.getExecutable(), this.jobRequest.getCommandArgs(), new JobSpecification.ExecutionResource(this.jobId, this.jobRequest.getResources()), new JobSpecification.ExecutionResource(this.cluster.getId(), this.cluster.getResources()), new JobSpecification.ExecutionResource(this.command.getId(), this.command.getResources()), this.applications.stream().map(application -> new JobSpecification.ExecutionResource(application.getId(), application.getResources())).collect(Collectors.toList()), this.environmentVariables, this.jobRequest.getRequestedAgentConfig().isInteractive(), this.jobDirectory, this.archiveLocation, this.timeout);
            JobEnvironment jobEnvironment = new JobEnvironment.Builder(this.jobMemory.intValue()).withEnvironmentVariables(this.environmentVariables).build();
            return new ResolvedJob(jobSpecification, jobEnvironment, this.jobRequest.getMetadata());
        }

        public JobResolutionContext(String jobId, JobRequest jobRequest, boolean apiJob) {
            this.jobId = jobId;
            this.jobRequest = jobRequest;
            this.apiJob = apiJob;
        }

        public String getJobId() {
            return this.jobId;
        }

        public JobRequest getJobRequest() {
            return this.jobRequest;
        }

        public boolean isApiJob() {
            return this.apiJob;
        }

        public void setCommand(Command command) {
            this.command = command;
        }

        public void setCluster(Cluster cluster) {
            this.cluster = cluster;
        }

        public void setApplications(List<Application> applications) {
            this.applications = applications;
        }

        public void setJobMemory(Integer jobMemory) {
            this.jobMemory = jobMemory;
        }

        public void setEnvironmentVariables(Map<String, String> environmentVariables) {
            this.environmentVariables = environmentVariables;
        }

        public void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        public void setArchiveLocation(String archiveLocation) {
            this.archiveLocation = archiveLocation;
        }

        public void setJobDirectory(File jobDirectory) {
            this.jobDirectory = jobDirectory;
        }

        public void setCommandClusters(Map<Command, Set<Cluster>> commandClusters) {
            this.commandClusters = commandClusters;
        }

        public String toString() {
            return "JobResolverServiceImpl.JobResolutionContext(jobId=" + this.jobId + ", jobRequest=" + this.jobRequest + ", apiJob=" + this.apiJob + ", command=" + this.command + ", cluster=" + this.cluster + ", applications=" + this.applications + ", jobMemory=" + this.jobMemory + ", environmentVariables=" + this.environmentVariables + ", timeout=" + this.timeout + ", archiveLocation=" + this.archiveLocation + ", jobDirectory=" + this.jobDirectory + ", commandClusters=" + this.commandClusters + ")";
        }
    }
}

