/*
 * Decompiled with CFR 0.152.
 */
package alluxio.util.network;

import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.exception.status.AlluxioStatusException;
import alluxio.grpc.GetServiceVersionPRequest;
import alluxio.grpc.GrpcChannel;
import alluxio.grpc.GrpcChannelBuilder;
import alluxio.grpc.GrpcServerAddress;
import alluxio.grpc.ServiceVersionClientServiceGrpc;
import alluxio.security.user.UserState;
import alluxio.util.CommonUtils;
import alluxio.util.OSUtils;
import alluxio.util.network.NettyUtils;
import alluxio.wire.WorkerNetAddress;
import com.google.common.base.Preconditions;
import io.grpc.Channel;
import io.netty.channel.unix.DomainSocketAddress;
import java.io.IOException;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class NetworkAddressUtils {
    private static final Logger LOG = LoggerFactory.getLogger(NetworkAddressUtils.class);
    public static final String WILDCARD_ADDRESS = "0.0.0.0";
    public static final String UNKNOWN_HOSTNAME = "<UNKNOWN>";
    public static final boolean WINDOWS = OSUtils.isWindows();
    private static String sLocalHost;
    private static String sLocalHostMetricName;
    private static String sLocalIP;

    private NetworkAddressUtils() {
    }

    public static void assertValidPort(int port) {
        Preconditions.checkArgument((port < 65536 ? 1 : 0) != 0, (Object)"Port must be less than 65536");
        Preconditions.checkArgument((port >= 0 ? 1 : 0) != 0, (Object)"Port must be non-negative");
    }

    public static void assertValidPort(InetSocketAddress address) {
        NetworkAddressUtils.assertValidPort(address.getPort());
    }

    public static InetSocketAddress getConnectAddress(ServiceType service, AlluxioConfiguration conf) {
        return InetSocketAddress.createUnresolved(NetworkAddressUtils.getConnectHost(service, conf), NetworkAddressUtils.getPort(service, conf));
    }

    public static String getConnectHost(ServiceType service, AlluxioConfiguration conf) {
        String bindHost;
        String connectHost;
        if (conf.isSet(service.mHostNameKey) && !(connectHost = conf.get(service.mHostNameKey)).isEmpty() && !connectHost.equals(WILDCARD_ADDRESS)) {
            return connectHost;
        }
        if (conf.isSet(service.mBindHostKey) && !(bindHost = conf.get(service.mBindHostKey)).isEmpty() && !bindHost.equals(WILDCARD_ADDRESS)) {
            return bindHost;
        }
        if (conf.getBoolean(PropertyKey.NETWORK_IP_ADDRESS_USED)) {
            return NetworkAddressUtils.getLocalIpAddress((int)conf.getMs(PropertyKey.NETWORK_HOST_RESOLUTION_TIMEOUT_MS));
        }
        return NetworkAddressUtils.getLocalHostName((int)conf.getMs(PropertyKey.NETWORK_HOST_RESOLUTION_TIMEOUT_MS));
    }

    public static int getPort(ServiceType service, AlluxioConfiguration conf) {
        return conf.getInt(service.mPortKey);
    }

    public static InetSocketAddress getBindAddress(ServiceType service, AlluxioConfiguration conf) {
        int port = NetworkAddressUtils.getPort(service, conf);
        NetworkAddressUtils.assertValidPort(port);
        return new InetSocketAddress(NetworkAddressUtils.getBindHost(service, conf), NetworkAddressUtils.getPort(service, conf));
    }

    public static String getBindHost(ServiceType service, AlluxioConfiguration conf) {
        if (conf.isSet(service.mBindHostKey) && !conf.get(service.mBindHostKey).isEmpty()) {
            return conf.get(service.mBindHostKey);
        }
        return NetworkAddressUtils.getLocalHostName((int)conf.getMs(PropertyKey.NETWORK_HOST_RESOLUTION_TIMEOUT_MS));
    }

    public static String getClientHostName(AlluxioConfiguration conf) {
        if (conf.isSet(PropertyKey.USER_HOSTNAME)) {
            return conf.get(PropertyKey.USER_HOSTNAME);
        }
        if (conf.isSet(PropertyKey.LOCALITY_TIER_NODE)) {
            return conf.get(PropertyKey.LOCALITY_TIER_NODE);
        }
        return NetworkAddressUtils.getLocalHostName((int)conf.getMs(PropertyKey.NETWORK_HOST_RESOLUTION_TIMEOUT_MS));
    }

    public static String getLocalNodeName(AlluxioConfiguration conf) {
        switch (CommonUtils.PROCESS_TYPE.get()) {
            case JOB_MASTER: {
                if (!conf.isSet(PropertyKey.JOB_MASTER_HOSTNAME)) break;
                return conf.get(PropertyKey.JOB_MASTER_HOSTNAME);
            }
            case JOB_WORKER: {
                if (!conf.isSet(PropertyKey.JOB_WORKER_HOSTNAME)) break;
                return conf.get(PropertyKey.JOB_WORKER_HOSTNAME);
            }
            case CLIENT: {
                if (!conf.isSet(PropertyKey.USER_HOSTNAME)) break;
                return conf.get(PropertyKey.USER_HOSTNAME);
            }
            case MASTER: {
                if (!conf.isSet(PropertyKey.MASTER_HOSTNAME)) break;
                return conf.get(PropertyKey.MASTER_HOSTNAME);
            }
            case WORKER: {
                if (!conf.isSet(PropertyKey.WORKER_HOSTNAME)) break;
                return conf.get(PropertyKey.WORKER_HOSTNAME);
            }
        }
        if (conf.isSet(PropertyKey.LOCALITY_TIER_NODE)) {
            return conf.get(PropertyKey.LOCALITY_TIER_NODE);
        }
        return NetworkAddressUtils.getLocalHostName((int)conf.getMs(PropertyKey.NETWORK_HOST_RESOLUTION_TIMEOUT_MS));
    }

    public static synchronized String getLocalHostName(int timeoutMs) {
        if (sLocalHost != null) {
            return sLocalHost;
        }
        try {
            sLocalHost = InetAddress.getByName(NetworkAddressUtils.getLocalIpAddress(timeoutMs)).getCanonicalHostName();
            return sLocalHost;
        }
        catch (UnknownHostException e) {
            throw new RuntimeException(e);
        }
    }

    public static synchronized String getLocalHostMetricName(int timeoutMs) {
        if (sLocalHostMetricName != null) {
            return sLocalHostMetricName;
        }
        sLocalHostMetricName = NetworkAddressUtils.getLocalHostName(timeoutMs).replace('.', '_');
        return sLocalHostMetricName;
    }

    public static synchronized String getLocalIpAddress(int timeoutMs) {
        if (sLocalIP != null) {
            return sLocalIP;
        }
        try {
            InetAddress address = InetAddress.getLocalHost();
            LOG.debug("address: {} isLoopbackAddress: {}, with host {} {}", new Object[]{address, address.isLoopbackAddress(), address.getHostAddress(), address.getHostName()});
            if (!NetworkAddressUtils.isValidAddress(address, timeoutMs)) {
                Enumeration<NetworkInterface> networkInterfaces = NetworkInterface.getNetworkInterfaces();
                if (!WINDOWS) {
                    ArrayList<NetworkInterface> netIFs = Collections.list(networkInterfaces);
                    Collections.reverse(netIFs);
                    networkInterfaces = Collections.enumeration(netIFs);
                }
                while (networkInterfaces.hasMoreElements()) {
                    NetworkInterface ni = networkInterfaces.nextElement();
                    Enumeration<InetAddress> addresses = ni.getInetAddresses();
                    while (addresses.hasMoreElements()) {
                        address = addresses.nextElement();
                        if (!NetworkAddressUtils.isValidAddress(address, timeoutMs)) continue;
                        sLocalIP = address.getHostAddress();
                        return sLocalIP;
                    }
                }
                LOG.warn("Your hostname, {} resolves to a loopback/non-reachable address: {}, but we couldn't find any external IP address!", (Object)InetAddress.getLocalHost().getHostName(), (Object)address.getHostAddress());
            }
            sLocalIP = address.getHostAddress();
            return sLocalIP;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static boolean isServing(String host, int port) {
        if (port < 0) {
            return false;
        }
        try {
            Socket socket = new Socket(host, port);
            socket.close();
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    private static boolean isValidAddress(InetAddress address, int timeoutMs) throws IOException {
        return !address.isAnyLocalAddress() && !address.isLinkLocalAddress() && !address.isLoopbackAddress() && address.isReachable(timeoutMs) && address instanceof Inet4Address;
    }

    @Nullable
    public static String resolveHostName(String hostname) throws UnknownHostException {
        if (hostname == null || hostname.isEmpty()) {
            return null;
        }
        return InetAddress.getByName(hostname).getCanonicalHostName();
    }

    public static String resolveIpAddress(String hostname) throws UnknownHostException {
        Preconditions.checkNotNull((Object)hostname, (Object)"hostname");
        Preconditions.checkArgument((!hostname.isEmpty() ? 1 : 0) != 0, (Object)"Cannot resolve IP address for empty hostname");
        return InetAddress.getByName(hostname).getHostAddress();
    }

    public static String getFqdnHost(InetSocketAddress addr) {
        Preconditions.checkNotNull((Object)addr.getAddress(), (Object)("the address of " + addr + " is invalid."));
        return addr.getAddress().getCanonicalHostName();
    }

    public static String getFqdnHost(WorkerNetAddress addr) throws UnknownHostException {
        return NetworkAddressUtils.resolveHostName(addr.getHost());
    }

    @Nullable
    public static InetSocketAddress parseInetSocketAddress(String address) throws IOException {
        if (address == null) {
            return null;
        }
        String[] strArr = address.split(":");
        if (strArr.length != 2) {
            throw new IOException("Invalid InetSocketAddress " + address);
        }
        return InetSocketAddress.createUnresolved(strArr[0], Integer.parseInt(strArr[1]));
    }

    public static InetSocketAddress getRpcPortSocketAddress(WorkerNetAddress netAddress) {
        String host = netAddress.getHost();
        int port = netAddress.getRpcPort();
        return new InetSocketAddress(host, port);
    }

    public static SocketAddress getDataPortSocketAddress(WorkerNetAddress netAddress, AlluxioConfiguration conf) {
        Object address;
        if (NettyUtils.isDomainSocketAccessible(netAddress, conf)) {
            address = new DomainSocketAddress(netAddress.getDomainSocketPath());
        } else {
            String host = netAddress.getHost();
            if (!netAddress.getContainerHost().equals("")) {
                LOG.debug("Worker is in a container. Use container host {} instead of physical host {}", (Object)netAddress.getContainerHost(), (Object)host);
                host = netAddress.getContainerHost();
            }
            int port = netAddress.getDataPort();
            address = new InetSocketAddress(host, port);
        }
        return address;
    }

    public static void pingService(InetSocketAddress address, alluxio.grpc.ServiceType serviceType, AlluxioConfiguration conf, UserState userState) throws AlluxioStatusException {
        Preconditions.checkNotNull((Object)address, (Object)"address");
        Preconditions.checkNotNull((Object)serviceType, (Object)"serviceType");
        GrpcChannel channel = GrpcChannelBuilder.newBuilder(GrpcServerAddress.create(address), conf).setClientType("PingService").disableAuthentication().setSubject(userState.getSubject()).build();
        try {
            ServiceVersionClientServiceGrpc.ServiceVersionClientServiceBlockingStub versionClient = (ServiceVersionClientServiceGrpc.ServiceVersionClientServiceBlockingStub)ServiceVersionClientServiceGrpc.newBlockingStub((Channel)channel).withDeadlineAfter(conf.getMs(PropertyKey.USER_MASTER_POLLING_TIMEOUT), TimeUnit.MILLISECONDS);
            versionClient.getServiceVersion(GetServiceVersionPRequest.newBuilder().setServiceType(serviceType).build());
        }
        catch (Throwable t) {
            throw AlluxioStatusException.fromThrowable((Throwable)t);
        }
        finally {
            channel.shutdown();
        }
    }

    public static enum ServiceType {
        JOB_MASTER_RAFT("Alluxio Job Master Raft service", PropertyKey.JOB_MASTER_HOSTNAME, PropertyKey.JOB_MASTER_HOSTNAME, PropertyKey.JOB_MASTER_EMBEDDED_JOURNAL_PORT),
        MASTER_RAFT("Alluxio Master Raft service", PropertyKey.MASTER_HOSTNAME, PropertyKey.MASTER_HOSTNAME, PropertyKey.MASTER_EMBEDDED_JOURNAL_PORT),
        JOB_MASTER_RPC("Alluxio Job Manager Master RPC service", PropertyKey.JOB_MASTER_HOSTNAME, PropertyKey.JOB_MASTER_BIND_HOST, PropertyKey.JOB_MASTER_RPC_PORT),
        JOB_MASTER_WEB("Alluxio Job Manager Master Web service", PropertyKey.JOB_MASTER_WEB_HOSTNAME, PropertyKey.JOB_MASTER_WEB_BIND_HOST, PropertyKey.JOB_MASTER_WEB_PORT),
        JOB_WORKER_RPC("Alluxio Job Manager Worker RPC service", PropertyKey.JOB_WORKER_HOSTNAME, PropertyKey.JOB_WORKER_BIND_HOST, PropertyKey.JOB_WORKER_RPC_PORT),
        JOB_WORKER_WEB("Alluxio Job Manager Worker Web service", PropertyKey.WORKER_WEB_HOSTNAME, PropertyKey.JOB_WORKER_WEB_BIND_HOST, PropertyKey.JOB_WORKER_WEB_PORT),
        MASTER_RPC("Alluxio Master RPC service", PropertyKey.MASTER_HOSTNAME, PropertyKey.MASTER_BIND_HOST, PropertyKey.MASTER_RPC_PORT),
        MASTER_WEB("Alluxio Master Web service", PropertyKey.MASTER_WEB_HOSTNAME, PropertyKey.MASTER_WEB_BIND_HOST, PropertyKey.MASTER_WEB_PORT),
        WORKER_RPC("Alluxio Worker RPC service", PropertyKey.WORKER_HOSTNAME, PropertyKey.WORKER_BIND_HOST, PropertyKey.WORKER_RPC_PORT),
        WORKER_WEB("Alluxio Worker Web service", PropertyKey.WORKER_WEB_HOSTNAME, PropertyKey.WORKER_WEB_BIND_HOST, PropertyKey.WORKER_WEB_PORT),
        PROXY_WEB("Alluxio Proxy Web service", PropertyKey.PROXY_WEB_HOSTNAME, PropertyKey.PROXY_WEB_BIND_HOST, PropertyKey.PROXY_WEB_PORT);

        private final String mServiceName;
        private final PropertyKey mHostNameKey;
        private final PropertyKey mBindHostKey;
        private final PropertyKey mPortKey;

        private ServiceType(String serviceName, PropertyKey hostNameKey, PropertyKey bindHostKey, PropertyKey portKey) {
            this.mServiceName = serviceName;
            this.mHostNameKey = hostNameKey;
            this.mBindHostKey = bindHostKey;
            this.mPortKey = portKey;
        }

        public String getServiceName() {
            return this.mServiceName;
        }

        public PropertyKey getHostNameKey() {
            return this.mHostNameKey;
        }

        public PropertyKey getBindHostKey() {
            return this.mBindHostKey;
        }

        public PropertyKey getPortKey() {
            return this.mPortKey;
        }

        public int getDefaultPort() {
            return Integer.parseInt(this.mPortKey.getDefaultValue());
        }
    }
}

