/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.RegionTransition;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.ServerLoad;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.catalog.CatalogTracker;
import org.apache.hadoop.hbase.catalog.MetaReader;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.master.RegionState;
import org.apache.hadoop.hbase.master.RegionStateStore;
import org.apache.hadoop.hbase.master.ServerManager;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.zookeeper.ZKAssign;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperWatcher;
import org.apache.zookeeper.KeeperException;

@InterfaceAudience.Private
public class RegionStates {
    private static final Log LOG = LogFactory.getLog(RegionStates.class);
    final HashMap<String, RegionState> regionsInTransition;
    private final HashMap<String, RegionState> regionStates = new HashMap();
    private final Map<ServerName, Set<HRegionInfo>> serverHoldings;
    private final TreeMap<HRegionInfo, ServerName> regionAssignments;
    private final HashMap<String, ServerName> lastAssignments;
    private final HashMap<String, Long> deadServers;
    private final HashMap<ServerName, Long> processedServers;
    private long lastProcessedServerCleanTime;
    private final RegionStateStore regionStateStore;
    private final ServerManager serverManager;
    private final Server server;
    static final String LOG_SPLIT_TIME = "hbase.master.maximum.logsplit.keeptime";
    static final long DEFAULT_LOG_SPLIT_TIME = 0x6DDD00L;

    RegionStates(Server master, ServerManager serverManager, RegionStateStore regionStateStore) {
        this.regionsInTransition = new HashMap();
        this.serverHoldings = new HashMap<ServerName, Set<HRegionInfo>>();
        this.regionAssignments = new TreeMap();
        this.lastAssignments = new HashMap();
        this.processedServers = new HashMap();
        this.deadServers = new HashMap();
        this.regionStateStore = regionStateStore;
        this.serverManager = serverManager;
        this.server = master;
    }

    public synchronized Map<HRegionInfo, ServerName> getRegionAssignments() {
        return Collections.unmodifiableMap(this.regionAssignments);
    }

    public synchronized ServerName getRegionServerOfRegion(HRegionInfo hri) {
        return this.regionAssignments.get(hri);
    }

    public synchronized Map<String, RegionState> getRegionsInTransition() {
        return (Map)this.regionsInTransition.clone();
    }

    public synchronized boolean isRegionInTransition(HRegionInfo hri) {
        return this.regionsInTransition.containsKey(hri.getEncodedName());
    }

    public synchronized boolean isRegionInTransition(String encodedName) {
        return this.regionsInTransition.containsKey(encodedName);
    }

    public synchronized boolean isRegionsInTransition() {
        return !this.regionsInTransition.isEmpty();
    }

    public synchronized boolean isRegionOnline(HRegionInfo hri) {
        return !this.isRegionInTransition(hri) && this.regionAssignments.containsKey(hri);
    }

    public synchronized boolean isRegionOffline(HRegionInfo hri) {
        return this.getRegionState(hri) == null || !this.isRegionInTransition(hri) && this.isRegionInState(hri, RegionState.State.OFFLINE, RegionState.State.CLOSED);
    }

    public boolean isRegionInState(HRegionInfo hri, RegionState.State ... states) {
        return this.isRegionInState(hri.getEncodedName(), states);
    }

    public boolean isRegionInState(String encodedName, RegionState.State ... states) {
        RegionState regionState = this.getRegionState(encodedName);
        return RegionStates.isOneOfStates(regionState, states);
    }

    public synchronized void waitForUpdate(long timeout) throws InterruptedException {
        this.wait(timeout);
    }

    public RegionState getRegionTransitionState(HRegionInfo hri) {
        return this.getRegionTransitionState(hri.getEncodedName());
    }

    public synchronized RegionState getRegionTransitionState(String encodedName) {
        return this.regionsInTransition.get(encodedName);
    }

    public void createRegionStates(List<HRegionInfo> hris) {
        for (HRegionInfo hri : hris) {
            this.createRegionState(hri);
        }
    }

    public RegionState createRegionState(HRegionInfo hri) {
        return this.createRegionState(hri, null, null);
    }

    public synchronized RegionState createRegionState(HRegionInfo hri, RegionState.State newState, ServerName serverName) {
        String encodedName;
        RegionState regionState;
        if (newState == null || newState == RegionState.State.OPEN && serverName == null) {
            newState = RegionState.State.OFFLINE;
        }
        if (hri.isOffline() && hri.isSplit()) {
            newState = RegionState.State.SPLIT;
            serverName = null;
        }
        if ((regionState = this.regionStates.get(encodedName = hri.getEncodedName())) != null) {
            LOG.warn((Object)("Tried to create a state for a region already in RegionStates, used existing: " + regionState + ", ignored new: " + newState));
        } else {
            regionState = new RegionState(hri, newState, serverName);
            this.regionStates.put(encodedName, regionState);
            if (newState == RegionState.State.OPEN) {
                this.regionAssignments.put(hri, serverName);
                this.lastAssignments.put(encodedName, serverName);
                Set<HRegionInfo> regions = this.serverHoldings.get(serverName);
                if (regions == null) {
                    regions = new HashSet<HRegionInfo>();
                    this.serverHoldings.put(serverName, regions);
                }
                regions.add(hri);
            } else if (!regionState.isUnassignable()) {
                this.regionsInTransition.put(encodedName, regionState);
            }
        }
        return regionState;
    }

    public RegionState updateRegionState(HRegionInfo hri, RegionState.State state) {
        RegionState regionState = this.getRegionState(hri.getEncodedName());
        return this.updateRegionState(hri, state, regionState == null ? null : regionState.getServerName());
    }

    public RegionState updateRegionState(RegionTransition transition, RegionState.State state) {
        byte[] regionName = transition.getRegionName();
        HRegionInfo regionInfo = this.getRegionInfo(regionName);
        if (regionInfo == null) {
            String prettyRegionName = HRegionInfo.prettyPrint((String)HRegionInfo.encodeRegionName((byte[])regionName));
            LOG.warn((Object)("Failed to find region " + prettyRegionName + " in updating its state to " + state + " based on region transition " + transition));
            return null;
        }
        return this.updateRegionState(regionInfo, state, transition.getServerName());
    }

    public RegionState updateRegionState(HRegionInfo hri, RegionState.State state, ServerName serverName) {
        return this.updateRegionState(hri, state, serverName, -1L);
    }

    public void regionOnline(HRegionInfo hri, ServerName serverName) {
        this.regionOnline(hri, serverName, -1L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void regionOnline(HRegionInfo hri, ServerName serverName, long openSeqNum) {
        if (!this.serverManager.isServerOnline(serverName)) {
            LOG.warn((Object)("Ignored, " + hri.getEncodedName() + " was opened on a dead server: " + serverName));
            return;
        }
        this.updateRegionState(hri, RegionState.State.OPEN, serverName, openSeqNum);
        RegionStates regionStates = this;
        synchronized (regionStates) {
            this.regionsInTransition.remove(hri.getEncodedName());
            ServerName oldServerName = this.regionAssignments.put(hri, serverName);
            if (!serverName.equals((Object)oldServerName)) {
                LOG.info((Object)("Onlined " + hri.getShortNameToLog() + " on " + serverName));
                Set<HRegionInfo> regions = this.serverHoldings.get(serverName);
                if (regions == null) {
                    regions = new HashSet<HRegionInfo>();
                    this.serverHoldings.put(serverName, regions);
                }
                regions.add(hri);
                if (oldServerName != null) {
                    LOG.info((Object)("Offlined " + hri.getShortNameToLog() + " from " + oldServerName));
                    Set<HRegionInfo> oldRegions = this.serverHoldings.get(oldServerName);
                    oldRegions.remove(hri);
                    if (oldRegions.isEmpty()) {
                        this.serverHoldings.remove(oldServerName);
                    }
                }
            }
        }
    }

    public synchronized void logSplit(ServerName serverName) {
        Iterator<Map.Entry<String, ServerName>> it = this.lastAssignments.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, ServerName> e = it.next();
            if (!e.getValue().equals((Object)serverName)) continue;
            it.remove();
        }
        long now = System.currentTimeMillis();
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Adding to processed servers " + serverName));
        }
        this.processedServers.put(serverName, now);
        Configuration conf = this.server.getConfiguration();
        long obsoleteTime = conf.getLong(LOG_SPLIT_TIME, 0x6DDD00L);
        if (now > this.lastProcessedServerCleanTime + obsoleteTime) {
            this.lastProcessedServerCleanTime = now;
            long cutoff = now - obsoleteTime;
            Iterator<Map.Entry<ServerName, Long>> it2 = this.processedServers.entrySet().iterator();
            while (it2.hasNext()) {
                Map.Entry<ServerName, Long> e = it2.next();
                if (e.getValue() >= cutoff) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("Removed from processed servers " + e.getKey()));
                }
                it2.remove();
            }
        }
    }

    public void logSplit(HRegionInfo region) {
        this.clearLastAssignment(region);
    }

    public synchronized void clearLastAssignment(HRegionInfo region) {
        this.lastAssignments.remove(region.getEncodedName());
    }

    public void regionOffline(HRegionInfo hri) {
        this.regionOffline(hri, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void regionOffline(HRegionInfo hri, RegionState.State expectedState) {
        Preconditions.checkArgument((expectedState == null || RegionState.isUnassignable((RegionState.State)expectedState) ? 1 : 0) != 0, (Object)("Offlined region should not be " + expectedState));
        if (this.isRegionInState(hri, RegionState.State.SPLITTING_NEW, RegionState.State.MERGING_NEW)) {
            this.deleteRegion(hri);
            return;
        }
        RegionState.State newState = expectedState == null ? RegionState.State.OFFLINE : expectedState;
        this.updateRegionState(hri, newState);
        RegionStates regionStates = this;
        synchronized (regionStates) {
            this.regionsInTransition.remove(hri.getEncodedName());
            ServerName oldServerName = this.regionAssignments.remove(hri);
            if (oldServerName != null && this.serverHoldings.containsKey(oldServerName)) {
                LOG.info((Object)("Offlined " + hri.getShortNameToLog() + " from " + oldServerName));
                Set<HRegionInfo> oldRegions = this.serverHoldings.get(oldServerName);
                oldRegions.remove(hri);
                if (oldRegions.isEmpty()) {
                    this.serverHoldings.remove(oldServerName);
                }
            }
        }
    }

    public synchronized List<HRegionInfo> serverOffline(ZooKeeperWatcher watcher, ServerName sn) {
        ArrayList<HRegionInfo> rits = new ArrayList<HRegionInfo>();
        Set<HRegionInfo> assignedRegions = this.serverHoldings.get(sn);
        if (assignedRegions == null) {
            assignedRegions = new HashSet<HRegionInfo>();
        }
        HashSet<HRegionInfo> regionsToOffline = new HashSet<HRegionInfo>();
        for (HRegionInfo region : assignedRegions) {
            if (this.isRegionOnline(region)) {
                regionsToOffline.add(region);
                continue;
            }
            if (!this.isRegionInState(region, RegionState.State.SPLITTING, RegionState.State.MERGING)) continue;
            LOG.debug((Object)("Offline splitting/merging region " + this.getRegionState(region)));
            try {
                ZKAssign.deleteNodeFailSilent((ZooKeeperWatcher)watcher, (HRegionInfo)region);
                regionsToOffline.add(region);
            }
            catch (KeeperException ke) {
                this.server.abort("Unexpected ZK exception deleting node " + region, (Throwable)ke);
            }
        }
        for (HRegionInfo hri : regionsToOffline) {
            this.regionOffline(hri);
        }
        for (RegionState state : this.regionsInTransition.values()) {
            HRegionInfo hri = state.getRegion();
            if (assignedRegions.contains(hri)) {
                LOG.info((Object)("Transitioning " + state + " will be handled by SSH for " + sn));
                continue;
            }
            if (!sn.equals((Object)state.getServerName())) continue;
            if (state.isPendingOpenOrOpening() || state.isFailedClose() || state.isOffline()) {
                LOG.info((Object)("Found region in " + state + " to be reassigned by SSH for " + sn));
                rits.add(hri);
                continue;
            }
            LOG.warn((Object)("THIS SHOULD NOT HAPPEN: unexpected " + state));
        }
        this.notifyAll();
        return rits;
    }

    public synchronized List<HRegionInfo> getRegionsOfTable(TableName tableName) {
        ArrayList<HRegionInfo> tableRegions = new ArrayList<HRegionInfo>();
        HRegionInfo boundary = new HRegionInfo(tableName, null, null, false, 0L);
        for (HRegionInfo hri : this.regionAssignments.tailMap(boundary).keySet()) {
            if (!hri.getTable().equals((Object)tableName)) break;
            tableRegions.add(hri);
        }
        return tableRegions;
    }

    public synchronized void waitOnRegionToClearRegionsInTransition(HRegionInfo hri) throws InterruptedException {
        if (!this.isRegionInTransition(hri)) {
            return;
        }
        while (!this.server.isStopped() && this.isRegionInTransition(hri)) {
            RegionState rs = this.getRegionState(hri);
            LOG.info((Object)("Waiting on " + rs + " to clear regions-in-transition"));
            this.waitForUpdate(100L);
        }
        if (this.server.isStopped()) {
            LOG.info((Object)"Giving up wait on region in transition because stoppable.isStopped is set");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void tableDeleted(TableName tableName) {
        HashSet<HRegionInfo> regionsToDelete = new HashSet<HRegionInfo>();
        RegionStates regionStates = this;
        synchronized (regionStates) {
            for (RegionState state : this.regionStates.values()) {
                HRegionInfo region = state.getRegion();
                if (!region.getTable().equals((Object)tableName)) continue;
                regionsToDelete.add(region);
            }
        }
        for (HRegionInfo region : regionsToDelete) {
            this.deleteRegion(region);
        }
    }

    synchronized boolean wasRegionOnDeadServer(String encodedName) {
        ServerName server = this.lastAssignments.get(encodedName);
        return this.isServerDeadAndNotProcessed(server);
    }

    synchronized boolean isServerDeadAndNotProcessed(ServerName server) {
        if (server == null) {
            return false;
        }
        if (this.serverManager.isServerOnline(server)) {
            String hostAndPort = server.getHostAndPort();
            long startCode = server.getStartcode();
            Long deadCode = this.deadServers.get(hostAndPort);
            if (deadCode == null || startCode > deadCode) {
                if (this.serverManager.isServerReachable(server)) {
                    return false;
                }
                this.deadServers.put(hostAndPort, startCode);
            }
            LOG.warn((Object)("Couldn't reach online server " + server));
        }
        return !this.processedServers.containsKey(server);
    }

    synchronized ServerName getLastRegionServerOfRegion(String encodedName) {
        return this.lastAssignments.get(encodedName);
    }

    synchronized void setLastRegionServerOfRegions(ServerName serverName, List<HRegionInfo> regionInfos) {
        for (HRegionInfo hri : regionInfos) {
            this.setLastRegionServerOfRegion(serverName, hri.getEncodedName());
        }
    }

    synchronized void setLastRegionServerOfRegion(ServerName serverName, String encodedName) {
        this.lastAssignments.put(encodedName, serverName);
    }

    synchronized void closeAllUserRegions(Set<TableName> excludedTables) {
        HashSet<HRegionInfo> toBeClosed = new HashSet<HRegionInfo>(this.regionStates.size());
        for (RegionState state : this.regionStates.values()) {
            HRegionInfo hri = state.getRegion();
            TableName tableName = hri.getTable();
            if (hri.isSplit() || hri.isMetaRegion() || excludedTables.contains(tableName)) continue;
            toBeClosed.add(hri);
        }
        for (HRegionInfo hri : toBeClosed) {
            this.updateRegionState(hri, RegionState.State.CLOSED);
        }
    }

    protected synchronized double getAverageLoad() {
        int numServers = 0;
        int totalLoad = 0;
        for (Map.Entry<ServerName, Set<HRegionInfo>> e : this.serverHoldings.entrySet()) {
            Set<HRegionInfo> regions = e.getValue();
            ServerName serverName = e.getKey();
            int regionCount = regions.size();
            if (!this.serverManager.isServerOnline(serverName)) continue;
            totalLoad += regionCount;
            ++numServers;
        }
        return numServers == 0 ? 0.0 : (double)totalLoad / (double)numServers;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected Map<TableName, Map<ServerName, List<HRegionInfo>>> getAssignmentsByTable() {
        HashMap<TableName, Map<ServerName, List<HRegionInfo>>> result = new HashMap<TableName, Map<ServerName, List<HRegionInfo>>>();
        RegionStates regionStates = this;
        synchronized (regionStates) {
            if (!this.server.getConfiguration().getBoolean("hbase.master.loadbalance.bytable", false)) {
                HashMap svrToRegions = new HashMap(this.serverHoldings.size());
                for (Map.Entry<ServerName, Set<HRegionInfo>> e : this.serverHoldings.entrySet()) {
                    svrToRegions.put(e.getKey(), new ArrayList(e.getValue()));
                }
                result.put(TableName.valueOf((String)"ensemble"), svrToRegions);
            } else {
                for (Map.Entry<ServerName, Set<HRegionInfo>> e : this.serverHoldings.entrySet()) {
                    for (HRegionInfo hri : e.getValue()) {
                        ArrayList<HRegionInfo> regions;
                        if (hri.isMetaRegion()) continue;
                        TableName tablename = hri.getTable();
                        HashMap<ServerName, ArrayList<HRegionInfo>> svrToRegions = (HashMap<ServerName, ArrayList<HRegionInfo>>)result.get(tablename);
                        if (svrToRegions == null) {
                            svrToRegions = new HashMap<ServerName, ArrayList<HRegionInfo>>(this.serverHoldings.size());
                            result.put(tablename, svrToRegions);
                        }
                        if ((regions = (ArrayList<HRegionInfo>)svrToRegions.get(e.getKey())) == null) {
                            regions = new ArrayList<HRegionInfo>();
                            svrToRegions.put(e.getKey(), regions);
                        }
                        regions.add(hri);
                    }
                }
            }
        }
        Map<ServerName, ServerLoad> onlineSvrs = this.serverManager.getOnlineServers();
        for (Map map : result.values()) {
            for (ServerName svr : onlineSvrs.keySet()) {
                if (map.containsKey(svr)) continue;
                map.put(svr, new ArrayList());
            }
        }
        return result;
    }

    protected RegionState getRegionState(HRegionInfo hri) {
        return this.getRegionState(hri.getEncodedName());
    }

    protected synchronized RegionState getRegionState(String encodedName) {
        return this.regionStates.get(encodedName);
    }

    protected HRegionInfo getRegionInfo(byte[] regionName) {
        String encodedName = HRegionInfo.encodeRegionName((byte[])regionName);
        RegionState regionState = this.getRegionState(encodedName);
        if (regionState != null) {
            return regionState.getRegion();
        }
        try {
            HRegionInfo hri;
            Pair p = MetaReader.getRegion((CatalogTracker)this.server.getCatalogTracker(), (byte[])regionName);
            HRegionInfo hRegionInfo = hri = p == null ? null : (HRegionInfo)p.getFirst();
            if (hri != null) {
                this.createRegionState(hri);
            }
            return hri;
        }
        catch (IOException e) {
            this.server.abort("Aborting because error occoured while reading " + Bytes.toStringBinary((byte[])regionName) + " from hbase:meta", (Throwable)e);
            return null;
        }
    }

    static boolean isOneOfStates(RegionState regionState, RegionState.State ... states) {
        RegionState.State s = regionState != null ? regionState.getState() : null;
        for (RegionState.State state : states) {
            if (s != state) continue;
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private RegionState updateRegionState(HRegionInfo hri, RegionState.State state, ServerName serverName, long openSeqNum) {
        RegionState oldState;
        if (state == RegionState.State.FAILED_CLOSE || state == RegionState.State.FAILED_OPEN) {
            LOG.warn((Object)("Failed to open/close " + hri.getShortNameToLog() + " on " + serverName + ", set to " + state));
        }
        String encodedName = hri.getEncodedName();
        RegionState regionState = new RegionState(hri, state, System.currentTimeMillis(), serverName);
        if (!regionState.equals((Object)(oldState = this.getRegionState(encodedName)))) {
            LOG.info((Object)("Transition " + oldState + " to " + regionState));
            this.regionStateStore.updateRegionState(openSeqNum, regionState, oldState);
        }
        RegionStates regionStates = this;
        synchronized (regionStates) {
            ServerName last;
            this.regionsInTransition.put(encodedName, regionState);
            this.regionStates.put(encodedName, regionState);
            if ((state == RegionState.State.CLOSED || state == RegionState.State.MERGED || state == RegionState.State.SPLIT) && this.lastAssignments.containsKey(encodedName)) {
                last = this.lastAssignments.get(encodedName);
                if (last.equals((Object)serverName)) {
                    this.lastAssignments.remove(encodedName);
                } else {
                    LOG.warn((Object)(encodedName + " moved to " + state + " on " + serverName + ", expected " + last));
                }
            }
            if (serverName != null && state == RegionState.State.OPEN && !serverName.equals((Object)(last = this.lastAssignments.get(encodedName)))) {
                this.lastAssignments.put(encodedName, serverName);
                if (last != null && this.isServerDeadAndNotProcessed(last)) {
                    LOG.warn((Object)(encodedName + " moved to " + serverName + ", while it's previous host " + last + " is dead but not processed yet"));
                }
            }
            this.notifyAll();
        }
        return regionState;
    }

    private synchronized void deleteRegion(HRegionInfo hri) {
        String encodedName = hri.getEncodedName();
        this.regionsInTransition.remove(encodedName);
        this.regionStates.remove(encodedName);
        this.lastAssignments.remove(encodedName);
        ServerName sn = this.regionAssignments.remove(hri);
        if (sn != null) {
            Set<HRegionInfo> regions = this.serverHoldings.get(sn);
            regions.remove(hri);
        }
    }
}

