/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.state.internals;

import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.SimpleTimeZone;
import java.util.TreeMap;
import org.apache.kafka.streams.errors.ProcessorStateException;
import org.apache.kafka.streams.processor.internals.InternalProcessorContext;
import org.apache.kafka.streams.state.internals.Segment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Segments {
    private static final Logger log = LoggerFactory.getLogger(Segments.class);
    private final TreeMap<Long, Segment> segments = new TreeMap();
    private final String name;
    private final long retentionPeriod;
    private final long segmentInterval;
    private final SimpleDateFormat formatter;

    Segments(String name, long retentionPeriod, long segmentInterval) {
        this.name = name;
        this.segmentInterval = segmentInterval;
        this.retentionPeriod = retentionPeriod;
        this.formatter = new SimpleDateFormat("yyyyMMddHHmm");
        this.formatter.setTimeZone(new SimpleTimeZone(0, "UTC"));
    }

    long segmentId(long timestamp) {
        return timestamp / this.segmentInterval;
    }

    String segmentName(long segmentId) {
        return this.name + "." + segmentId * this.segmentInterval;
    }

    Segment getSegmentForTimestamp(long timestamp) {
        return this.segments.get(this.segmentId(timestamp));
    }

    Segment getOrCreateSegmentIfLive(long segmentId, InternalProcessorContext context) {
        long minLiveTimestamp = context.streamTime() - this.retentionPeriod;
        long minLiveSegment = this.segmentId(minLiveTimestamp);
        Segment toReturn = segmentId >= minLiveSegment ? this.getOrCreateSegment(segmentId, context) : null;
        this.cleanupEarlierThan(minLiveSegment);
        return toReturn;
    }

    private Segment getOrCreateSegment(long segmentId, InternalProcessorContext context) {
        if (this.segments.containsKey(segmentId)) {
            return this.segments.get(segmentId);
        }
        Segment newSegment = new Segment(this.segmentName(segmentId), this.name, segmentId);
        Segment shouldBeNull = this.segments.put(segmentId, newSegment);
        if (shouldBeNull != null) {
            throw new IllegalStateException("Segment already exists. Possible concurrent access.");
        }
        newSegment.openDB(context);
        return newSegment;
    }

    void openExisting(InternalProcessorContext context) {
        try {
            File dir = new File(context.stateDir(), this.name);
            if (dir.exists()) {
                String[] list = dir.list();
                if (list != null) {
                    long[] segmentIds = new long[list.length];
                    for (int i = 0; i < list.length; ++i) {
                        segmentIds[i] = this.segmentIdFromSegmentName(list[i], dir);
                    }
                    Arrays.sort(segmentIds);
                    for (long segmentId : segmentIds) {
                        if (segmentId < 0L) continue;
                        this.getOrCreateSegment(segmentId, context);
                    }
                }
            } else if (!dir.mkdir()) {
                throw new ProcessorStateException(String.format("dir %s doesn't exist and cannot be created for segments %s", dir, this.name));
            }
        }
        catch (Exception dir) {
            // empty catch block
        }
        long minLiveSegment = this.segmentId(context.streamTime() - this.retentionPeriod);
        this.cleanupEarlierThan(minLiveSegment);
    }

    List<Segment> segments(long timeFrom, long timeTo) {
        ArrayList<Segment> result = new ArrayList<Segment>();
        NavigableMap<Long, Segment> segmentsInRange = this.segments.subMap(this.segmentId(timeFrom), true, this.segmentId(timeTo), true);
        for (Segment segment : segmentsInRange.values()) {
            if (!segment.isOpen()) continue;
            result.add(segment);
        }
        return result;
    }

    List<Segment> allSegments() {
        ArrayList<Segment> result = new ArrayList<Segment>();
        for (Segment segment : this.segments.values()) {
            if (!segment.isOpen()) continue;
            result.add(segment);
        }
        return result;
    }

    void flush() {
        for (Segment segment : this.segments.values()) {
            segment.flush();
        }
    }

    public void close() {
        for (Segment segment : this.segments.values()) {
            segment.close();
        }
        this.segments.clear();
    }

    private void cleanupEarlierThan(long minLiveSegment) {
        Iterator toRemove = this.segments.headMap(minLiveSegment, false).entrySet().iterator();
        while (toRemove.hasNext()) {
            Map.Entry next = toRemove.next();
            toRemove.remove();
            Segment segment = (Segment)next.getValue();
            segment.close();
            try {
                segment.destroy();
            }
            catch (IOException e) {
                log.error("Error destroying {}", (Object)segment, (Object)e);
            }
        }
    }

    private long segmentIdFromSegmentName(String segmentName, File parent) {
        long segmentId;
        int segmentSeparatorIndex = this.name.length();
        char segmentSeparator = segmentName.charAt(segmentSeparatorIndex);
        String segmentIdString = segmentName.substring(segmentSeparatorIndex + 1);
        if (segmentSeparator == '-') {
            try {
                segmentId = this.formatter.parse(segmentIdString).getTime() / this.segmentInterval;
            }
            catch (ParseException e) {
                log.warn("Unable to parse segmentName {} to a date. This segment will be skipped", (Object)segmentName);
                return -1L;
            }
            this.renameSegmentFile(parent, segmentName, segmentId);
        } else {
            try {
                segmentId = Long.parseLong(segmentIdString) / this.segmentInterval;
            }
            catch (NumberFormatException e) {
                throw new ProcessorStateException("Unable to parse segment id as long from segmentName: " + segmentName);
            }
            if (segmentSeparator == ':') {
                this.renameSegmentFile(parent, segmentName, segmentId);
            }
        }
        return segmentId;
    }

    private void renameSegmentFile(File parent, String segmentName, long segmentId) {
        File oldName = new File(parent, segmentName);
        File newName = new File(parent, this.segmentName(segmentId));
        if (!oldName.renameTo(newName)) {
            throw new ProcessorStateException("Unable to rename old style segment from: " + oldName + " to new name: " + newName);
        }
    }
}

