/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.Restricted;
import org.apache.nifi.annotation.behavior.Restriction;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.RequiredPermission;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.logging.LogLevel;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.standard.GetFile;
import org.apache.nifi.processors.standard.ListFile;
import org.apache.nifi.processors.standard.PutFile;
import org.apache.nifi.util.StopWatch;

@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@Tags(value={"local", "files", "filesystem", "ingest", "ingress", "get", "source", "input", "fetch"})
@CapabilityDescription(value="Reads the contents of a file from disk and streams it into the contents of an incoming FlowFile. Once this is done, the file is optionally moved elsewhere or deleted to help keep the file system organized.")
@SeeAlso(value={GetFile.class, PutFile.class, ListFile.class})
@Restricted(restrictions={@Restriction(requiredPermission=RequiredPermission.READ_FILESYSTEM, explanation="Provides operator the ability to read from any file that NiFi has access to."), @Restriction(requiredPermission=RequiredPermission.WRITE_FILESYSTEM, explanation="Provides operator the ability to delete any file that NiFi has access to.")})
public class FetchFile
extends AbstractProcessor {
    static final AllowableValue COMPLETION_NONE = new AllowableValue("None", "None", "Leave the file as-is");
    static final AllowableValue COMPLETION_MOVE = new AllowableValue("Move File", "Move File", "Moves the file to the directory specified by the <Move Destination Directory> property");
    static final AllowableValue COMPLETION_DELETE = new AllowableValue("Delete File", "Delete File", "Deletes the original file from the file system");
    static final AllowableValue CONFLICT_REPLACE = new AllowableValue("Replace File", "Replace File", "The newly ingested file should replace the existing file in the Destination Directory");
    static final AllowableValue CONFLICT_KEEP_INTACT = new AllowableValue("Keep Existing", "Keep Existing", "The existing file should in the Destination Directory should stay intact and the newly ingested file should be deleted");
    static final AllowableValue CONFLICT_FAIL = new AllowableValue("Fail", "Fail", "The existing destination file should remain intact and the incoming FlowFile should be routed to failure");
    static final AllowableValue CONFLICT_RENAME = new AllowableValue("Rename", "Rename", "The existing destination file should remain intact. The newly ingested file should be moved to the destination directory but be renamed to a random filename");
    static final PropertyDescriptor FILENAME = new PropertyDescriptor.Builder().name("File to Fetch").description("The fully-qualified filename of the file to fetch from the file system").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).defaultValue("${absolute.path}/${filename}").required(true).build();
    static final PropertyDescriptor COMPLETION_STRATEGY = new PropertyDescriptor.Builder().name("Completion Strategy").description("Specifies what to do with the original file on the file system once it has been pulled into NiFi").expressionLanguageSupported(ExpressionLanguageScope.NONE).allowableValues(new AllowableValue[]{COMPLETION_NONE, COMPLETION_MOVE, COMPLETION_DELETE}).defaultValue(COMPLETION_NONE.getValue()).required(true).build();
    static final PropertyDescriptor MOVE_DESTINATION_DIR = new PropertyDescriptor.Builder().name("Move Destination Directory").description("The directory to the move the original file to once it has been fetched from the file system. This property is ignored unless the Completion Strategy is set to \"Move File\". If the directory does not exist, it will be created.").expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).required(false).build();
    static final PropertyDescriptor CONFLICT_STRATEGY = new PropertyDescriptor.Builder().name("Move Conflict Strategy").description("If Completion Strategy is set to Move File and a file already exists in the destination directory with the same name, this property specifies how that naming conflict should be resolved").allowableValues(new AllowableValue[]{CONFLICT_RENAME, CONFLICT_REPLACE, CONFLICT_KEEP_INTACT, CONFLICT_FAIL}).defaultValue(CONFLICT_RENAME.getValue()).required(true).build();
    static final PropertyDescriptor FILE_NOT_FOUND_LOG_LEVEL = new PropertyDescriptor.Builder().name("Log level when file not found").description("Log level to use in case the file does not exist when the processor is triggered").allowableValues((Enum[])LogLevel.values()).defaultValue(LogLevel.ERROR.toString()).required(true).build();
    static final PropertyDescriptor PERM_DENIED_LOG_LEVEL = new PropertyDescriptor.Builder().name("Log level when permission denied").description("Log level to use in case user " + System.getProperty("user.name") + " does not have sufficient permissions to read the file").allowableValues((Enum[])LogLevel.values()).defaultValue(LogLevel.ERROR.toString()).required(true).build();
    static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("Any FlowFile that is successfully fetched from the file system will be transferred to this Relationship.").build();
    static final Relationship REL_NOT_FOUND = new Relationship.Builder().name("not.found").description("Any FlowFile that could not be fetched from the file system because the file could not be found will be transferred to this Relationship.").build();
    static final Relationship REL_PERMISSION_DENIED = new Relationship.Builder().name("permission.denied").description("Any FlowFile that could not be fetched from the file system due to the user running NiFi not having sufficient permissions will be transferred to this Relationship.").build();
    static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("Any FlowFile that could not be fetched from the file system for any reason other than insufficient permissions or the file not existing will be transferred to this Relationship.").build();

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        ArrayList<PropertyDescriptor> properties = new ArrayList<PropertyDescriptor>();
        properties.add(FILENAME);
        properties.add(COMPLETION_STRATEGY);
        properties.add(MOVE_DESTINATION_DIR);
        properties.add(CONFLICT_STRATEGY);
        properties.add(FILE_NOT_FOUND_LOG_LEVEL);
        properties.add(PERM_DENIED_LOG_LEVEL);
        return properties;
    }

    public Set<Relationship> getRelationships() {
        HashSet<Relationship> relationships = new HashSet<Relationship>();
        relationships.add(REL_SUCCESS);
        relationships.add(REL_NOT_FOUND);
        relationships.add(REL_PERMISSION_DENIED);
        relationships.add(REL_FAILURE);
        return relationships;
    }

    protected Collection<ValidationResult> customValidate(ValidationContext validationContext) {
        ArrayList<ValidationResult> results = new ArrayList<ValidationResult>();
        if (COMPLETION_MOVE.getValue().equalsIgnoreCase(validationContext.getProperty(COMPLETION_STRATEGY).getValue()) && !validationContext.getProperty(MOVE_DESTINATION_DIR).isSet()) {
            results.add(new ValidationResult.Builder().subject(MOVE_DESTINATION_DIR.getName()).input(null).valid(false).explanation(MOVE_DESTINATION_DIR.getName() + " must be specified if " + COMPLETION_STRATEGY.getName() + " is set to " + COMPLETION_MOVE.getDisplayName()).build());
        }
        return results;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        File targetFile2;
        Object conflictStrategy;
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        StopWatch stopWatch = new StopWatch(true);
        String filename = context.getProperty(FILENAME).evaluateAttributeExpressions(flowFile).getValue();
        LogLevel levelFileNotFound = LogLevel.valueOf((String)context.getProperty(FILE_NOT_FOUND_LOG_LEVEL).getValue());
        LogLevel levelPermDenied = LogLevel.valueOf((String)context.getProperty(PERM_DENIED_LOG_LEVEL).getValue());
        File file = new File(filename);
        Path filePath = file.toPath();
        if (!Files.exists(filePath, new LinkOption[0]) && !Files.notExists(filePath, new LinkOption[0])) {
            this.getLogger().log(levelFileNotFound, "Could not fetch file {} from file system for {} because the existence of the file cannot be verified; routing to failure", new Object[]{file, flowFile});
            session.transfer(session.penalize(flowFile), REL_FAILURE);
            return;
        }
        if (!Files.exists(filePath, new LinkOption[0])) {
            this.getLogger().log(levelFileNotFound, "Could not fetch file {} from file system for {} because the file does not exist; routing to not.found", new Object[]{file, flowFile});
            session.getProvenanceReporter().route(flowFile, REL_NOT_FOUND);
            session.transfer(session.penalize(flowFile), REL_NOT_FOUND);
            return;
        }
        String user = System.getProperty("user.name");
        if (!this.isReadable(file)) {
            this.getLogger().log(levelPermDenied, "Could not fetch file {} from file system for {} due to user {} not having sufficient permissions to read the file; routing to permission.denied", new Object[]{file, flowFile, user});
            session.getProvenanceReporter().route(flowFile, REL_PERMISSION_DENIED);
            session.transfer(session.penalize(flowFile), REL_PERMISSION_DENIED);
            return;
        }
        String completionStrategy = context.getProperty(COMPLETION_STRATEGY).getValue();
        String targetDirectoryName = context.getProperty(MOVE_DESTINATION_DIR).evaluateAttributeExpressions(flowFile).getValue();
        if (targetDirectoryName != null) {
            File targetDir = new File(targetDirectoryName);
            if (COMPLETION_MOVE.getValue().equalsIgnoreCase(completionStrategy)) {
                if (!(!targetDir.exists() || this.isWritable(targetDir) && this.isDirectory(targetDir))) {
                    this.getLogger().error("Could not fetch file {} from file system for {} because Completion Strategy is configured to move the original file to {}, but that is not a directory or user {} does not have permissions to write to that directory", new Object[]{file, flowFile, targetDir, user});
                    session.transfer(flowFile, REL_FAILURE);
                    return;
                }
                if (!targetDir.exists()) {
                    try {
                        Files.createDirectories(targetDir.toPath(), new FileAttribute[0]);
                    }
                    catch (Exception e) {
                        this.getLogger().error("Could not fetch file {} from file system for {} because Completion Strategy is configured to move the original file to {}, but that directory does not exist and could not be created due to: {}", new Object[]{file, flowFile, targetDir, e.getMessage()}, (Throwable)e);
                        session.transfer(flowFile, REL_FAILURE);
                        return;
                    }
                }
                conflictStrategy = context.getProperty(CONFLICT_STRATEGY).getValue();
                if (CONFLICT_FAIL.getValue().equalsIgnoreCase((String)conflictStrategy) && (targetFile2 = new File(targetDir, file.getName())).exists()) {
                    this.getLogger().error("Could not fetch file {} from file system for {} because Completion Strategy is configured to move the original file to {}, but a file with name {} already exists in that directory and the Move Conflict Strategy is configured for failure", new Object[]{file, flowFile, targetDir, file.getName()});
                    session.transfer(flowFile, REL_FAILURE);
                    return;
                }
            }
        }
        try {
            FileInputStream fis = new FileInputStream(file);
            conflictStrategy = null;
            try {
                flowFile = session.importFrom((InputStream)fis, flowFile);
            }
            catch (Throwable targetFile2) {
                conflictStrategy = targetFile2;
                throw targetFile2;
            }
            finally {
                if (fis != null) {
                    if (conflictStrategy != null) {
                        try {
                            fis.close();
                        }
                        catch (Throwable targetFile2) {
                            ((Throwable)conflictStrategy).addSuppressed(targetFile2);
                        }
                    } else {
                        fis.close();
                    }
                }
            }
        }
        catch (IOException ioe) {
            this.getLogger().error("Could not fetch file {} from file system for {} due to {}; routing to failure", new Object[]{file, flowFile, ioe.toString()}, (Throwable)ioe);
            session.transfer(session.penalize(flowFile), REL_FAILURE);
            return;
        }
        session.getProvenanceReporter().fetch(flowFile, file.toURI().toString(), "Replaced content of FlowFile with contents of " + file.toURI(), stopWatch.getElapsed(TimeUnit.MILLISECONDS));
        session.transfer(flowFile, REL_SUCCESS);
        session.commit();
        IOException completionFailureException = null;
        if (COMPLETION_DELETE.getValue().equalsIgnoreCase(completionStrategy)) {
            try {
                this.delete(file);
            }
            catch (IOException ioe) {
                completionFailureException = ioe;
            }
        } else if (COMPLETION_MOVE.getValue().equalsIgnoreCase(completionStrategy)) {
            File targetDirectory = new File(targetDirectoryName);
            targetFile2 = new File(targetDirectory, file.getName());
            try {
                if (targetFile2.exists()) {
                    String conflictStrategy2 = context.getProperty(CONFLICT_STRATEGY).getValue();
                    if (CONFLICT_KEEP_INTACT.getValue().equalsIgnoreCase(conflictStrategy2)) {
                        Files.delete(file.toPath());
                    } else if (CONFLICT_RENAME.getValue().equalsIgnoreCase(conflictStrategy2)) {
                        String simpleFilename = targetFile2.getName();
                        String newName = simpleFilename.contains(".") ? StringUtils.substringBeforeLast((String)simpleFilename, (String)".") + "-" + UUID.randomUUID().toString() + "." + StringUtils.substringAfterLast((String)simpleFilename, (String)".") : simpleFilename + "-" + UUID.randomUUID().toString();
                        this.move(file, new File(targetDirectory, newName), false);
                    } else if (CONFLICT_REPLACE.getValue().equalsIgnoreCase(conflictStrategy2)) {
                        this.move(file, targetFile2, true);
                    }
                } else {
                    this.move(file, targetFile2, false);
                }
            }
            catch (IOException ioe) {
                completionFailureException = ioe;
            }
        }
        if (completionFailureException != null) {
            this.getLogger().warn("Successfully fetched the content from {} for {} but failed to perform Completion Action due to {}; routing to success", new Object[]{file, flowFile, completionFailureException}, (Throwable)completionFailureException);
        }
    }

    protected void move(File source, File target, boolean overwrite) throws IOException {
        CopyOption[] copyOptionArray;
        File targetDirectory = target.getParentFile();
        Path targetPath = target.toPath();
        if (!targetDirectory.exists()) {
            Files.createDirectories(targetDirectory.toPath(), new FileAttribute[0]);
        }
        if (overwrite) {
            CopyOption[] copyOptionArray2 = new CopyOption[1];
            copyOptionArray = copyOptionArray2;
            copyOptionArray2[0] = StandardCopyOption.REPLACE_EXISTING;
        } else {
            copyOptionArray = new CopyOption[]{};
        }
        CopyOption[] copyOptions = copyOptionArray;
        Files.move(source.toPath(), targetPath, copyOptions);
    }

    protected void delete(File file) throws IOException {
        Files.delete(file.toPath());
    }

    protected boolean isReadable(File file) {
        return file.canRead();
    }

    protected boolean isWritable(File file) {
        return file.canWrite();
    }

    protected boolean isDirectory(File file) {
        return file.isDirectory();
    }
}

