/*
 * Decompiled with CFR 0.152.
 */
package org.ballerinalang.util.debugger;

import io.netty.channel.Channel;
import java.util.List;
import java.util.concurrent.Semaphore;
import org.ballerinalang.bre.bvm.BLangScheduler;
import org.ballerinalang.bre.bvm.WorkerExecutionContext;
import org.ballerinalang.model.values.BBlob;
import org.ballerinalang.model.values.BBoolean;
import org.ballerinalang.model.values.BFloat;
import org.ballerinalang.model.values.BInteger;
import org.ballerinalang.model.values.BString;
import org.ballerinalang.util.codegen.LineNumberInfo;
import org.ballerinalang.util.codegen.ProgramFile;
import org.ballerinalang.util.codegen.attributes.AttributeInfo;
import org.ballerinalang.util.codegen.attributes.LocalVariableAttributeInfo;
import org.ballerinalang.util.debugger.DebugClientHandler;
import org.ballerinalang.util.debugger.DebugCommand;
import org.ballerinalang.util.debugger.DebugException;
import org.ballerinalang.util.debugger.DebugInfoHolder;
import org.ballerinalang.util.debugger.DebugServer;
import org.ballerinalang.util.debugger.VMDebugClientHandler;
import org.ballerinalang.util.debugger.dto.BreakPointDTO;
import org.ballerinalang.util.debugger.dto.CommandDTO;
import org.ballerinalang.util.debugger.dto.FrameDTO;
import org.ballerinalang.util.debugger.dto.MessageDTO;
import org.ballerinalang.util.debugger.dto.VariableDTO;
import org.ballerinalang.util.debugger.util.DebugMsgUtil;

public class Debugger {
    private boolean debugEnabled = false;
    private ProgramFile programFile;
    private DebugClientHandler clientHandler;
    private DebugInfoHolder debugInfoHolder;
    private volatile Semaphore executionSem;

    public Debugger(ProgramFile programFile) {
        this.programFile = programFile;
        String debug = System.getProperty("debug");
        if (debug != null && !debug.isEmpty()) {
            this.debugEnabled = true;
        }
    }

    public void init() {
        this.setupDebugger();
        this.setClientHandler(new VMDebugClientHandler());
        DebugServer debugServer = new DebugServer(this);
        debugServer.startServer();
        Runtime.getRuntime().addShutdownHook(new DebuggerShutDownHook(this, debugServer));
    }

    protected void setupDebugger() {
        this.executionSem = new Semaphore(0);
        this.debugInfoHolder = new DebugInfoHolder(this.programFile);
    }

    public void waitTillDebuggeeResponds() {
        try {
            this.executionSem.acquire();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public void pauseWorker(WorkerExecutionContext ctx) {
        ctx.getDebugContext().setWorkerPaused(true);
        BLangScheduler.workerPaused(ctx);
    }

    public LineNumberInfo getLineNumber(String packagePath, int ip) {
        return this.debugInfoHolder.getLineNumber(packagePath, ip);
    }

    void processDebugCommand(String json) {
        try {
            this.processCommand(json);
        }
        catch (Exception e) {
            MessageDTO message = new MessageDTO("INVALID", e.getMessage());
            this.clientHandler.sendCustomMsg(message);
            this.stopDebugging();
        }
    }

    private void processCommand(String json) {
        CommandDTO command;
        try {
            command = DebugMsgUtil.buildCommandDTO(json);
        }
        catch (Exception e) {
            throw new DebugException("Invalid Command");
        }
        switch (command.getCommand()) {
            case "RESUME": {
                this.resume(command.getThreadId());
                break;
            }
            case "STEP_OVER": {
                this.stepOver(command.getThreadId());
                break;
            }
            case "STEP_IN": {
                this.stepIn(command.getThreadId());
                break;
            }
            case "STEP_OUT": {
                this.stepOut(command.getThreadId());
                break;
            }
            case "STOP": {
                this.stopDebugging();
                break;
            }
            case "SET_POINTS": {
                this.addDebugPoints(command.getPoints());
                this.sendAcknowledge("Debug points updated");
                break;
            }
            case "START": {
                this.sendAcknowledge("Debug started.");
                this.startDebug();
                break;
            }
            default: {
                throw new DebugException("Invalid Command");
            }
        }
    }

    public void addDebugPoints(List<BreakPointDTO> breakPointDTOS) {
        this.debugInfoHolder.addDebugPoints(breakPointDTOS);
    }

    public void startDebug() {
        this.executionSem.release();
    }

    public void resume(String workerId) {
        WorkerExecutionContext ctx = this.getWorkerContext(workerId);
        ctx.getDebugContext().setCurrentCommand(DebugCommand.RESUME);
        BLangScheduler.resume(ctx);
    }

    public void stepIn(String workerId) {
        WorkerExecutionContext ctx = this.getWorkerContext(workerId);
        ctx.getDebugContext().setCurrentCommand(DebugCommand.STEP_IN);
        BLangScheduler.resume(ctx);
    }

    public void stepOver(String workerId) {
        WorkerExecutionContext ctx = this.getWorkerContext(workerId);
        ctx.getDebugContext().setCurrentCommand(DebugCommand.STEP_OVER);
        BLangScheduler.resume(ctx);
    }

    public void stepOut(String workerId) {
        WorkerExecutionContext ctx = this.getWorkerContext(workerId);
        ctx.getDebugContext().setCurrentCommand(DebugCommand.STEP_OUT);
        BLangScheduler.resume(ctx);
    }

    public void stopDebugging() {
        this.debugInfoHolder.clearDebugLocations();
        this.clientHandler.getAllWorkerContexts().forEach((k, v) -> {
            v.getDebugContext().setCurrentCommand(DebugCommand.RESUME);
            BLangScheduler.resume(v);
        });
        this.clientHandler.clearChannel();
    }

    private WorkerExecutionContext getWorkerContext(String workerId) {
        WorkerExecutionContext ctx = this.clientHandler.getWorkerContext(workerId);
        if (ctx == null) {
            throw new DebugException("Invalid Worker ID : " + workerId);
        }
        return ctx;
    }

    void addDebugSession(Channel channel) throws DebugException {
        this.clientHandler.setChannel(channel);
        this.sendAcknowledge("Channel registered.");
    }

    public void addWorkerContext(WorkerExecutionContext ctx) {
        this.clientHandler.addWorkerContext(ctx);
    }

    public boolean isClientSessionActive() {
        return this.clientHandler.isChannelActive();
    }

    public void notifyDebugHit(WorkerExecutionContext ctx, LineNumberInfo currentExecLine, String workerId) {
        MessageDTO message = this.generateDebugHitMessage(ctx, currentExecLine, workerId);
        this.clientHandler.notifyHalt(message);
    }

    public void notifyExit() {
        if (!this.isClientSessionActive()) {
            return;
        }
        this.clientHandler.notifyExit();
    }

    private void sendAcknowledge(String messageText) {
        MessageDTO message = new MessageDTO("ACK", messageText);
        this.clientHandler.sendCustomMsg(message);
    }

    private MessageDTO generateDebugHitMessage(WorkerExecutionContext ctx, LineNumberInfo currentExecLine, String workerId) {
        MessageDTO message = new MessageDTO("DEBUG_HIT", "Debug point hit.");
        message.setThreadId(workerId);
        BreakPointDTO breakPointDTO = new BreakPointDTO(currentExecLine.getPackageInfo().getPkgPath(), currentExecLine.getFileName(), currentExecLine.getLineNumber());
        message.setLocation(breakPointDTO);
        int callingIp = currentExecLine.getIp();
        String pck = ctx.callableUnitInfo.getPackageInfo().getPkgPath();
        String functionName = ctx.callableUnitInfo.getName();
        LineNumberInfo callingLine = this.getLineNumber(ctx.callableUnitInfo.getPackageInfo().getPkgPath(), callingIp);
        FrameDTO frameDTO = new FrameDTO(pck, functionName, callingLine.getFileName(), callingLine.getLineNumber());
        message.addFrame(frameDTO);
        LocalVariableAttributeInfo localVarAttrInfo = (LocalVariableAttributeInfo)ctx.workerInfo.getAttributeInfo(AttributeInfo.Kind.LOCAL_VARIABLES_ATTRIBUTE);
        localVarAttrInfo.getLocalVariables().forEach(l -> {
            VariableDTO variableDTO = new VariableDTO(l.getVariableName(), "Local");
            switch (l.getVariableType().getTag()) {
                case 1: {
                    variableDTO.setBValue(new BInteger(workerExecutionContext.workerLocal.longRegs[l.getVariableIndex()]));
                    break;
                }
                case 2: {
                    variableDTO.setBValue(new BFloat(workerExecutionContext.workerLocal.doubleRegs[l.getVariableIndex()]));
                    break;
                }
                case 3: {
                    variableDTO.setBValue(new BString(workerExecutionContext.workerLocal.stringRegs[l.getVariableIndex()]));
                    break;
                }
                case 4: {
                    variableDTO.setBValue(new BBoolean(workerExecutionContext.workerLocal.intRegs[l.getVariableIndex()] == 1));
                    break;
                }
                case 5: {
                    variableDTO.setBValue(new BBlob(workerExecutionContext.workerLocal.byteRegs[l.getVariableIndex()]));
                    break;
                }
                default: {
                    variableDTO.setBValue(workerExecutionContext.workerLocal.refRegs[l.getVariableIndex()]);
                }
            }
            frameDTO.addVariable(variableDTO);
        });
        return message;
    }

    public boolean isDebugEnabled() {
        return this.debugEnabled;
    }

    public void setDebugEnabled() {
        this.debugEnabled = true;
    }

    protected void setClientHandler(DebugClientHandler clientHandler) {
        this.clientHandler = clientHandler;
    }

    static class DebuggerShutDownHook
    extends Thread {
        private Debugger debugger;
        private DebugServer debugServer;

        DebuggerShutDownHook(Debugger debugger, DebugServer debugServer) {
            this.debugger = debugger;
            this.debugServer = debugServer;
        }

        @Override
        public void run() {
            this.debugger.notifyExit();
            this.debugServer.closeServerChannel();
        }
    }
}

