/**********************************************************************
 Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Contributors:
 ...
 **********************************************************************/
package org.datanucleus.store.rdbms.scostore;

import java.sql.PreparedStatement;
import java.sql.SQLException;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.store.ExecutionContext;
import org.datanucleus.store.ObjectProvider;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.exceptions.NotYetFlushedException;
import org.datanucleus.store.mapped.exceptions.MappedDatastoreException;
import org.datanucleus.store.rdbms.RDBMSStoreManager;
import org.datanucleus.store.rdbms.SQLController;
import org.datanucleus.util.Localiser;

/**
 * RDBMS-specific implementation of an {@link AbstractArrayStoreSpecialization}.
 */
abstract class RDBMSAbstractArrayStoreSpecialization extends RDBMSAbstractCollectionStoreSpecialization 
    implements AbstractArrayStoreSpecialization
{
    RDBMSAbstractArrayStoreSpecialization(Localiser localiser, ClassLoaderResolver clr, 
            RDBMSStoreManager storeMgr)
    {
        super(localiser, clr, storeMgr);
    }

    public void clear(ObjectProvider ownerSM, ElementContainerStore ecs)
    {
        String clearStmt = getClearStmt(ecs);
        try
        {
            ExecutionContext ec = ownerSM.getExecutionContext();
            ManagedConnection mconn = ecs.getStoreManager().getConnection(ec);
            SQLController sqlControl = storeMgr.getSQLController();
            try
            {
                PreparedStatement ps = sqlControl.getStatementForUpdate(mconn, clearStmt, false);
                try
                {
                    int jdbcPosition = 1;
                    jdbcPosition = BackingStoreHelper.populateOwnerInStatement(ownerSM, ec, ps, jdbcPosition, ecs);
                    if (ecs.getRelationDiscriminatorMapping() != null)
                    {
                        BackingStoreHelper.populateRelationDiscriminatorInStatement(ec, ps, jdbcPosition, ecs);
                    }

                    sqlControl.executeStatementUpdate(mconn, clearStmt, ps, true);
                }
                finally
                {
                    sqlControl.closeStatement(mconn, ps);
                }
            }
            finally
            {
                mconn.release();
            }
        }
        catch (SQLException e)
        {
            throw new NucleusDataStoreException(localiser.msg("056013", clearStmt), e);
        }
    }

    /**
     * Internal method to add a row to the join table.
     * Used by add() and set() to add a row to the join table.
     * @param sm StateManager for the owner of the collection
     * @param element The element to add the relation to
     * @param conn The connection
     * @param batched Whether we are batching
     * @param orderId The order id to use for this element relation
     * @param executeNow Whether to execute the statement now (and not wait for any batch)
     * @return Whether a row was inserted
     * @throws SQLException Thrown if an error occurs
     */
    public int[] internalAdd(ObjectProvider sm, AbstractArrayStore aas, Object element, ManagedConnection conn, 
            boolean batched, int orderId, boolean executeNow)
        throws MappedDatastoreException
    {
        ExecutionContext ec = sm.getExecutionContext();
        SQLController sqlControl = storeMgr.getSQLController();
        String addStmt = getAddStmt(aas);
        try
        {
            PreparedStatement ps = sqlControl.getStatementForUpdate(conn, addStmt, false);
            boolean notYetFlushedError = false;
            try
            {
                // Insert the join table row
                int jdbcPosition = 1;
                jdbcPosition = BackingStoreHelper.populateOwnerInStatement(sm, ec, ps, jdbcPosition, aas);
                jdbcPosition = BackingStoreHelper.populateElementInStatement(ec, ps, element, jdbcPosition, aas.getElementMapping());
                jdbcPosition = BackingStoreHelper.populateOrderInStatement(ec, ps, orderId, jdbcPosition, aas.getOrderMapping());
                if (aas.getRelationDiscriminatorMapping() != null)
                {
                    jdbcPosition =
                        BackingStoreHelper.populateRelationDiscriminatorInStatement(ec, ps, jdbcPosition, aas);
                }

                // Execute the statement
                return sqlControl.executeStatementUpdate(conn, addStmt, ps, executeNow);
            }
            catch (NotYetFlushedException nfe)
            {
                notYetFlushedError = true;
                throw nfe;
            }
            finally
            {
                if (notYetFlushedError)
                {
                    sqlControl.abortStatementForConnection(conn, ps);
                }
                else
                {
                    sqlControl.closeStatement(conn, ps);
                }
            }
        }
        catch (SQLException e)
        {
            throw new MappedDatastoreException(addStmt, e);
        }
    }

    public void processBatchedWrites(ManagedConnection mconn) throws MappedDatastoreException
    {
        SQLController sqlControl = storeMgr.getSQLController();
        try
        {
            sqlControl.processStatementsForConnection(mconn); // Process all waiting batched statements before we start our work
        }
        catch (SQLException e)
        {
            throw new MappedDatastoreException("SQLException", e);
        }
    }
}