/*
 * Copyright (c) 2003-2006 OFFIS, Henri Tremblay. 
 * This program is made available under the terms of the MIT License.
 */
package org.easymock.classextension.internal;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.sf.cglib.core.CollectionUtils;
import net.sf.cglib.core.VisibilityPredicate;
import net.sf.cglib.proxy.Callback;
import net.sf.cglib.proxy.Enhancer;
import net.sf.cglib.proxy.MethodInterceptor;
import net.sf.cglib.proxy.MethodProxy;

import org.easymock.internal.IProxyFactory;
import org.easymock.internal.ObjectMethodsFilter;

/**
 * Factory generating a mock for a class.
 * <p>
 * Note that this class is stateful
 */
public class ClassProxyFactory<T> implements IProxyFactory<T> {

    public static interface MockMethodInterceptor extends MethodInterceptor {
        InvocationHandler getHandler();

        void setMockedMethods(Method[] mockedMethods);
    }

    public T createProxy(Class<T> toMock, final InvocationHandler handler) {

        // Dirty trick to fix ObjectMethodsFilter
        // It will replace the equals, hashCode, toString methods it kept that
        // are the ones
        // from Object.class by the correct ones since they might have been
        // overloaded
        // in the mocked class.
        try {
            updateMethod(handler, toMock.getMethod("equals",
                    new Class[] { Object.class }));
            updateMethod(handler, toMock.getMethod("hashCode", new Class[0]));
            updateMethod(handler, toMock.getMethod("toString", new Class[0]));
        } catch (NoSuchMethodException e) {
            // ///CLOVER:OFF
            throw new InternalError(
                    "We strangly failed to retrieve methods that always exist on an object...");
            // ///CLOVER:ON
        }

        MethodInterceptor interceptor = new MockMethodInterceptor() {

            private Set<Method> mockedMethods;

            public Object intercept(Object obj, Method method, Object[] args,
                    MethodProxy proxy) throws Throwable {
                if (mockedMethods != null && !mockedMethods.contains(method)) {
                    return proxy.invokeSuper(obj, args);
                }
                return handler.invoke(obj, method, args);
            }

            public InvocationHandler getHandler() {
                return handler;
            }

            public void setMockedMethods(Method[] mockedMethods) {
                this.mockedMethods = new HashSet<Method>(Arrays
                        .asList(mockedMethods));
            }
        };

        // Create the mock
        Enhancer enhancer = new Enhancer() {
            /**
             * Filter all private constructors but do not check that there are
             * some left
             */
            protected void filterConstructors(Class sc, List constructors) {
                CollectionUtils.filter(constructors, new VisibilityPredicate(
                        sc, true));
            }
        };
        enhancer.setSuperclass(toMock);
        enhancer.setCallbackType(interceptor.getClass());
        // do not use cache to prevent having the same class since we
        // are registering the callback for a given class and a callback
        // must be used by only one mock instance (they are stateful)
        enhancer.setUseCache(false);

        Class mock = enhancer.createClass();
        Enhancer.registerCallbacks(mock, new Callback[] { interceptor });
        try {
            return (T) ClassInstantiatorFactory.getInstantiator().newInstance(
                    mock);
        } catch (InstantiationException e) {
            // ///CLOVER:OFF
            throw new RuntimeException("Fail to instantiate mock for " + toMock
                    + " on " + ClassInstantiatorFactory.getJVM() + " JVM");
            // ///CLOVER:ON
        }
    }

    private void updateMethod(InvocationHandler objectMethodsFilter,
            Method correctMethod) {
        Field methodField = retrieveField(ObjectMethodsFilter.class,
                correctMethod.getName() + "Method");
        updateField(objectMethodsFilter, correctMethod, methodField);
    }

    private Field retrieveField(Class clazz, String field) {
        try {
            return clazz.getDeclaredField(field);
        } catch (NoSuchFieldException e) {
            // ///CLOVER:OFF
            throw new InternalError(
                    "There must be some refactoring because the " + field
                            + " field was there...");
            // ///CLOVER:ON
        }
    }

    private void updateField(Object instance, Object value, Field field) {
        boolean accessible = field.isAccessible();
        field.setAccessible(true);
        try {
            field.set(instance, value);
        } catch (IllegalAccessException e) {
            // ///CLOVER:OFF
            throw new InternalError(
                    "Should be accessible since we set it ourselves");
            // ///CLOVER:ON
        }
        field.setAccessible(accessible);
    }
}