/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a network interface.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceNetworkInterface implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceNetworkInterface.Builder, InstanceNetworkInterface> {
    private static final SdkField<InstanceNetworkInterfaceAssociation> ASSOCIATION_FIELD = SdkField
            .<InstanceNetworkInterfaceAssociation> builder(MarshallingType.SDK_POJO)
            .memberName("Association")
            .getter(getter(InstanceNetworkInterface::association))
            .setter(setter(Builder::association))
            .constructor(InstanceNetworkInterfaceAssociation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Association")
                    .unmarshallLocationName("association").build()).build();

    private static final SdkField<InstanceNetworkInterfaceAttachment> ATTACHMENT_FIELD = SdkField
            .<InstanceNetworkInterfaceAttachment> builder(MarshallingType.SDK_POJO)
            .memberName("Attachment")
            .getter(getter(InstanceNetworkInterface::attachment))
            .setter(setter(Builder::attachment))
            .constructor(InstanceNetworkInterfaceAttachment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attachment")
                    .unmarshallLocationName("attachment").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(InstanceNetworkInterface::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<List<GroupIdentifier>> GROUPS_FIELD = SdkField
            .<List<GroupIdentifier>> builder(MarshallingType.LIST)
            .memberName("Groups")
            .getter(getter(InstanceNetworkInterface::groups))
            .setter(setter(Builder::groups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupSet")
                    .unmarshallLocationName("groupSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<GroupIdentifier> builder(MarshallingType.SDK_POJO)
                                            .constructor(GroupIdentifier::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<InstanceIpv6Address>> IPV6_ADDRESSES_FIELD = SdkField
            .<List<InstanceIpv6Address>> builder(MarshallingType.LIST)
            .memberName("Ipv6Addresses")
            .getter(getter(InstanceNetworkInterface::ipv6Addresses))
            .setter(setter(Builder::ipv6Addresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6AddressesSet")
                    .unmarshallLocationName("ipv6AddressesSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<InstanceIpv6Address> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceIpv6Address::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> MAC_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MacAddress")
            .getter(getter(InstanceNetworkInterface::macAddress))
            .setter(setter(Builder::macAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MacAddress")
                    .unmarshallLocationName("macAddress").build()).build();

    private static final SdkField<String> NETWORK_INTERFACE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NetworkInterfaceId")
            .getter(getter(InstanceNetworkInterface::networkInterfaceId))
            .setter(setter(Builder::networkInterfaceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceId")
                    .unmarshallLocationName("networkInterfaceId").build()).build();

    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(InstanceNetworkInterface::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> PRIVATE_DNS_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PrivateDnsName")
            .getter(getter(InstanceNetworkInterface::privateDnsName))
            .setter(setter(Builder::privateDnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateDnsName")
                    .unmarshallLocationName("privateDnsName").build()).build();

    private static final SdkField<String> PRIVATE_IP_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PrivateIpAddress")
            .getter(getter(InstanceNetworkInterface::privateIpAddress))
            .setter(setter(Builder::privateIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddress")
                    .unmarshallLocationName("privateIpAddress").build()).build();

    private static final SdkField<List<InstancePrivateIpAddress>> PRIVATE_IP_ADDRESSES_FIELD = SdkField
            .<List<InstancePrivateIpAddress>> builder(MarshallingType.LIST)
            .memberName("PrivateIpAddresses")
            .getter(getter(InstanceNetworkInterface::privateIpAddresses))
            .setter(setter(Builder::privateIpAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddressesSet")
                    .unmarshallLocationName("privateIpAddressesSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<InstancePrivateIpAddress> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstancePrivateIpAddress::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> SOURCE_DEST_CHECK_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SourceDestCheck")
            .getter(getter(InstanceNetworkInterface::sourceDestCheck))
            .setter(setter(Builder::sourceDestCheck))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDestCheck")
                    .unmarshallLocationName("sourceDestCheck").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Status")
            .getter(getter(InstanceNetworkInterface::statusAsString))
            .setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("status").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SubnetId")
            .getter(getter(InstanceNetworkInterface::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("subnetId").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(InstanceNetworkInterface::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final SdkField<String> INTERFACE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InterfaceType")
            .getter(getter(InstanceNetworkInterface::interfaceType))
            .setter(setter(Builder::interfaceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InterfaceType")
                    .unmarshallLocationName("interfaceType").build()).build();

    private static final SdkField<List<InstanceIpv4Prefix>> IPV4_PREFIXES_FIELD = SdkField
            .<List<InstanceIpv4Prefix>> builder(MarshallingType.LIST)
            .memberName("Ipv4Prefixes")
            .getter(getter(InstanceNetworkInterface::ipv4Prefixes))
            .setter(setter(Builder::ipv4Prefixes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv4PrefixSet")
                    .unmarshallLocationName("ipv4PrefixSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<InstanceIpv4Prefix> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceIpv4Prefix::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<InstanceIpv6Prefix>> IPV6_PREFIXES_FIELD = SdkField
            .<List<InstanceIpv6Prefix>> builder(MarshallingType.LIST)
            .memberName("Ipv6Prefixes")
            .getter(getter(InstanceNetworkInterface::ipv6Prefixes))
            .setter(setter(Builder::ipv6Prefixes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6PrefixSet")
                    .unmarshallLocationName("ipv6PrefixSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<InstanceIpv6Prefix> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceIpv6Prefix::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<ConnectionTrackingSpecificationResponse> CONNECTION_TRACKING_CONFIGURATION_FIELD = SdkField
            .<ConnectionTrackingSpecificationResponse> builder(MarshallingType.SDK_POJO)
            .memberName("ConnectionTrackingConfiguration")
            .getter(getter(InstanceNetworkInterface::connectionTrackingConfiguration))
            .setter(setter(Builder::connectionTrackingConfiguration))
            .constructor(ConnectionTrackingSpecificationResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionTrackingConfiguration")
                    .unmarshallLocationName("connectionTrackingConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSOCIATION_FIELD,
            ATTACHMENT_FIELD, DESCRIPTION_FIELD, GROUPS_FIELD, IPV6_ADDRESSES_FIELD, MAC_ADDRESS_FIELD,
            NETWORK_INTERFACE_ID_FIELD, OWNER_ID_FIELD, PRIVATE_DNS_NAME_FIELD, PRIVATE_IP_ADDRESS_FIELD,
            PRIVATE_IP_ADDRESSES_FIELD, SOURCE_DEST_CHECK_FIELD, STATUS_FIELD, SUBNET_ID_FIELD, VPC_ID_FIELD,
            INTERFACE_TYPE_FIELD, IPV4_PREFIXES_FIELD, IPV6_PREFIXES_FIELD, CONNECTION_TRACKING_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final InstanceNetworkInterfaceAssociation association;

    private final InstanceNetworkInterfaceAttachment attachment;

    private final String description;

    private final List<GroupIdentifier> groups;

    private final List<InstanceIpv6Address> ipv6Addresses;

    private final String macAddress;

    private final String networkInterfaceId;

    private final String ownerId;

    private final String privateDnsName;

    private final String privateIpAddress;

    private final List<InstancePrivateIpAddress> privateIpAddresses;

    private final Boolean sourceDestCheck;

    private final String status;

    private final String subnetId;

    private final String vpcId;

    private final String interfaceType;

    private final List<InstanceIpv4Prefix> ipv4Prefixes;

    private final List<InstanceIpv6Prefix> ipv6Prefixes;

    private final ConnectionTrackingSpecificationResponse connectionTrackingConfiguration;

    private InstanceNetworkInterface(BuilderImpl builder) {
        this.association = builder.association;
        this.attachment = builder.attachment;
        this.description = builder.description;
        this.groups = builder.groups;
        this.ipv6Addresses = builder.ipv6Addresses;
        this.macAddress = builder.macAddress;
        this.networkInterfaceId = builder.networkInterfaceId;
        this.ownerId = builder.ownerId;
        this.privateDnsName = builder.privateDnsName;
        this.privateIpAddress = builder.privateIpAddress;
        this.privateIpAddresses = builder.privateIpAddresses;
        this.sourceDestCheck = builder.sourceDestCheck;
        this.status = builder.status;
        this.subnetId = builder.subnetId;
        this.vpcId = builder.vpcId;
        this.interfaceType = builder.interfaceType;
        this.ipv4Prefixes = builder.ipv4Prefixes;
        this.ipv6Prefixes = builder.ipv6Prefixes;
        this.connectionTrackingConfiguration = builder.connectionTrackingConfiguration;
    }

    /**
     * <p>
     * The association information for an Elastic IPv4 associated with the network interface.
     * </p>
     * 
     * @return The association information for an Elastic IPv4 associated with the network interface.
     */
    public final InstanceNetworkInterfaceAssociation association() {
        return association;
    }

    /**
     * <p>
     * The network interface attachment.
     * </p>
     * 
     * @return The network interface attachment.
     */
    public final InstanceNetworkInterfaceAttachment attachment() {
        return attachment;
    }

    /**
     * <p>
     * The description.
     * </p>
     * 
     * @return The description.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Groups property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasGroups() {
        return groups != null && !(groups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security groups.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGroups} method.
     * </p>
     * 
     * @return The security groups.
     */
    public final List<GroupIdentifier> groups() {
        return groups;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ipv6Addresses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIpv6Addresses() {
        return ipv6Addresses != null && !(ipv6Addresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IPv6 addresses associated with the network interface.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIpv6Addresses} method.
     * </p>
     * 
     * @return The IPv6 addresses associated with the network interface.
     */
    public final List<InstanceIpv6Address> ipv6Addresses() {
        return ipv6Addresses;
    }

    /**
     * <p>
     * The MAC address.
     * </p>
     * 
     * @return The MAC address.
     */
    public final String macAddress() {
        return macAddress;
    }

    /**
     * <p>
     * The ID of the network interface.
     * </p>
     * 
     * @return The ID of the network interface.
     */
    public final String networkInterfaceId() {
        return networkInterfaceId;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services account that created the network interface.
     * </p>
     * 
     * @return The ID of the Amazon Web Services account that created the network interface.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The private DNS name.
     * </p>
     * 
     * @return The private DNS name.
     */
    public final String privateDnsName() {
        return privateDnsName;
    }

    /**
     * <p>
     * The IPv4 address of the network interface within the subnet.
     * </p>
     * 
     * @return The IPv4 address of the network interface within the subnet.
     */
    public final String privateIpAddress() {
        return privateIpAddress;
    }

    /**
     * For responses, this returns true if the service returned a value for the PrivateIpAddresses property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPrivateIpAddresses() {
        return privateIpAddresses != null && !(privateIpAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The private IPv4 addresses associated with the network interface.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPrivateIpAddresses} method.
     * </p>
     * 
     * @return The private IPv4 addresses associated with the network interface.
     */
    public final List<InstancePrivateIpAddress> privateIpAddresses() {
        return privateIpAddresses;
    }

    /**
     * <p>
     * Indicates whether source/destination checking is enabled.
     * </p>
     * 
     * @return Indicates whether source/destination checking is enabled.
     */
    public final Boolean sourceDestCheck() {
        return sourceDestCheck;
    }

    /**
     * <p>
     * The status of the network interface.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link NetworkInterfaceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the network interface.
     * @see NetworkInterfaceStatus
     */
    public final NetworkInterfaceStatus status() {
        return NetworkInterfaceStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the network interface.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link NetworkInterfaceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the network interface.
     * @see NetworkInterfaceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The ID of the subnet.
     * </p>
     * 
     * @return The ID of the subnet.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The ID of the VPC.
     * </p>
     * 
     * @return The ID of the VPC.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The type of network interface.
     * </p>
     * <p>
     * Valid values: <code>interface</code> | <code>efa</code> | <code>trunk</code>
     * </p>
     * 
     * @return The type of network interface.</p>
     *         <p>
     *         Valid values: <code>interface</code> | <code>efa</code> | <code>trunk</code>
     */
    public final String interfaceType() {
        return interfaceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ipv4Prefixes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIpv4Prefixes() {
        return ipv4Prefixes != null && !(ipv4Prefixes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IPv4 delegated prefixes that are assigned to the network interface.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIpv4Prefixes} method.
     * </p>
     * 
     * @return The IPv4 delegated prefixes that are assigned to the network interface.
     */
    public final List<InstanceIpv4Prefix> ipv4Prefixes() {
        return ipv4Prefixes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ipv6Prefixes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIpv6Prefixes() {
        return ipv6Prefixes != null && !(ipv6Prefixes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IPv6 delegated prefixes that are assigned to the network interface.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIpv6Prefixes} method.
     * </p>
     * 
     * @return The IPv6 delegated prefixes that are assigned to the network interface.
     */
    public final List<InstanceIpv6Prefix> ipv6Prefixes() {
        return ipv6Prefixes;
    }

    /**
     * <p>
     * A security group connection tracking configuration that enables you to set the timeout for connection tracking on
     * an Elastic network interface. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#connection-tracking-timeouts"
     * >Connection tracking timeouts</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * 
     * @return A security group connection tracking configuration that enables you to set the timeout for connection
     *         tracking on an Elastic network interface. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#connection-tracking-timeouts"
     *         >Connection tracking timeouts</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     */
    public final ConnectionTrackingSpecificationResponse connectionTrackingConfiguration() {
        return connectionTrackingConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(association());
        hashCode = 31 * hashCode + Objects.hashCode(attachment());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasGroups() ? groups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasIpv6Addresses() ? ipv6Addresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(macAddress());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceId());
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(privateDnsName());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(hasPrivateIpAddresses() ? privateIpAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sourceDestCheck());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(interfaceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasIpv4Prefixes() ? ipv4Prefixes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasIpv6Prefixes() ? ipv6Prefixes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(connectionTrackingConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceNetworkInterface)) {
            return false;
        }
        InstanceNetworkInterface other = (InstanceNetworkInterface) obj;
        return Objects.equals(association(), other.association()) && Objects.equals(attachment(), other.attachment())
                && Objects.equals(description(), other.description()) && hasGroups() == other.hasGroups()
                && Objects.equals(groups(), other.groups()) && hasIpv6Addresses() == other.hasIpv6Addresses()
                && Objects.equals(ipv6Addresses(), other.ipv6Addresses()) && Objects.equals(macAddress(), other.macAddress())
                && Objects.equals(networkInterfaceId(), other.networkInterfaceId()) && Objects.equals(ownerId(), other.ownerId())
                && Objects.equals(privateDnsName(), other.privateDnsName())
                && Objects.equals(privateIpAddress(), other.privateIpAddress())
                && hasPrivateIpAddresses() == other.hasPrivateIpAddresses()
                && Objects.equals(privateIpAddresses(), other.privateIpAddresses())
                && Objects.equals(sourceDestCheck(), other.sourceDestCheck())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(subnetId(), other.subnetId())
                && Objects.equals(vpcId(), other.vpcId()) && Objects.equals(interfaceType(), other.interfaceType())
                && hasIpv4Prefixes() == other.hasIpv4Prefixes() && Objects.equals(ipv4Prefixes(), other.ipv4Prefixes())
                && hasIpv6Prefixes() == other.hasIpv6Prefixes() && Objects.equals(ipv6Prefixes(), other.ipv6Prefixes())
                && Objects.equals(connectionTrackingConfiguration(), other.connectionTrackingConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceNetworkInterface").add("Association", association()).add("Attachment", attachment())
                .add("Description", description()).add("Groups", hasGroups() ? groups() : null)
                .add("Ipv6Addresses", hasIpv6Addresses() ? ipv6Addresses() : null).add("MacAddress", macAddress())
                .add("NetworkInterfaceId", networkInterfaceId()).add("OwnerId", ownerId())
                .add("PrivateDnsName", privateDnsName()).add("PrivateIpAddress", privateIpAddress())
                .add("PrivateIpAddresses", hasPrivateIpAddresses() ? privateIpAddresses() : null)
                .add("SourceDestCheck", sourceDestCheck()).add("Status", statusAsString()).add("SubnetId", subnetId())
                .add("VpcId", vpcId()).add("InterfaceType", interfaceType())
                .add("Ipv4Prefixes", hasIpv4Prefixes() ? ipv4Prefixes() : null)
                .add("Ipv6Prefixes", hasIpv6Prefixes() ? ipv6Prefixes() : null)
                .add("ConnectionTrackingConfiguration", connectionTrackingConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Association":
            return Optional.ofNullable(clazz.cast(association()));
        case "Attachment":
            return Optional.ofNullable(clazz.cast(attachment()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Groups":
            return Optional.ofNullable(clazz.cast(groups()));
        case "Ipv6Addresses":
            return Optional.ofNullable(clazz.cast(ipv6Addresses()));
        case "MacAddress":
            return Optional.ofNullable(clazz.cast(macAddress()));
        case "NetworkInterfaceId":
            return Optional.ofNullable(clazz.cast(networkInterfaceId()));
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "PrivateDnsName":
            return Optional.ofNullable(clazz.cast(privateDnsName()));
        case "PrivateIpAddress":
            return Optional.ofNullable(clazz.cast(privateIpAddress()));
        case "PrivateIpAddresses":
            return Optional.ofNullable(clazz.cast(privateIpAddresses()));
        case "SourceDestCheck":
            return Optional.ofNullable(clazz.cast(sourceDestCheck()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "InterfaceType":
            return Optional.ofNullable(clazz.cast(interfaceType()));
        case "Ipv4Prefixes":
            return Optional.ofNullable(clazz.cast(ipv4Prefixes()));
        case "Ipv6Prefixes":
            return Optional.ofNullable(clazz.cast(ipv6Prefixes()));
        case "ConnectionTrackingConfiguration":
            return Optional.ofNullable(clazz.cast(connectionTrackingConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceNetworkInterface, T> g) {
        return obj -> g.apply((InstanceNetworkInterface) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceNetworkInterface> {
        /**
         * <p>
         * The association information for an Elastic IPv4 associated with the network interface.
         * </p>
         * 
         * @param association
         *        The association information for an Elastic IPv4 associated with the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder association(InstanceNetworkInterfaceAssociation association);

        /**
         * <p>
         * The association information for an Elastic IPv4 associated with the network interface.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link InstanceNetworkInterfaceAssociation.Builder} avoiding the need to create one manually via
         * {@link InstanceNetworkInterfaceAssociation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceNetworkInterfaceAssociation.Builder#build()} is called
         * immediately and its result is passed to {@link #association(InstanceNetworkInterfaceAssociation)}.
         * 
         * @param association
         *        a consumer that will call methods on {@link InstanceNetworkInterfaceAssociation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #association(InstanceNetworkInterfaceAssociation)
         */
        default Builder association(Consumer<InstanceNetworkInterfaceAssociation.Builder> association) {
            return association(InstanceNetworkInterfaceAssociation.builder().applyMutation(association).build());
        }

        /**
         * <p>
         * The network interface attachment.
         * </p>
         * 
         * @param attachment
         *        The network interface attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachment(InstanceNetworkInterfaceAttachment attachment);

        /**
         * <p>
         * The network interface attachment.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link InstanceNetworkInterfaceAttachment.Builder} avoiding the need to create one manually via
         * {@link InstanceNetworkInterfaceAttachment#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceNetworkInterfaceAttachment.Builder#build()} is called
         * immediately and its result is passed to {@link #attachment(InstanceNetworkInterfaceAttachment)}.
         * 
         * @param attachment
         *        a consumer that will call methods on {@link InstanceNetworkInterfaceAttachment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachment(InstanceNetworkInterfaceAttachment)
         */
        default Builder attachment(Consumer<InstanceNetworkInterfaceAttachment.Builder> attachment) {
            return attachment(InstanceNetworkInterfaceAttachment.builder().applyMutation(attachment).build());
        }

        /**
         * <p>
         * The description.
         * </p>
         * 
         * @param description
         *        The description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The security groups.
         * </p>
         * 
         * @param groups
         *        The security groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<GroupIdentifier> groups);

        /**
         * <p>
         * The security groups.
         * </p>
         * 
         * @param groups
         *        The security groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(GroupIdentifier... groups);

        /**
         * <p>
         * The security groups.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.GroupIdentifier.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.GroupIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.GroupIdentifier.Builder#build()} is called immediately and
         * its result is passed to {@link #groups(List<GroupIdentifier>)}.
         * 
         * @param groups
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.GroupIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groups(java.util.Collection<GroupIdentifier>)
         */
        Builder groups(Consumer<GroupIdentifier.Builder>... groups);

        /**
         * <p>
         * The IPv6 addresses associated with the network interface.
         * </p>
         * 
         * @param ipv6Addresses
         *        The IPv6 addresses associated with the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(Collection<InstanceIpv6Address> ipv6Addresses);

        /**
         * <p>
         * The IPv6 addresses associated with the network interface.
         * </p>
         * 
         * @param ipv6Addresses
         *        The IPv6 addresses associated with the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(InstanceIpv6Address... ipv6Addresses);

        /**
         * <p>
         * The IPv6 addresses associated with the network interface.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.InstanceIpv6Address.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.InstanceIpv6Address#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.InstanceIpv6Address.Builder#build()} is called immediately
         * and its result is passed to {@link #ipv6Addresses(List<InstanceIpv6Address>)}.
         * 
         * @param ipv6Addresses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.InstanceIpv6Address.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipv6Addresses(java.util.Collection<InstanceIpv6Address>)
         */
        Builder ipv6Addresses(Consumer<InstanceIpv6Address.Builder>... ipv6Addresses);

        /**
         * <p>
         * The MAC address.
         * </p>
         * 
         * @param macAddress
         *        The MAC address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder macAddress(String macAddress);

        /**
         * <p>
         * The ID of the network interface.
         * </p>
         * 
         * @param networkInterfaceId
         *        The ID of the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceId(String networkInterfaceId);

        /**
         * <p>
         * The ID of the Amazon Web Services account that created the network interface.
         * </p>
         * 
         * @param ownerId
         *        The ID of the Amazon Web Services account that created the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The private DNS name.
         * </p>
         * 
         * @param privateDnsName
         *        The private DNS name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateDnsName(String privateDnsName);

        /**
         * <p>
         * The IPv4 address of the network interface within the subnet.
         * </p>
         * 
         * @param privateIpAddress
         *        The IPv4 address of the network interface within the subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddress(String privateIpAddress);

        /**
         * <p>
         * The private IPv4 addresses associated with the network interface.
         * </p>
         * 
         * @param privateIpAddresses
         *        The private IPv4 addresses associated with the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(Collection<InstancePrivateIpAddress> privateIpAddresses);

        /**
         * <p>
         * The private IPv4 addresses associated with the network interface.
         * </p>
         * 
         * @param privateIpAddresses
         *        The private IPv4 addresses associated with the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(InstancePrivateIpAddress... privateIpAddresses);

        /**
         * <p>
         * The private IPv4 addresses associated with the network interface.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.InstancePrivateIpAddress.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.ec2.model.InstancePrivateIpAddress#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.InstancePrivateIpAddress.Builder#build()} is called
         * immediately and its result is passed to {@link #privateIpAddresses(List<InstancePrivateIpAddress>)}.
         * 
         * @param privateIpAddresses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.InstancePrivateIpAddress.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #privateIpAddresses(java.util.Collection<InstancePrivateIpAddress>)
         */
        Builder privateIpAddresses(Consumer<InstancePrivateIpAddress.Builder>... privateIpAddresses);

        /**
         * <p>
         * Indicates whether source/destination checking is enabled.
         * </p>
         * 
         * @param sourceDestCheck
         *        Indicates whether source/destination checking is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDestCheck(Boolean sourceDestCheck);

        /**
         * <p>
         * The status of the network interface.
         * </p>
         * 
         * @param status
         *        The status of the network interface.
         * @see NetworkInterfaceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkInterfaceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the network interface.
         * </p>
         * 
         * @param status
         *        The status of the network interface.
         * @see NetworkInterfaceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkInterfaceStatus
         */
        Builder status(NetworkInterfaceStatus status);

        /**
         * <p>
         * The ID of the subnet.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The ID of the VPC.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The type of network interface.
         * </p>
         * <p>
         * Valid values: <code>interface</code> | <code>efa</code> | <code>trunk</code>
         * </p>
         * 
         * @param interfaceType
         *        The type of network interface.</p>
         *        <p>
         *        Valid values: <code>interface</code> | <code>efa</code> | <code>trunk</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interfaceType(String interfaceType);

        /**
         * <p>
         * The IPv4 delegated prefixes that are assigned to the network interface.
         * </p>
         * 
         * @param ipv4Prefixes
         *        The IPv4 delegated prefixes that are assigned to the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv4Prefixes(Collection<InstanceIpv4Prefix> ipv4Prefixes);

        /**
         * <p>
         * The IPv4 delegated prefixes that are assigned to the network interface.
         * </p>
         * 
         * @param ipv4Prefixes
         *        The IPv4 delegated prefixes that are assigned to the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv4Prefixes(InstanceIpv4Prefix... ipv4Prefixes);

        /**
         * <p>
         * The IPv4 delegated prefixes that are assigned to the network interface.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.InstanceIpv4Prefix.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.InstanceIpv4Prefix#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.InstanceIpv4Prefix.Builder#build()} is called immediately
         * and its result is passed to {@link #ipv4Prefixes(List<InstanceIpv4Prefix>)}.
         * 
         * @param ipv4Prefixes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.InstanceIpv4Prefix.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipv4Prefixes(java.util.Collection<InstanceIpv4Prefix>)
         */
        Builder ipv4Prefixes(Consumer<InstanceIpv4Prefix.Builder>... ipv4Prefixes);

        /**
         * <p>
         * The IPv6 delegated prefixes that are assigned to the network interface.
         * </p>
         * 
         * @param ipv6Prefixes
         *        The IPv6 delegated prefixes that are assigned to the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Prefixes(Collection<InstanceIpv6Prefix> ipv6Prefixes);

        /**
         * <p>
         * The IPv6 delegated prefixes that are assigned to the network interface.
         * </p>
         * 
         * @param ipv6Prefixes
         *        The IPv6 delegated prefixes that are assigned to the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Prefixes(InstanceIpv6Prefix... ipv6Prefixes);

        /**
         * <p>
         * The IPv6 delegated prefixes that are assigned to the network interface.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.InstanceIpv6Prefix.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.InstanceIpv6Prefix#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.InstanceIpv6Prefix.Builder#build()} is called immediately
         * and its result is passed to {@link #ipv6Prefixes(List<InstanceIpv6Prefix>)}.
         * 
         * @param ipv6Prefixes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.InstanceIpv6Prefix.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipv6Prefixes(java.util.Collection<InstanceIpv6Prefix>)
         */
        Builder ipv6Prefixes(Consumer<InstanceIpv6Prefix.Builder>... ipv6Prefixes);

        /**
         * <p>
         * A security group connection tracking configuration that enables you to set the timeout for connection
         * tracking on an Elastic network interface. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#connection-tracking-timeouts"
         * >Connection tracking timeouts</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * 
         * @param connectionTrackingConfiguration
         *        A security group connection tracking configuration that enables you to set the timeout for connection
         *        tracking on an Elastic network interface. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#connection-tracking-timeouts"
         *        >Connection tracking timeouts</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionTrackingConfiguration(ConnectionTrackingSpecificationResponse connectionTrackingConfiguration);

        /**
         * <p>
         * A security group connection tracking configuration that enables you to set the timeout for connection
         * tracking on an Elastic network interface. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#connection-tracking-timeouts"
         * >Connection tracking timeouts</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ConnectionTrackingSpecificationResponse.Builder} avoiding the need to create one manually via
         * {@link ConnectionTrackingSpecificationResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ConnectionTrackingSpecificationResponse.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #connectionTrackingConfiguration(ConnectionTrackingSpecificationResponse)}.
         * 
         * @param connectionTrackingConfiguration
         *        a consumer that will call methods on {@link ConnectionTrackingSpecificationResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #connectionTrackingConfiguration(ConnectionTrackingSpecificationResponse)
         */
        default Builder connectionTrackingConfiguration(
                Consumer<ConnectionTrackingSpecificationResponse.Builder> connectionTrackingConfiguration) {
            return connectionTrackingConfiguration(ConnectionTrackingSpecificationResponse.builder()
                    .applyMutation(connectionTrackingConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private InstanceNetworkInterfaceAssociation association;

        private InstanceNetworkInterfaceAttachment attachment;

        private String description;

        private List<GroupIdentifier> groups = DefaultSdkAutoConstructList.getInstance();

        private List<InstanceIpv6Address> ipv6Addresses = DefaultSdkAutoConstructList.getInstance();

        private String macAddress;

        private String networkInterfaceId;

        private String ownerId;

        private String privateDnsName;

        private String privateIpAddress;

        private List<InstancePrivateIpAddress> privateIpAddresses = DefaultSdkAutoConstructList.getInstance();

        private Boolean sourceDestCheck;

        private String status;

        private String subnetId;

        private String vpcId;

        private String interfaceType;

        private List<InstanceIpv4Prefix> ipv4Prefixes = DefaultSdkAutoConstructList.getInstance();

        private List<InstanceIpv6Prefix> ipv6Prefixes = DefaultSdkAutoConstructList.getInstance();

        private ConnectionTrackingSpecificationResponse connectionTrackingConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceNetworkInterface model) {
            association(model.association);
            attachment(model.attachment);
            description(model.description);
            groups(model.groups);
            ipv6Addresses(model.ipv6Addresses);
            macAddress(model.macAddress);
            networkInterfaceId(model.networkInterfaceId);
            ownerId(model.ownerId);
            privateDnsName(model.privateDnsName);
            privateIpAddress(model.privateIpAddress);
            privateIpAddresses(model.privateIpAddresses);
            sourceDestCheck(model.sourceDestCheck);
            status(model.status);
            subnetId(model.subnetId);
            vpcId(model.vpcId);
            interfaceType(model.interfaceType);
            ipv4Prefixes(model.ipv4Prefixes);
            ipv6Prefixes(model.ipv6Prefixes);
            connectionTrackingConfiguration(model.connectionTrackingConfiguration);
        }

        public final InstanceNetworkInterfaceAssociation.Builder getAssociation() {
            return association != null ? association.toBuilder() : null;
        }

        public final void setAssociation(InstanceNetworkInterfaceAssociation.BuilderImpl association) {
            this.association = association != null ? association.build() : null;
        }

        @Override
        public final Builder association(InstanceNetworkInterfaceAssociation association) {
            this.association = association;
            return this;
        }

        public final InstanceNetworkInterfaceAttachment.Builder getAttachment() {
            return attachment != null ? attachment.toBuilder() : null;
        }

        public final void setAttachment(InstanceNetworkInterfaceAttachment.BuilderImpl attachment) {
            this.attachment = attachment != null ? attachment.build() : null;
        }

        @Override
        public final Builder attachment(InstanceNetworkInterfaceAttachment attachment) {
            this.attachment = attachment;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<GroupIdentifier.Builder> getGroups() {
            List<GroupIdentifier.Builder> result = GroupIdentifierListCopier.copyToBuilder(this.groups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setGroups(Collection<GroupIdentifier.BuilderImpl> groups) {
            this.groups = GroupIdentifierListCopier.copyFromBuilder(groups);
        }

        @Override
        public final Builder groups(Collection<GroupIdentifier> groups) {
            this.groups = GroupIdentifierListCopier.copy(groups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(GroupIdentifier... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(Consumer<GroupIdentifier.Builder>... groups) {
            groups(Stream.of(groups).map(c -> GroupIdentifier.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<InstanceIpv6Address.Builder> getIpv6Addresses() {
            List<InstanceIpv6Address.Builder> result = InstanceIpv6AddressListCopier.copyToBuilder(this.ipv6Addresses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIpv6Addresses(Collection<InstanceIpv6Address.BuilderImpl> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListCopier.copyFromBuilder(ipv6Addresses);
        }

        @Override
        public final Builder ipv6Addresses(Collection<InstanceIpv6Address> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListCopier.copy(ipv6Addresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Addresses(InstanceIpv6Address... ipv6Addresses) {
            ipv6Addresses(Arrays.asList(ipv6Addresses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Addresses(Consumer<InstanceIpv6Address.Builder>... ipv6Addresses) {
            ipv6Addresses(Stream.of(ipv6Addresses).map(c -> InstanceIpv6Address.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getMacAddress() {
            return macAddress;
        }

        public final void setMacAddress(String macAddress) {
            this.macAddress = macAddress;
        }

        @Override
        public final Builder macAddress(String macAddress) {
            this.macAddress = macAddress;
            return this;
        }

        public final String getNetworkInterfaceId() {
            return networkInterfaceId;
        }

        public final void setNetworkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
        }

        @Override
        public final Builder networkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
            return this;
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getPrivateDnsName() {
            return privateDnsName;
        }

        public final void setPrivateDnsName(String privateDnsName) {
            this.privateDnsName = privateDnsName;
        }

        @Override
        public final Builder privateDnsName(String privateDnsName) {
            this.privateDnsName = privateDnsName;
            return this;
        }

        public final String getPrivateIpAddress() {
            return privateIpAddress;
        }

        public final void setPrivateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
        }

        @Override
        public final Builder privateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
            return this;
        }

        public final List<InstancePrivateIpAddress.Builder> getPrivateIpAddresses() {
            List<InstancePrivateIpAddress.Builder> result = InstancePrivateIpAddressListCopier
                    .copyToBuilder(this.privateIpAddresses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPrivateIpAddresses(Collection<InstancePrivateIpAddress.BuilderImpl> privateIpAddresses) {
            this.privateIpAddresses = InstancePrivateIpAddressListCopier.copyFromBuilder(privateIpAddresses);
        }

        @Override
        public final Builder privateIpAddresses(Collection<InstancePrivateIpAddress> privateIpAddresses) {
            this.privateIpAddresses = InstancePrivateIpAddressListCopier.copy(privateIpAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder privateIpAddresses(InstancePrivateIpAddress... privateIpAddresses) {
            privateIpAddresses(Arrays.asList(privateIpAddresses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder privateIpAddresses(Consumer<InstancePrivateIpAddress.Builder>... privateIpAddresses) {
            privateIpAddresses(Stream.of(privateIpAddresses)
                    .map(c -> InstancePrivateIpAddress.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getSourceDestCheck() {
            return sourceDestCheck;
        }

        public final void setSourceDestCheck(Boolean sourceDestCheck) {
            this.sourceDestCheck = sourceDestCheck;
        }

        @Override
        public final Builder sourceDestCheck(Boolean sourceDestCheck) {
            this.sourceDestCheck = sourceDestCheck;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(NetworkInterfaceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getInterfaceType() {
            return interfaceType;
        }

        public final void setInterfaceType(String interfaceType) {
            this.interfaceType = interfaceType;
        }

        @Override
        public final Builder interfaceType(String interfaceType) {
            this.interfaceType = interfaceType;
            return this;
        }

        public final List<InstanceIpv4Prefix.Builder> getIpv4Prefixes() {
            List<InstanceIpv4Prefix.Builder> result = InstanceIpv4PrefixListCopier.copyToBuilder(this.ipv4Prefixes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIpv4Prefixes(Collection<InstanceIpv4Prefix.BuilderImpl> ipv4Prefixes) {
            this.ipv4Prefixes = InstanceIpv4PrefixListCopier.copyFromBuilder(ipv4Prefixes);
        }

        @Override
        public final Builder ipv4Prefixes(Collection<InstanceIpv4Prefix> ipv4Prefixes) {
            this.ipv4Prefixes = InstanceIpv4PrefixListCopier.copy(ipv4Prefixes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv4Prefixes(InstanceIpv4Prefix... ipv4Prefixes) {
            ipv4Prefixes(Arrays.asList(ipv4Prefixes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv4Prefixes(Consumer<InstanceIpv4Prefix.Builder>... ipv4Prefixes) {
            ipv4Prefixes(Stream.of(ipv4Prefixes).map(c -> InstanceIpv4Prefix.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<InstanceIpv6Prefix.Builder> getIpv6Prefixes() {
            List<InstanceIpv6Prefix.Builder> result = InstanceIpv6PrefixListCopier.copyToBuilder(this.ipv6Prefixes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIpv6Prefixes(Collection<InstanceIpv6Prefix.BuilderImpl> ipv6Prefixes) {
            this.ipv6Prefixes = InstanceIpv6PrefixListCopier.copyFromBuilder(ipv6Prefixes);
        }

        @Override
        public final Builder ipv6Prefixes(Collection<InstanceIpv6Prefix> ipv6Prefixes) {
            this.ipv6Prefixes = InstanceIpv6PrefixListCopier.copy(ipv6Prefixes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Prefixes(InstanceIpv6Prefix... ipv6Prefixes) {
            ipv6Prefixes(Arrays.asList(ipv6Prefixes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Prefixes(Consumer<InstanceIpv6Prefix.Builder>... ipv6Prefixes) {
            ipv6Prefixes(Stream.of(ipv6Prefixes).map(c -> InstanceIpv6Prefix.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final ConnectionTrackingSpecificationResponse.Builder getConnectionTrackingConfiguration() {
            return connectionTrackingConfiguration != null ? connectionTrackingConfiguration.toBuilder() : null;
        }

        public final void setConnectionTrackingConfiguration(
                ConnectionTrackingSpecificationResponse.BuilderImpl connectionTrackingConfiguration) {
            this.connectionTrackingConfiguration = connectionTrackingConfiguration != null ? connectionTrackingConfiguration
                    .build() : null;
        }

        @Override
        public final Builder connectionTrackingConfiguration(
                ConnectionTrackingSpecificationResponse connectionTrackingConfiguration) {
            this.connectionTrackingConfiguration = connectionTrackingConfiguration;
            return this;
        }

        @Override
        public InstanceNetworkInterface build() {
            return new InstanceNetworkInterface(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
