/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelsv2.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.lexmodelsv2.LexModelsV2AsyncClient;
import software.amazon.awssdk.services.lexmodelsv2.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotAliasRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotAliasResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotLocaleRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotLocaleResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotVersionRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotVersionResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeExportRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeExportResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeImportRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeImportResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.LexModelsV2Request;
import software.amazon.awssdk.services.lexmodelsv2.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultLexModelsV2AsyncWaiter implements LexModelsV2AsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final LexModelsV2AsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeBotResponse> botAvailableWaiter;

    private final AsyncWaiter<DescribeBotAliasResponse> botAliasAvailableWaiter;

    private final AsyncWaiter<DescribeExportResponse> botExportCompletedWaiter;

    private final AsyncWaiter<DescribeImportResponse> botImportCompletedWaiter;

    private final AsyncWaiter<DescribeBotLocaleResponse> botLocaleBuiltWaiter;

    private final AsyncWaiter<DescribeBotLocaleResponse> botLocaleExpressTestingAvailableWaiter;

    private final AsyncWaiter<DescribeBotVersionResponse> botVersionAvailableWaiter;

    private final AsyncWaiter<DescribeBotLocaleResponse> botLocaleCreatedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultLexModelsV2AsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = LexModelsV2AsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.botAvailableWaiter = AsyncWaiter.builder(DescribeBotResponse.class).acceptors(botAvailableWaiterAcceptors())
                .overrideConfiguration(botAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.botAliasAvailableWaiter = AsyncWaiter.builder(DescribeBotAliasResponse.class)
                .acceptors(botAliasAvailableWaiterAcceptors())
                .overrideConfiguration(botAliasAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.botExportCompletedWaiter = AsyncWaiter.builder(DescribeExportResponse.class)
                .acceptors(botExportCompletedWaiterAcceptors())
                .overrideConfiguration(botExportCompletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.botImportCompletedWaiter = AsyncWaiter.builder(DescribeImportResponse.class)
                .acceptors(botImportCompletedWaiterAcceptors())
                .overrideConfiguration(botImportCompletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.botLocaleBuiltWaiter = AsyncWaiter.builder(DescribeBotLocaleResponse.class)
                .acceptors(botLocaleBuiltWaiterAcceptors())
                .overrideConfiguration(botLocaleBuiltWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.botLocaleExpressTestingAvailableWaiter = AsyncWaiter.builder(DescribeBotLocaleResponse.class)
                .acceptors(botLocaleExpressTestingAvailableWaiterAcceptors())
                .overrideConfiguration(botLocaleExpressTestingAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.botVersionAvailableWaiter = AsyncWaiter.builder(DescribeBotVersionResponse.class)
                .acceptors(botVersionAvailableWaiterAcceptors())
                .overrideConfiguration(botVersionAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.botLocaleCreatedWaiter = AsyncWaiter.builder(DescribeBotLocaleResponse.class)
                .acceptors(botLocaleCreatedWaiterAcceptors())
                .overrideConfiguration(botLocaleCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotAliasResponse>> waitUntilBotAliasAvailable(
            DescribeBotAliasRequest describeBotAliasRequest) {
        return botAliasAvailableWaiter.runAsync(() -> client.describeBotAlias(applyWaitersUserAgent(describeBotAliasRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotAliasResponse>> waitUntilBotAliasAvailable(
            DescribeBotAliasRequest describeBotAliasRequest, WaiterOverrideConfiguration overrideConfig) {
        return botAliasAvailableWaiter.runAsync(() -> client.describeBotAlias(applyWaitersUserAgent(describeBotAliasRequest)),
                botAliasAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotResponse>> waitUntilBotAvailable(DescribeBotRequest describeBotRequest) {
        return botAvailableWaiter.runAsync(() -> client.describeBot(applyWaitersUserAgent(describeBotRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotResponse>> waitUntilBotAvailable(DescribeBotRequest describeBotRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return botAvailableWaiter.runAsync(() -> client.describeBot(applyWaitersUserAgent(describeBotRequest)),
                botAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeExportResponse>> waitUntilBotExportCompleted(
            DescribeExportRequest describeExportRequest) {
        return botExportCompletedWaiter.runAsync(() -> client.describeExport(applyWaitersUserAgent(describeExportRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeExportResponse>> waitUntilBotExportCompleted(
            DescribeExportRequest describeExportRequest, WaiterOverrideConfiguration overrideConfig) {
        return botExportCompletedWaiter.runAsync(() -> client.describeExport(applyWaitersUserAgent(describeExportRequest)),
                botExportCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImportResponse>> waitUntilBotImportCompleted(
            DescribeImportRequest describeImportRequest) {
        return botImportCompletedWaiter.runAsync(() -> client.describeImport(applyWaitersUserAgent(describeImportRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImportResponse>> waitUntilBotImportCompleted(
            DescribeImportRequest describeImportRequest, WaiterOverrideConfiguration overrideConfig) {
        return botImportCompletedWaiter.runAsync(() -> client.describeImport(applyWaitersUserAgent(describeImportRequest)),
                botImportCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotLocaleResponse>> waitUntilBotLocaleBuilt(
            DescribeBotLocaleRequest describeBotLocaleRequest) {
        return botLocaleBuiltWaiter.runAsync(() -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotLocaleResponse>> waitUntilBotLocaleBuilt(
            DescribeBotLocaleRequest describeBotLocaleRequest, WaiterOverrideConfiguration overrideConfig) {
        return botLocaleBuiltWaiter.runAsync(() -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)),
                botLocaleBuiltWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotLocaleResponse>> waitUntilBotLocaleCreated(
            DescribeBotLocaleRequest describeBotLocaleRequest) {
        return botLocaleCreatedWaiter.runAsync(() -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotLocaleResponse>> waitUntilBotLocaleCreated(
            DescribeBotLocaleRequest describeBotLocaleRequest, WaiterOverrideConfiguration overrideConfig) {
        return botLocaleCreatedWaiter.runAsync(() -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)),
                botLocaleCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotLocaleResponse>> waitUntilBotLocaleExpressTestingAvailable(
            DescribeBotLocaleRequest describeBotLocaleRequest) {
        return botLocaleExpressTestingAvailableWaiter.runAsync(() -> client
                .describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotLocaleResponse>> waitUntilBotLocaleExpressTestingAvailable(
            DescribeBotLocaleRequest describeBotLocaleRequest, WaiterOverrideConfiguration overrideConfig) {
        return botLocaleExpressTestingAvailableWaiter.runAsync(
                () -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)),
                botLocaleExpressTestingAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotVersionResponse>> waitUntilBotVersionAvailable(
            DescribeBotVersionRequest describeBotVersionRequest) {
        return botVersionAvailableWaiter.runAsync(() -> client
                .describeBotVersion(applyWaitersUserAgent(describeBotVersionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBotVersionResponse>> waitUntilBotVersionAvailable(
            DescribeBotVersionRequest describeBotVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return botVersionAvailableWaiter.runAsync(
                () -> client.describeBotVersion(applyWaitersUserAgent(describeBotVersionRequest)),
                botVersionAvailableWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeBotResponse>> botAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("botStatus").value(), "Available");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botStatus").value(), "Deleting");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botStatus=Deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botStatus").value(), "Inactive");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botStatus=Inactive) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotAliasResponse>> botAliasAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotAliasResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("botAliasStatus").value(), "Available");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botAliasStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botAliasStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botAliasStatus").value(), "Deleting");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botAliasStatus=Deleting) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeExportResponse>> botExportCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeExportResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("exportStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("exportStatus").value(), "Deleting");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (exportStatus=Deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("exportStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (exportStatus=Failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImportResponse>> botImportCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImportResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("importStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("importStatus").value(), "Deleting");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (importStatus=Deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("importStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (importStatus=Failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotLocaleResponse>> botLocaleBuiltWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotLocaleResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Built");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botLocaleStatus").value(), "Deleting");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botLocaleStatus=Deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botLocaleStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botLocaleStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botLocaleStatus").value(), "NotBuilt");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botLocaleStatus=NotBuilt) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotLocaleResponse>> botLocaleExpressTestingAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotLocaleResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Built");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "ReadyExpressTesting");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botLocaleStatus").value(), "Deleting");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botLocaleStatus=Deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botLocaleStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botLocaleStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botLocaleStatus").value(), "NotBuilt");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botLocaleStatus=NotBuilt) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotVersionResponse>> botVersionAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("botStatus").value(), "Available");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botStatus").value(), "Deleting");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botStatus=Deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botStatus=Failed) and transitioned the waiter to failure state"));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(404, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotLocaleResponse>> botLocaleCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotLocaleResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Built");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "ReadyExpressTesting");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "NotBuilt");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botLocaleStatus").value(), "Deleting");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botLocaleStatus=Deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("botLocaleStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (botLocaleStatus=Failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration botAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botAliasAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botExportCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botImportCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botLocaleBuiltWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botLocaleExpressTestingAvailableWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botVersionAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botLocaleCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static LexModelsV2AsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends LexModelsV2Request> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements LexModelsV2AsyncWaiter.Builder {
        private LexModelsV2AsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public LexModelsV2AsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public LexModelsV2AsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public LexModelsV2AsyncWaiter.Builder client(LexModelsV2AsyncClient client) {
            this.client = client;
            return this;
        }

        public LexModelsV2AsyncWaiter build() {
            return new DefaultLexModelsV2AsyncWaiter(this);
        }
    }
}
