/*
Copyright 2015 Alexander Bunkenburg <alex@inspiracio.cat>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package cat.inspiracio.html;

import java.io.Serializable;

import org.w3c.dom.DOMException;

/** A factory for html elements.
 * 
 * Public visibility so that we can extend. */
public class Decorator implements Serializable{
	private static final long serialVersionUID = -5325529085204909410L;

	/** Protected visibility so that we can extend the class. */
	protected Decorator(){}
	
	/** Creates an html5 element for this tag.
	 * 
	 * This method has protected visibility so that we can override.
	 * 
	 * @param d The html document that will contain the element.
	 * @param tag The element tag, like "div". 
	 * @return the new html5 element */
	protected HTMLElement decorate(HTMLDocumentImp d, String tag) throws DOMException {
		tag=tag.toLowerCase();

		//Find the element implementation class for this tag.
		switch(tag){
		
		//Root element and document metadata
		case"html":return new HTMLHtmlElementImp(d);
		case"head":return new HTMLHeadElementImp(d);
		case"title":return new HTMLTitleElementImp(d);
		case"base":return new HTMLBaseElementImp(d);
		case"link":return new HTMLLinkElementImp(d);
		case"meta":return new HTMLMetaElementImp(d);
		case"style":return new HTMLStyleElementImp(d);
		
		//Sections
		case"body":return new HTMLBodyElementImp(d);
		case"article":return new HTMLElementImp(d, tag);
		case"section":return new HTMLElementImp(d, tag);
		case"nav":return new HTMLElementImp(d, tag);
		case"aside":return new HTMLElementImp(d, tag);
		case"h1":return new HTMLElementImp(d, tag);
		case"h2":return new HTMLElementImp(d, tag);
		case"h3":return new HTMLElementImp(d, tag);
		case"h4":return new HTMLElementImp(d, tag);
		case"h5":return new HTMLElementImp(d, tag);
		case"h6":return new HTMLElementImp(d, tag);
		case"header":return new HTMLElementImp(d, tag);
		case"footer":return new HTMLElementImp(d, tag);
		case"address":return new HTMLElementImp(d, tag);
		
		//Grouping content
		case"p":return new HTMLParagraphElementImp(d);
		case"hr":return new HTMLHRElementImp(d);
		case"pre":return new HTMLPreElementImp(d);
		case"blockquote":return new HTMLQuoteElementImp(d, tag);
		case"ol":return new HTMLOListElementImp(d);
		case"ul":return new HTMLUListElementImp(d);
		case"li":return new HTMLLIElementImp(d);
		case"dl":return new HTMLDListElementImp(d);
		case"dt":return new HTMLElementImp(d, tag);
		case"dd":return new HTMLElementImp(d, tag);
		case"figure":return new HTMLElementImp(d, tag);
		case"figcaption":return new HTMLElementImp(d, tag);
		case"div":return new HTMLDivElementImp(d);
		case"main":return new HTMLElementImp(d, tag);
		
		//Text-level semantics
		case"a":return new HTMLAnchorElementImp(d);
		case"em":return new HTMLElementImp(d, tag);
		case"strong":return new HTMLElementImp(d, tag);
		case"small":return new HTMLElementImp(d, tag);
		case"s":return new HTMLElementImp(d, tag);
		case"cite":return new HTMLElementImp(d, tag);
		case"q":return new HTMLQuoteElementImp(d, tag);
		case"dfn":return new HTMLElementImp(d, tag);
		case"abbr":return new HTMLElementImp(d, tag);
		case"data":return new HTMLDataElementImp(d);
		case"time":return new HTMLTimeElementImp(d);
		case"code":return new HTMLElementImp(d, tag);
		case"var":return new HTMLElementImp(d, tag);
		case"samp":return new HTMLElementImp(d, tag);
		case"kbd":return new HTMLElementImp(d, tag);
		case"sub":return new HTMLElementImp(d, tag);
		case"sup":return new HTMLElementImp(d, tag);
		case"i":return new HTMLElementImp(d, tag);
		case"b":return new HTMLElementImp(d, tag);
		case"u":return new HTMLElementImp(d, tag);
		case"mark":return new HTMLElementImp(d, tag);
		case"ruby":return new HTMLElementImp(d, tag);
		case"rb":return new HTMLElementImp(d, tag);
		case"rt":return new HTMLElementImp(d, tag);
		case"rtc":return new HTMLElementImp(d, tag);
		case"rp":return new HTMLElementImp(d, tag);
		case"bdi":return new HTMLElementImp(d, tag);
		case"bdo":return new HTMLElementImp(d, tag);
		case"span":return new HTMLSpanElementImp(d);
		case"br":return new HTMLBRElementImp(d);
		case"wbr":return new HTMLElementImp(d, tag);
		
		//Edits
		case"ins":return new HTMLModElementImp(d, tag);
		case"del":return new HTMLModElementImp(d, tag);
		
		//Embedded content 
		case"img":return new HTMLImageElementImp(d);
		case"picture":return new HTMLPictureElementImp(d);
		case"iframe":return new HTMLIFrameElementImp(d);
		case"embed":return new HTMLEmbedElementImp(d);
		case"object":return new HTMLObjectElementImp(d);
		case"param":return new HTMLParamElementImp(d);
		case"video":return new HTMLVideoElementImp(d);
		case"audio":return new HTMLAudioElementImp(d);
		case"source":return new HTMLSourceElementImp(d);
		case"track":return new HTMLTrackElementImp(d);
		case"map":return new HTMLMapElementImp(d);
		case"area":return new HTMLAreaElementImp(d);
		
		//Tabular data
		case"table":return new HTMLTableElementImp(d);
		case"caption":return new HTMLTableCaptionElementImp(d);
		case"colgroup":return new HTMLTableColElementImp(d, tag);
		case"col":return new HTMLTableColElementImp(d, tag);
		case"tbody":return new HTMLTableSectionElementImp(d, tag);
		case"thead":return new HTMLTableSectionElementImp(d, tag);
		case"tfoot":return new HTMLTableSectionElementImp(d, tag);
		case"tr":return new HTMLTableRowElementImp(d);
		case"td":return new HTMLTableDataCellElementImp(d);
		case"th":return new HTMLTableHeaderCellElementImp(d);
		
		//Forms
		case"form":return new HTMLFormElementImp(d);
		case"label":return new HTMLLabelElementImp(d);
		case"input":return new HTMLInputElementImp(d);
		case"button":return new HTMLButtonElementImp(d);
		case"select":return new HTMLSelectElementImp(d);
		case"datalist":return new HTMLDataListElementImp(d);
		case"optgroup":return new HTMLOptGroupElementImp(d);
		case"option":return new HTMLOptionElementImp(d);
		case"textarea":return new HTMLTextAreaElementImp(d);
		case"keygen":return new HTMLKeygenElementImp(d);
		case"output":return new HTMLOutputElementImp(d);
		case"progress":return new HTMLProgressElementImp(d);
		case"meter":return new HTMLMeterElementImp(d);
		case"fieldset":return new HTMLFieldSetElementImp(d);
		case"legend":return new HTMLLegendElementImp(d);
		
		//Script
		case"script":return new HTMLScriptElementImp(d);
		case"noscript":return new HTMLElementImp(d, tag);
		case"template":return new HTMLTemplateElementImp(d);
		case"canvas":return new HTMLCanvasElementImp(d);
		
		//obsolete: I don't implement them.
		//applet
		//marquee
		}
		
		//Unknown tags.
		return new HTMLUnknownElementImp(d, tag);
	}

}
