/*
 * Decompiled with CFR 0.152.
 */
package com.adobe.granite.crx2oak.util.filesystem;

import com.adobe.granite.crx2oak.model.Topics;
import com.adobe.granite.crx2oak.pipeline.InputAggregatingComponent;
import com.adobe.granite.crx2oak.pipeline.PipeData;
import com.adobe.granite.crx2oak.pipeline.Pipeline;
import com.adobe.granite.crx2oak.util.FilesystemUtils;
import com.adobe.granite.crx2oak.util.UncheckedIOException;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.util.Formatter;
import java.util.Locale;
import javax.annotation.Nonnull;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DirectoryElementsSwitcher
extends InputAggregatingComponent {
    private static final String UNIX_FORMAT = "    mv %s %s\n";
    private static final String WIN_FORMAT = "    move %s %s\n";
    private static final int MAX_PATH_LENGTH = 50;
    private static final String FORMAT = "    %50s | %10s | %50s \n";
    private static final String MOVE_OPERATION = "-- MOVE ->";
    private static final String ERROR_MSG_THREE_PLACEHOLDERS = "The I/O error occurred when switching directories.\n  To complete migration you need to manually move the following files in the following order:\n%s\n  Under UNIX you can try with the following commands:\n%s\n  Under Windows you can try with the following commands:\n%s";
    private final Logger LOGGER = LoggerFactory.getLogger(DirectoryElementsSwitcher.class);
    private final File blessedDirectory;
    private final File candidateDirectory;
    private final File backupDirectory;

    public DirectoryElementsSwitcher(@Nonnull File blessedDirectory, @Nonnull File candidateDirectory, @Nonnull String partialBackupSuffix) throws IOException {
        Preconditions.checkNotNull((Object)blessedDirectory);
        Preconditions.checkNotNull((Object)candidateDirectory);
        Preconditions.checkNotNull((Object)partialBackupSuffix);
        this.blessedDirectory = blessedDirectory.getCanonicalFile();
        this.candidateDirectory = candidateDirectory.getCanonicalFile();
        String fullSuffix = "-" + partialBackupSuffix + "backup-" + FilesystemUtils.getCurrentTimestampFilenameSuffix();
        this.backupDirectory = new File(this.blessedDirectory.getAbsolutePath() + fullSuffix);
    }

    @Override
    public PipeData preprocess(PipeData input) {
        File[] files = this.candidateDirectory.listFiles();
        return files != null ? this.switchElementsAndReturnOptionalWarnings(files, input) : this.abortSwitchingWithError();
    }

    private PipeData switchElementsAndReturnOptionalWarnings(File[] files, PipeData input) {
        Optional<String> switchWarning = this.switchElements(files);
        this.candidateDirectory.delete();
        return switchWarning.isPresent() ? PipeData.join(input, Topics.WARNINGS, (String)switchWarning.get()).toPipe() : PipeData.EMPTY;
    }

    private PipeData abortSwitchingWithError() {
        this.LOGGER.error("Not able to switch elements in {} directory: missing the candidate directory: {}. Internal error.", (Object)this.blessedDirectory.getAbsolutePath(), (Object)this.candidateDirectory.getAbsolutePath());
        return Pipeline.streamClosed();
    }

    private Optional<String> switchElements(File[] files) {
        try {
            this.switchElementsUnchecked(files);
            return Optional.absent();
        }
        catch (UncheckedIOException e) {
            return this.reportSwitchErrorAndManualSteps(e, files);
        }
    }

    private void switchElementsUnchecked(File[] files) {
        for (File file : files) {
            this.switchElement(file);
        }
    }

    private Optional<String> reportSwitchErrorAndManualSteps(UncheckedIOException e, File[] files) {
        String table = this.getTableOfRequiredOperations(files);
        String nixOps = this.getUnixRequiredOperations(files);
        String winOps = this.getWindowsRequiredOperations(files);
        String errorMessage = String.format(ERROR_MSG_THREE_PLACEHOLDERS, table, nixOps, winOps);
        this.LOGGER.error(errorMessage, (Throwable)e);
        System.err.println(errorMessage);
        return Optional.of((Object)errorMessage);
    }

    private String getTableOfRequiredOperations(File[] files) {
        Formatter tblFmt = new Formatter(new StringBuilder(), Locale.US);
        tblFmt.format(FORMAT, "Source", "Operation", "Destination");
        for (File file : files) {
            File targetFile = new File(this.blessedDirectory, file.getName());
            String absolutePath = targetFile.getAbsolutePath();
            if (targetFile.exists()) {
                File backupFile = new File(this.backupDirectory, file.getName());
                this.buildOperationTableRow(tblFmt, backupFile.getAbsolutePath(), absolutePath);
            }
            this.buildOperationTableRow(tblFmt, absolutePath, file.getAbsolutePath());
        }
        return tblFmt.toString();
    }

    private void buildOperationTableRow(Formatter tblFmt, String absolutePath, String absolutePath1) {
        tblFmt.format(FORMAT, StringUtils.abbreviate((String)absolutePath1, (int)50, (int)50), MOVE_OPERATION, StringUtils.abbreviate((String)absolutePath, (int)50, (int)50));
    }

    private String getUnixRequiredOperations(File[] files) {
        Formatter unixFmt = new Formatter(new StringBuilder(), Locale.US);
        return this.getRequiredOperations(files, unixFmt, UNIX_FORMAT);
    }

    private String getWindowsRequiredOperations(File[] files) {
        Formatter winFmt = new Formatter(new StringBuilder(), Locale.US);
        return this.getRequiredOperations(files, winFmt, WIN_FORMAT);
    }

    private String getRequiredOperations(File[] files, Formatter formatter, String format) {
        for (File file : files) {
            File targetFile = new File(this.blessedDirectory, file.getName());
            if (targetFile.exists()) {
                File backupFile = new File(this.backupDirectory, file.getName());
                formatter.format(format, targetFile.getAbsolutePath(), backupFile.getAbsolutePath());
            }
            formatter.format(format, file.getAbsolutePath(), targetFile.getAbsolutePath());
        }
        return formatter.toString();
    }

    private void switchElement(File sourceFile) {
        File targetFile = new File(this.blessedDirectory, sourceFile.getName());
        this.LOGGER.debug("Switching filesystem item from: {} to {}: ", (Object)sourceFile.getAbsolutePath(), (Object)targetFile.getAbsolutePath());
        this.ensureTargetFileGetsDeactivated(sourceFile, targetFile);
        this.move(sourceFile, targetFile);
    }

    private void ensureTargetFileGetsDeactivated(File sourceFile, File targetFile) {
        if (targetFile.exists()) {
            this.LOGGER.trace("Target filesystem item exist: {} and it needs to be backed up first. ", (Object)targetFile.getAbsolutePath());
            this.deactivateTargetFile(sourceFile, targetFile);
        }
    }

    private void deactivateTargetFile(File sourceFile, File targetFile) {
        if (this.ensureBackupDirectoryGetsCreated()) {
            File backupFile = new File(this.backupDirectory, sourceFile.getName());
            this.LOGGER.trace("Making a backup of existing target item: {} as {}", (Object)targetFile.getAbsolutePath(), (Object)backupFile.getAbsolutePath());
            this.move(targetFile, backupFile);
        } else {
            this.LOGGER.error("Not able to create backup directory: {}. Cannot continue switching: {} into {}", new Object[]{this.backupDirectory.getAbsolutePath(), sourceFile.getAbsolutePath(), targetFile.getAbsolutePath()});
        }
    }

    private void move(File sourceFile, File targetFile) {
        try {
            if (sourceFile.isDirectory()) {
                FileUtils.moveDirectory((File)sourceFile, (File)targetFile);
            } else {
                FileUtils.moveFile((File)sourceFile, (File)targetFile);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e, String.format("Cannot move filesystem item from: %s to %s", sourceFile.getAbsolutePath(), targetFile.getAbsolutePath()));
        }
    }

    private boolean ensureBackupDirectoryGetsCreated() {
        return this.backupDirectory.exists() || this.createBackupDirectory();
    }

    private boolean createBackupDirectory() {
        this.backupDirectory.mkdir();
        return this.backupDirectory.exists();
    }
}

