package com.aliyun.sdk.gateway.pop.interceptor.httpRequest;

import com.aliyun.auth.credentials.Credential;
import com.aliyun.auth.signature.SignerParams;
import com.aliyun.sdk.gateway.pop.auth.RpcSignatureComposer;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.credentials.provider.AnonymousCredentialProvider;
import com.aliyun.sdk.gateway.pop.auth.signer.PopSigner;
import com.aliyun.core.http.FormatType;
import com.aliyun.core.http.HttpHeaders;
import com.aliyun.core.http.HttpMethod;
import com.aliyun.core.http.HttpRequest;
import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.MapUtils;
import com.aliyun.core.utils.StringUtils;
import darabonba.core.RequestStyle;
import darabonba.core.TeaConfiguration;
import darabonba.core.TeaPair;
import darabonba.core.TeaRequest;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientOption;
import darabonba.core.interceptor.HttpRequestInterceptor;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.utils.CommonUtil;
import darabonba.core.utils.ModelUtil;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

public class RpcHttpReqInterceptor implements HttpRequestInterceptor {
    private final ClientLogger logger = new ClientLogger(RpcHttpReqInterceptor.class);

    @Override
    public HttpRequest modifyHttpRequest(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        HttpRequest httpRequest = context.httpRequest();
        if (!request.style().equals(RequestStyle.RPC)) {
            return httpRequest;
        }
        TeaConfiguration configuration = context.configuration();
        ClientConfiguration clientConfiguration = configuration.clientConfiguration();
        PopSigner signer = (PopSigner) clientConfiguration.option(ClientOption.SIGNER);
        if (signer.getSignerVersion() != SignatureVersion.V1) {
            return httpRequest;
        }
        HttpMethod method = Optional.ofNullable(configuration.method()).orElseGet(request::method);
        Map<String, String> query = CommonUtil.merge(String.class, CommonUtil.buildMap(
                new TeaPair("Timestamp", CommonUtil.getTimestamp()),
                new TeaPair("SignatureNonce", CommonUtil.getNonce())
                ),
                request.query()
        );

        Map<String, String> headers = new HashMap<>(request.headers().toMap());
        Map<String, String> body = new HashMap<>();
        if (!CommonUtil.isUnset(request.body())) {
            body = ModelUtil.query(CommonUtil.assertAsMap(request.body()));
            headers.put("content-type", FormatType.FORM);
        }
        Map<String, String> paramsToSign = MapUtils.concat(query, body);
        if (!(configuration.credentialProvider() instanceof AnonymousCredentialProvider)) {
            ICredential credential = configuration.credentialProvider().getCredentials();
            String securityToken = credential.securityToken();
            if (!StringUtils.isEmpty(securityToken)) {
                query.put("SecurityToken", securityToken);
            }
            query.putAll(CommonUtil.buildMap(
                    new TeaPair("SignatureMethod", signer.getSignerName().algorithmName()),
                    new TeaPair("SignatureVersion", "1.0"),
                    new TeaPair("AccessKeyId", credential.accessKeyId())
            ));
            paramsToSign.putAll(query);
            String strToSign = RpcSignatureComposer.composeStringToSign(method, paramsToSign, headers, request.pathname());
            SignerParams params = SignerParams.create();
            Credential newCredential = Credential.builder()
                    .accessKeyId(credential.accessKeyId())
                    .securityToken(securityToken)
                    .accessKeySecret(credential.accessKeySecret() + "&")
                    .build();
            params.setCredentials(newCredential);
            query.put("Signature", signer.signString(strToSign, params));
        }

        HttpHeaders httpHeaders = new HttpHeaders(headers);
        httpRequest = new HttpRequest(method,
                ModelUtil.composeUrl(configuration.endpoint(), query, configuration.protocol(), request.pathname()));
        httpRequest.setHeaders(httpHeaders);
        if (!body.isEmpty()) {
            httpRequest.setBody(StringUtils.toBytes(Objects.requireNonNull(ModelUtil.toFormString(body))));
        }
        return httpRequest;
    }
}
