package com.bettercloud.vault.api.pki;

import java.io.Serializable;
import java.util.List;

/**
 * <p>A container for credential details returned endpoints on the PKI backend.  (see:
 * {@link Pki#issue(String, String, List, List, String, CredentialFormat)} Pki.issue }.
 * This class is meant for use with a builder pattern style.  Example usage:</p>
 *
 * <blockquote>
 * <pre>{@code
 * final Credential options = new Credential()
 *                              .certificate(...)
 *                              .issuingCa(...)
 *                              .privateKey(...)
 *                              .privateKeyType(...)
 *                              .serialNumber(...);
 * }</pre>
 * </blockquote>
 */
public class Credential implements Serializable {

    private String certificate;
    private String issuingCa;
    private String privateKey;
    private String privateKeyType;
    private String serialNumber;

    /**
     * @param certificate A certificate, in PEM format
     * @return This object, with the certificate populated, ready for other builder methods or immediate use.
     */
    public Credential certificate(final String certificate) {
        this.certificate = certificate;
        return this;
    }

    /**
     * @param issuingCa The issuing CA certificate, in PEM format
     * @return This object, with the issuing CA certificate populated, ready for other builder methods or immediate use.
     */
    public Credential issuingCa(final String issuingCa) {
        this.issuingCa = issuingCa;
        return this;
    }

    /**
     * @param privateKey The private key, in PEM format
     * @return This object, with the private key populated, ready for other builder methods or immediate use.
     */
    public Credential privateKey(final String privateKey) {
        this.privateKey = privateKey;
        return this;
    }

    /**
     * @param privateKeyType The private key type (e.g. "rsa")
     * @return This object, with the private key type populated, ready for other builder methods or immediate use.
     */
    public Credential privateKeyType(final String privateKeyType) {
        this.privateKeyType = privateKeyType;
        return this;
    }

    /**
     * @param serialNumber An identifier generated by Vault
     * @return This object, with the serial number populated, ready for other builder methods or immediate use.
     */
    public Credential serialNumber(final String serialNumber) {
        this.serialNumber = serialNumber;
        return this;
    }

    public String getCertificate() {
        return certificate;
    }

    public String getIssuingCa() {
        return issuingCa;
    }

    public String getPrivateKey() {
        return privateKey;
    }

    public String getPrivateKeyType() {
        return privateKeyType;
    }

    public String getSerialNumber() {
        return serialNumber;
    }
}
