/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.jdbc.internal.client;

import com.facebook.presto.jdbc.internal.airlift.json.JsonCodec;
import com.facebook.presto.jdbc.internal.client.ClientException;
import com.facebook.presto.jdbc.internal.client.ClientSession;
import com.facebook.presto.jdbc.internal.client.JsonResponse;
import com.facebook.presto.jdbc.internal.client.OkHttpUtil;
import com.facebook.presto.jdbc.internal.client.QueryResults;
import com.facebook.presto.jdbc.internal.client.StatementStats;
import com.facebook.presto.jdbc.internal.guava.base.MoreObjects;
import com.facebook.presto.jdbc.internal.guava.base.Preconditions;
import com.facebook.presto.jdbc.internal.guava.base.Splitter;
import com.facebook.presto.jdbc.internal.guava.collect.ImmutableMap;
import com.facebook.presto.jdbc.internal.guava.collect.ImmutableSet;
import com.facebook.presto.jdbc.internal.guava.collect.Sets;
import com.facebook.presto.jdbc.internal.okhttp3.Headers;
import com.facebook.presto.jdbc.internal.okhttp3.HttpUrl;
import com.facebook.presto.jdbc.internal.okhttp3.MediaType;
import com.facebook.presto.jdbc.internal.okhttp3.OkHttpClient;
import com.facebook.presto.jdbc.internal.okhttp3.Request;
import com.facebook.presto.jdbc.internal.okhttp3.RequestBody;
import com.facebook.presto.jdbc.internal.spi.type.TimeZoneKey;
import java.io.Closeable;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
public class StatementClient
implements Closeable {
    private static final MediaType MEDIA_TYPE_JSON = MediaType.parse("application/json; charset=utf-8");
    private static final JsonCodec<QueryResults> QUERY_RESULTS_CODEC = JsonCodec.jsonCodec(QueryResults.class);
    private static final Splitter SESSION_HEADER_SPLITTER = Splitter.on('=').limit(2).trimResults();
    private static final String USER_AGENT_VALUE = StatementClient.class.getSimpleName() + "/" + MoreObjects.firstNonNull(StatementClient.class.getPackage().getImplementationVersion(), "unknown");
    private final OkHttpClient httpClient;
    private final boolean debug;
    private final String query;
    private final AtomicReference<QueryResults> currentResults = new AtomicReference();
    private final Map<String, String> setSessionProperties = new ConcurrentHashMap<String, String>();
    private final Set<String> resetSessionProperties = Sets.newConcurrentHashSet();
    private final Map<String, String> addedPreparedStatements = new ConcurrentHashMap<String, String>();
    private final Set<String> deallocatedPreparedStatements = Sets.newConcurrentHashSet();
    private final AtomicReference<String> startedtransactionId = new AtomicReference();
    private final AtomicBoolean clearTransactionId = new AtomicBoolean();
    private final AtomicBoolean closed = new AtomicBoolean();
    private final AtomicBoolean gone = new AtomicBoolean();
    private final AtomicBoolean valid = new AtomicBoolean(true);
    private final TimeZoneKey timeZone;
    private final long requestTimeoutNanos;
    private final String user;

    public StatementClient(OkHttpClient httpClient, ClientSession session, String query) {
        Objects.requireNonNull(httpClient, "httpClient is null");
        Objects.requireNonNull(session, "session is null");
        Objects.requireNonNull(query, "query is null");
        this.httpClient = httpClient;
        this.debug = session.isDebug();
        this.timeZone = session.getTimeZone();
        this.query = query;
        this.requestTimeoutNanos = session.getClientRequestTimeout().roundTo(TimeUnit.NANOSECONDS);
        this.user = session.getUser();
        Request request = this.buildQueryRequest(session, query);
        JsonResponse<QueryResults> response = JsonResponse.execute(QUERY_RESULTS_CODEC, httpClient, request);
        if (response.getStatusCode() != 200 || !response.hasValue()) {
            throw this.requestFailedException("starting query", request, response);
        }
        this.processResponse(response.getHeaders(), response.getValue());
    }

    private Request buildQueryRequest(ClientSession session, String query) {
        HttpUrl url = HttpUrl.get(session.getServer());
        if (url == null) {
            throw new ClientException("Invalid server URL: " + session.getServer());
        }
        url = url.newBuilder().encodedPath("/v1/statement").build();
        Request.Builder builder = this.prepareRequest(url).post(RequestBody.create(MEDIA_TYPE_JSON, query));
        if (session.getSource() != null) {
            builder.addHeader("X-Presto-Source", session.getSource());
        }
        if (session.getClientInfo() != null) {
            builder.addHeader("X-Presto-Client-Info", session.getClientInfo());
        }
        if (session.getCatalog() != null) {
            builder.addHeader("X-Presto-Catalog", session.getCatalog());
        }
        if (session.getSchema() != null) {
            builder.addHeader("X-Presto-Schema", session.getSchema());
        }
        builder.addHeader("X-Presto-Time-Zone", session.getTimeZone().getId());
        if (session.getLocale() != null) {
            builder.addHeader("X-Presto-Language", session.getLocale().toLanguageTag());
        }
        Map<String, String> property = session.getProperties();
        for (Map.Entry<String, String> entry : property.entrySet()) {
            builder.addHeader("X-Presto-Session", entry.getKey() + "=" + entry.getValue());
        }
        Map<String, String> statements = session.getPreparedStatements();
        for (Map.Entry<String, String> entry : statements.entrySet()) {
            builder.addHeader("X-Presto-Prepared-Statement", StatementClient.urlEncode(entry.getKey()) + "=" + StatementClient.urlEncode(entry.getValue()));
        }
        builder.addHeader("X-Presto-Transaction-Id", session.getTransactionId() == null ? "NONE" : session.getTransactionId());
        return builder.build();
    }

    public String getQuery() {
        return this.query;
    }

    public TimeZoneKey getTimeZone() {
        return this.timeZone;
    }

    public boolean isDebug() {
        return this.debug;
    }

    public boolean isClosed() {
        return this.closed.get();
    }

    public boolean isGone() {
        return this.gone.get();
    }

    public boolean isFailed() {
        return this.currentResults.get().getError() != null;
    }

    public StatementStats getStats() {
        return this.currentResults.get().getStats();
    }

    public QueryResults current() {
        Preconditions.checkState(this.isValid(), "current position is not valid (cursor past end)");
        return this.currentResults.get();
    }

    public QueryResults finalResults() {
        Preconditions.checkState(!this.isValid() || this.isFailed(), "current position is still valid");
        return this.currentResults.get();
    }

    public Map<String, String> getSetSessionProperties() {
        return ImmutableMap.copyOf(this.setSessionProperties);
    }

    public Set<String> getResetSessionProperties() {
        return ImmutableSet.copyOf(this.resetSessionProperties);
    }

    public Map<String, String> getAddedPreparedStatements() {
        return ImmutableMap.copyOf(this.addedPreparedStatements);
    }

    public Set<String> getDeallocatedPreparedStatements() {
        return ImmutableSet.copyOf(this.deallocatedPreparedStatements);
    }

    public String getStartedtransactionId() {
        return this.startedtransactionId.get();
    }

    public boolean isClearTransactionId() {
        return this.clearTransactionId.get();
    }

    public boolean isValid() {
        return this.valid.get() && !this.isGone() && !this.isClosed();
    }

    private Request.Builder prepareRequest(HttpUrl url) {
        return new Request.Builder().addHeader("X-Presto-User", this.user).addHeader("User-Agent", USER_AGENT_VALUE).url(url);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean advance() {
        URI nextUri = this.current().getNextUri();
        if (this.isClosed() || nextUri == null) {
            this.valid.set(false);
            return false;
        }
        Request request = this.prepareRequest(HttpUrl.get(nextUri)).build();
        RuntimeException cause = null;
        long start = System.nanoTime();
        long attempts = 0L;
        do {
            JsonResponse<QueryResults> response;
            if (attempts > 0L) {
                try {
                    TimeUnit.MILLISECONDS.sleep(attempts * 100L);
                }
                catch (InterruptedException e) {
                    try {
                        this.close();
                    }
                    finally {
                        Thread.currentThread().interrupt();
                    }
                    throw new RuntimeException("StatementClient thread was interrupted");
                }
            }
            ++attempts;
            try {
                response = JsonResponse.execute(QUERY_RESULTS_CODEC, this.httpClient, request);
            }
            catch (RuntimeException e) {
                cause = e;
                continue;
            }
            if (response.getStatusCode() == 200 && response.hasValue()) {
                this.processResponse(response.getHeaders(), response.getValue());
                return true;
            }
            if (response.getStatusCode() == 503) continue;
            throw this.requestFailedException("fetching next", request, response);
        } while (System.nanoTime() - start < this.requestTimeoutNanos && !this.isClosed());
        this.gone.set(true);
        throw new RuntimeException("Error fetching next", cause);
    }

    private void processResponse(Headers headers, QueryResults results) {
        List<String> keyValue;
        for (String setSession : headers.values("X-Presto-Set-Session")) {
            keyValue = SESSION_HEADER_SPLITTER.splitToList(setSession);
            if (keyValue.size() != 2) continue;
            this.setSessionProperties.put(keyValue.get(0), keyValue.get(1));
        }
        this.resetSessionProperties.addAll(headers.values("X-Presto-Clear-Session"));
        for (String entry : headers.values("X-Presto-Added-Prepare")) {
            keyValue = SESSION_HEADER_SPLITTER.splitToList(entry);
            if (keyValue.size() != 2) continue;
            this.addedPreparedStatements.put(StatementClient.urlDecode(keyValue.get(0)), StatementClient.urlDecode(keyValue.get(1)));
        }
        for (String entry : headers.values("X-Presto-Deallocated-Prepare")) {
            this.deallocatedPreparedStatements.add(StatementClient.urlDecode(entry));
        }
        String startedTransactionId = headers.get("X-Presto-Started-Transaction-Id");
        if (startedTransactionId != null) {
            this.startedtransactionId.set(startedTransactionId);
        }
        if (headers.values("X-Presto-Clear-Transaction-Id") != null) {
            this.clearTransactionId.set(true);
        }
        this.currentResults.set(results);
    }

    private RuntimeException requestFailedException(String task, Request request, JsonResponse<QueryResults> response) {
        this.gone.set(true);
        if (!response.hasValue()) {
            if (response.getStatusCode() == 401) {
                return new ClientException("Authentication failed" + Optional.ofNullable(response.getStatusMessage()).map(message -> ": " + message).orElse(""));
            }
            return new RuntimeException(String.format("Error %s at %s returned an invalid response: %s [Error: %s]", task, request.url(), response, response.getResponseBody()), response.getException());
        }
        return new RuntimeException(String.format("Error %s at %s returned HTTP %s", task, request.url(), response.getStatusCode()));
    }

    public void cancelLeafStage() {
        Preconditions.checkState(!this.isClosed(), "client is closed");
        URI uri = this.current().getPartialCancelUri();
        if (uri != null) {
            this.httpDelete(uri);
        }
    }

    @Override
    public void close() {
        URI uri;
        if (!this.closed.getAndSet(true) && (uri = this.currentResults.get().getNextUri()) != null) {
            this.httpDelete(uri);
        }
    }

    private void httpDelete(URI uri) {
        Request request = this.prepareRequest(HttpUrl.get(uri)).delete().build();
        this.httpClient.newCall(request).enqueue(new OkHttpUtil.NullCallback());
    }

    private static String urlEncode(String value) {
        try {
            return URLEncoder.encode(value, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new AssertionError((Object)e);
        }
    }

    private static String urlDecode(String value) {
        try {
            return URLDecoder.decode(value, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new AssertionError((Object)e);
        }
    }
}

