/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.commondatamodel.objectmodel.utilities.logger;

import com.microsoft.commondatamodel.objectmodel.cdm.CdmCorpusContext;
import com.microsoft.commondatamodel.objectmodel.enums.CdmLogCode;
import com.microsoft.commondatamodel.objectmodel.enums.CdmStatusLevel;
import com.microsoft.commondatamodel.objectmodel.enums.EnvironmentType;
import com.microsoft.commondatamodel.objectmodel.utilities.StringUtils;
import com.microsoft.commondatamodel.objectmodel.utilities.logger.Logger;
import com.microsoft.commondatamodel.objectmodel.utilities.logger.TelemetryClient;
import com.microsoft.commondatamodel.objectmodel.utilities.logger.TelemetryConfig;
import com.microsoft.commondatamodel.objectmodel.utilities.network.CdmHttpClient;
import com.microsoft.commondatamodel.objectmodel.utilities.network.CdmHttpRequest;
import com.microsoft.commondatamodel.objectmodel.utilities.network.CdmHttpRequestException;
import com.microsoft.commondatamodel.objectmodel.utilities.network.CdmHttpResponse;
import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

public class TelemetryKustoClient
implements TelemetryClient {
    private int maxNumRetries = 5;
    private int maxTimeoutMilliseconds = 10000;
    private int timeoutMilliseconds = 1000;
    private CdmCorpusContext ctx;
    private TelemetryConfig config;
    private ConcurrentLinkedQueue<ImmutablePair<CdmStatusLevel, String>> requestQueue;
    private CdmHttpClient httpClient;
    private static final String INGESTION_COMMAND = ".ingest inline into table";
    private static final int INGESTION_FREQUENCY = 500;
    private static final int BACKOFF_FOR_THROTTLING = 200;
    private static final ArrayList<String> LOG_EXEC_TIME_METHODS = new ArrayList<String>(){
        {
            this.add("createResolvedEntityAsync");
            this.add("calculateEntityGraphAsync");
            this.add("calculateEntityGraphAsync(perEntity)");
        }
    };

    public TelemetryKustoClient(CdmCorpusContext ctx, TelemetryConfig config) {
        this.ctx = ctx;
        this.config = config;
        this.httpClient = new CdmHttpClient();
        this.requestQueue = new ConcurrentLinkedQueue();
    }

    @Override
    public void addToIngestionQueue(String timestamp, CdmStatusLevel level, String className, String method, String corpusPath, String message, boolean requireIngestion, CdmLogCode code) {
        if (this.config == null || this.requestQueue == null) {
            return;
        }
        if (className.equals(TelemetryKustoClient.class.getSimpleName())) {
            return;
        }
        if (level == CdmStatusLevel.Progress && !requireIngestion) {
            if (LOG_EXEC_TIME_METHODS.contains(method)) {
                String execTimeMessage = "Leaving scope. Time Elapsed:";
                if (!message.startsWith("Leaving scope. Time Elapsed:")) {
                    return;
                }
            } else {
                return;
            }
        }
        if (this.config.getRemoveUserContent()) {
            corpusPath = null;
            if (level == CdmStatusLevel.Warning || level == CdmStatusLevel.Error) {
                message = null;
            }
        }
        String logEntry = this.processLogEntry(timestamp, className, method, message, code, corpusPath, this.ctx.getCorrelationId(), this.ctx.getEvents().getApiCorrelationId(), this.ctx.getCorpus().getAppId());
        this.requestQueue.add((ImmutablePair<CdmStatusLevel, String>)new ImmutablePair((Object)level, (Object)logEntry));
    }

    public boolean checkRequestQueueIsEmpty() {
        return this.requestQueue == null || this.requestQueue.peek() == null;
    }

    @Override
    public void enable() {
        if (this.config == null || this.requestQueue == null) {
            String message = "The telemetry client has not been initialized";
            Logger.warning(this.ctx, TelemetryKustoClient.class.getSimpleName(), "enable", null, CdmLogCode.WarnTelemetryIngestionFailed, "The telemetry client has not been initialized");
            return;
        }
        Runnable ingestionRunnable = new Runnable(){

            @Override
            public void run() {
                TelemetryKustoClient.this.ingestRequestQueue().join();
            }
        };
        Thread ingestionThread = new Thread(ingestionRunnable);
        ingestionThread.setDaemon(true);
        ingestionThread.start();
    }

    public CompletableFuture<Void> postKustoQuery(String query) {
        return CompletableFuture.runAsync(() -> {
            String authToken = this.config.getAuthenticationToken();
            String queryEndpoint = String.format("https://%s.kusto.windows.net/v1/rest/mgmt", this.config.getKustoClusterName());
            String kustoHost = String.format("%s.kusto.windows.net", this.config.getKustoClusterName());
            String queryBody = String.format("{\"db\":\"%s\",\"csl\":\"%s\"}", this.config.getKustoDatabaseName(), query);
            LinkedHashMap<String, String> headers = new LinkedHashMap<String, String>();
            headers.put("Accept", "application/json");
            headers.put("Authorization", authToken);
            headers.put("Host", kustoHost);
            CdmHttpRequest httpRequest = new CdmHttpRequest(queryEndpoint, this.maxNumRetries, "POST");
            httpRequest.setHeaders(headers);
            httpRequest.setContent(queryBody);
            httpRequest.setContentType("application/json");
            httpRequest.setTimeout(Duration.ofMillis(this.getTimeoutMilliseconds()));
            httpRequest.setMaximumTimeout(Duration.ofMillis(this.getMaxTimeoutMilliseconds()));
            CdmHttpResponse response = this.httpClient.sendAsync(httpRequest, this::getRetryWaitTime, this.ctx).join();
            if (response == null) {
                throw new CdmHttpRequestException("Kusto query post failed. The result of a request is undefined.");
            }
            if (!response.isSuccessful()) {
                throw new CdmHttpRequestException(String.format("Kusto query post failed. HTTP %s - %s.", response.getStatusCode(), response.getReason()));
            }
        });
    }

    private Duration getRetryWaitTime(CdmHttpResponse response, boolean hasFailed, int retryNumber) {
        if (response != null && response.isSuccessful() && !hasFailed) {
            return null;
        }
        Random random = new Random();
        int waitTime = random.nextInt(1 << retryNumber) * 200;
        return Duration.ofMillis(waitTime);
    }

    private CompletableFuture<Void> ingestIntoTable(String logTable, String logEntries) {
        return CompletableFuture.runAsync(() -> {
            if (!StringUtils.isNullOrEmpty(logEntries)) {
                String query = String.format("%s %s <|\n%s", INGESTION_COMMAND, logTable, logEntries);
                try {
                    this.postKustoQuery(query).join();
                }
                catch (Exception ex) {
                    Logger.info(this.ctx, TelemetryKustoClient.class.getSimpleName(), "ingestIntoTable", null, ex.getMessage());
                }
            }
        });
    }

    private CompletableFuture<Void> ingestRequestQueue() {
        return CompletableFuture.runAsync(() -> {
            while (true) {
                if (this.requestQueue.peek() != null) {
                    String infoLogEntries = "";
                    String warningLogEntries = "";
                    String errorLogEntries = "";
                    while (this.requestQueue.peek() != null) {
                        Pair request = (Pair)this.requestQueue.poll();
                        String logEntry = (String)request.getRight();
                        switch ((CdmStatusLevel)((Object)((Object)request.getLeft()))) {
                            case Progress: {
                                infoLogEntries = infoLogEntries + logEntry;
                                break;
                            }
                            case Warning: {
                                warningLogEntries = warningLogEntries + logEntry;
                                break;
                            }
                            case Error: {
                                errorLogEntries = errorLogEntries + logEntry;
                                break;
                            }
                        }
                    }
                    if (this.ctx != null && this.config != null) {
                        if (!StringUtils.isNullOrEmpty(infoLogEntries)) {
                            this.ingestIntoTable(this.config.getKustoInfoLogTable(), infoLogEntries).join();
                        }
                        if (!StringUtils.isNullOrEmpty(warningLogEntries)) {
                            this.ingestIntoTable(this.config.getKustoWarningLogTable(), warningLogEntries).join();
                        }
                        if (!StringUtils.isNullOrEmpty(errorLogEntries)) {
                            this.ingestIntoTable(this.config.getKustoErrorLogTable(), errorLogEntries).join();
                        }
                    }
                }
                try {
                    Thread.sleep(500L);
                    continue;
                }
                catch (Exception ex) {
                    Logger.info(this.ctx, TelemetryKustoClient.class.getSimpleName(), "ingestRequestQueue", null, ex.getMessage());
                    continue;
                }
                break;
            }
        });
    }

    private String processLogEntry(String timestamp, String className, String method, String message, CdmLogCode logCode, String corpusPath, String correlationId, UUID apiCorrelationId, String appId) {
        if (this.config.getIngestAtLevel() == EnvironmentType.PROD || this.config.getIngestAtLevel() == EnvironmentType.TEST) {
            corpusPath = null;
        }
        if (message == null) {
            message = "";
        }
        message = message.replace(",", ";");
        String code = logCode.toString();
        LinkedHashMap<String, String> property = new LinkedHashMap<String, String>();
        property.put("Environment", this.config.getIngestAtLevel().toString());
        property.put("SDKLanguage", "Java");
        property.put("Region", this.config.getRegion());
        String propertyJson = TelemetryKustoClient.serializeMap(property);
        return String.format("%s,%s,%s,%s,%s,%s,%s,%s,%s,%s\n", timestamp, className, method, message, code, corpusPath, correlationId, apiCorrelationId, appId, propertyJson);
    }

    private static String serializeMap(Map<String, String> map) {
        StringBuilder mapAsString = new StringBuilder();
        for (String key : map.keySet()) {
            mapAsString.append(key + ":" + map.get(key) + ";");
        }
        return mapAsString.toString();
    }

    public int getMaxNumRetries() {
        return this.maxNumRetries;
    }

    public void setMaxNumRetries(int maxNumRetries) {
        this.maxNumRetries = maxNumRetries;
    }

    public int getMaxTimeoutMilliseconds() {
        return this.maxTimeoutMilliseconds;
    }

    public void setMaxTimeoutMilliseconds(int maxTimeoutMilliseconds) {
        this.maxTimeoutMilliseconds = maxTimeoutMilliseconds;
    }

    public int getTimeoutMilliseconds() {
        return this.timeoutMilliseconds;
    }

    public void setTimeoutMilliseconds(int timeoutMilliseconds) {
        this.timeoutMilliseconds = timeoutMilliseconds;
    }
}

