/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.curator.framework.recipes.cache;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.netflix.curator.framework.CuratorFramework;
import com.netflix.curator.framework.api.BackgroundPathable;
import com.netflix.curator.framework.api.CuratorEvent;
import com.netflix.curator.framework.api.CuratorListener;
import com.netflix.curator.framework.api.Pathable;
import com.netflix.curator.framework.listen.ListenerContainer;
import com.netflix.curator.framework.recipes.cache.ChildData;
import com.netflix.curator.framework.recipes.cache.PathChildrenCacheEvent;
import com.netflix.curator.framework.recipes.cache.PathChildrenCacheListener;
import com.netflix.curator.framework.recipes.cache.PathChildrenCacheMode;
import com.netflix.curator.framework.state.ConnectionState;
import com.netflix.curator.framework.state.ConnectionStateListener;
import com.netflix.curator.utils.ZKPaths;
import java.io.Closeable;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PathChildrenCache
implements Closeable {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final CuratorFramework client;
    private final String path;
    private final PathChildrenCacheMode mode;
    private final ExecutorService executorService;
    private static final ChildData existingDataMarker = new ChildData(null, null, null);
    private final BlockingQueue<PathChildrenCacheEvent> listenerEvents = new LinkedBlockingQueue<PathChildrenCacheEvent>();
    private final ListenerContainer<PathChildrenCacheListener> listeners = new ListenerContainer();
    private final Map<String, ChildData> currentData = Maps.newConcurrentMap();
    private final Map<String, ChildData> incomingData = Maps.newConcurrentMap();
    private final Watcher watcher = new Watcher(){

        public void process(WatchedEvent event) {
            try {
                PathChildrenCache.this.processWatched(event);
            }
            catch (Exception e) {
                PathChildrenCache.this.handleException(e);
            }
        }
    };
    private final CuratorListener curatorListener = new CuratorListener(){

        public void eventReceived(CuratorFramework client, CuratorEvent event) throws Exception {
            PathChildrenCache.this.processEvent(event);
        }
    };
    private final ConnectionStateListener connectionStateListener = new ConnectionStateListener(){

        public void stateChanged(CuratorFramework client, ConnectionState newState) {
            PathChildrenCache.this.handleStateChange(newState);
        }
    };
    private static final ThreadFactory defaultThreadFactory = new ThreadFactoryBuilder().setNameFormat("PathChildrenCache-%d").build();

    public PathChildrenCache(CuratorFramework client, String path, PathChildrenCacheMode mode) {
        this(client, path, mode, defaultThreadFactory);
    }

    public PathChildrenCache(CuratorFramework client, String path, PathChildrenCacheMode mode, ThreadFactory threadFactory) {
        this.client = client;
        this.path = path;
        this.mode = mode;
        this.executorService = Executors.newFixedThreadPool(1, threadFactory);
    }

    public void start() throws Exception {
        Preconditions.checkArgument((!this.executorService.isShutdown() ? 1 : 0) != 0);
        this.client.getCuratorListenable().addListener((Object)this.curatorListener);
        this.client.getConnectionStateListenable().addListener((Object)this.connectionStateListener);
        this.executorService.submit(new Callable<Object>(){

            @Override
            public Object call() throws Exception {
                PathChildrenCache.this.listenerLoop();
                return null;
            }
        });
        this.refresh();
    }

    @Override
    public void close() throws IOException {
        Preconditions.checkArgument((!this.executorService.isShutdown() ? 1 : 0) != 0);
        this.client.getCuratorListenable().removeListener((Object)this.curatorListener);
        this.client.getConnectionStateListenable().removeListener((Object)this.connectionStateListener);
        this.executorService.shutdownNow();
    }

    public ListenerContainer<PathChildrenCacheListener> getListenable() {
        return this.listeners;
    }

    public List<ChildData> getCurrentData() {
        return ImmutableList.copyOf((Collection)Sets.newTreeSet(this.currentData.values()));
    }

    public ChildData getCurrentData(String fullPath) {
        return this.currentData.get(fullPath);
    }

    public void clearAndRefresh() throws Exception {
        this.currentData.clear();
        this.refresh();
    }

    protected void handleException(Throwable e) {
        this.log.error("", e);
    }

    private void handleStateChange(ConnectionState newState) {
        switch (newState) {
            case SUSPENDED: {
                this.currentData.clear();
                this.listenerEvents.offer(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.RESET, null));
                break;
            }
            case LOST: 
            case RECONNECTED: {
                try {
                    this.clearAndRefresh();
                    this.listenerEvents.offer(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.RESET, null));
                    break;
                }
                catch (Exception e) {
                    this.handleException(e);
                }
            }
        }
    }

    private void refresh() throws Exception {
        this.incomingData.clear();
        ((Pathable)((BackgroundPathable)this.client.getChildren().usingWatcher(this.watcher)).inBackground()).forPath(this.path);
    }

    private void listenerLoop() {
        while (!Thread.currentThread().isInterrupted()) {
            try {
                PathChildrenCacheEvent event = this.listenerEvents.take();
                this.callListeners(event);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
        }
    }

    private void callListeners(final PathChildrenCacheEvent event) {
        this.listeners.forEach((Function)new Function<PathChildrenCacheListener, Void>(){

            public Void apply(PathChildrenCacheListener listener) {
                try {
                    listener.childEvent(PathChildrenCache.this.client, event);
                }
                catch (Exception e) {
                    PathChildrenCache.this.handleException(e);
                }
                return null;
            }
        });
    }

    private void processEvent(CuratorEvent event) throws Exception {
        switch (event.getType()) {
            case CHILDREN: {
                this.processChildren(event.getChildren());
                break;
            }
            case GET_DATA: {
                this.processGetData(event.getPath(), event.getData());
                break;
            }
            case EXISTS: {
                this.processExists(event.getPath(), event.getStat());
                break;
            }
            case WATCHED: {
                this.processWatched(event.getWatchedEvent());
                break;
            }
        }
    }

    private void processWatched(WatchedEvent watchedEvent) throws Exception {
        switch (watchedEvent.getType()) {
            case NodeDataChanged: {
                this.addIncomingPath(watchedEvent.getPath());
                break;
            }
            case None: {
                break;
            }
            default: {
                this.refresh();
            }
        }
    }

    private void checkSetCurrent() {
        for (Map.Entry<String, ChildData> entry : this.incomingData.entrySet()) {
            String path = entry.getKey();
            ChildData data = entry.getValue();
            if (data.isComplete(this.mode)) {
                boolean isNew = this.currentData.put(data.getPath(), data) == null;
                this.incomingData.remove(path);
                this.listenerEvents.offer(new PathChildrenCacheEvent(isNew ? PathChildrenCacheEvent.Type.CHILD_ADDED : PathChildrenCacheEvent.Type.CHILD_UPDATED, data));
                continue;
            }
            if (!PathChildrenCache.isTheExistingDataMarker(data)) continue;
            ChildData removedData = this.currentData.remove(path);
            this.incomingData.remove(path);
            this.listenerEvents.offer(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CHILD_REMOVED, removedData));
        }
    }

    private void processExists(String path, Stat stat) {
        ChildData data = this.incomingData.get(path);
        if (data != null) {
            data = data.setStat(stat);
            this.incomingData.put(path, data);
        }
        this.checkSetCurrent();
    }

    private void processGetData(String path, byte[] bytes) {
        ChildData data = this.incomingData.get(path);
        if (data != null) {
            data = data.setData(bytes);
            this.incomingData.put(path, data);
        }
        this.checkSetCurrent();
    }

    private void processChildren(List<String> children) throws Exception {
        for (String path : this.currentData.keySet()) {
            this.incomingData.put(path, existingDataMarker);
        }
        for (String child : children) {
            String actualPath = ZKPaths.makePath((String)this.path, (String)child);
            this.addIncomingPath(actualPath);
        }
        this.checkSetCurrent();
    }

    private void addIncomingPath(String actualPath) throws Exception {
        this.incomingData.put(actualPath, new ChildData(actualPath, null, null));
        switch (this.mode) {
            case CACHE_DATA_AND_STAT: {
                ((Pathable)this.client.checkExists().inBackground()).forPath(actualPath);
                ((Pathable)((BackgroundPathable)this.client.getData().usingWatcher(this.watcher)).inBackground()).forPath(actualPath);
                break;
            }
            case CACHE_DATA: {
                ((Pathable)((BackgroundPathable)this.client.getData().usingWatcher(this.watcher)).inBackground()).forPath(actualPath);
                break;
            }
        }
    }

    private static boolean isTheExistingDataMarker(ChildData data) {
        return data == existingDataMarker;
    }
}

