/*
 * Decompiled with CFR 0.152.
 */
package edu.stanford.nlp.trees;

import edu.stanford.nlp.ling.Word;
import edu.stanford.nlp.stats.ClassicCounter;
import edu.stanford.nlp.stats.Counters;
import edu.stanford.nlp.trees.EnglishGrammaticalRelations;
import edu.stanford.nlp.trees.GrammaticalRelation;
import edu.stanford.nlp.trees.GrammaticalStructure;
import edu.stanford.nlp.trees.GraphLessGrammaticalStructureFactory;
import edu.stanford.nlp.trees.TreeGraphNode;
import edu.stanford.nlp.trees.TypedDependency;
import edu.stanford.nlp.trees.fakeShortNameToGRel;
import edu.stanford.nlp.util.Generics;
import edu.stanford.nlp.util.StringUtils;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

public class DependencyScoring {
    public static final boolean VERBOSE = false;
    public final List<Set<TypedDependency>> goldDeps;
    public final List<Set<TypedDependency>> goldDepsUnlabeled;
    public final boolean ignorePunc;

    private static List<Set<TypedDependency>> toSets(Collection<TypedDependency> depCollection) {
        Set<TypedDependencyStringEquality> depSet = Generics.newHashSet();
        Set<TypedDependencyStringEquality> unlabeledDepSet = Generics.newHashSet();
        for (TypedDependency dep : depCollection) {
            unlabeledDepSet.add(new TypedDependencyStringEquality(null, dep.gov(), dep.dep()));
            depSet.add(new TypedDependencyStringEquality(dep.reln(), dep.gov(), dep.dep()));
        }
        ArrayList<Set<TypedDependency>> l = new ArrayList<Set<TypedDependency>>(2);
        l.add(depSet);
        l.add(unlabeledDepSet);
        return l;
    }

    public DependencyScoring(List<Collection<TypedDependency>> goldDeps, boolean ignorePunc) {
        this.goldDeps = new ArrayList<Set<TypedDependency>>(goldDeps.size());
        this.goldDepsUnlabeled = new ArrayList<Set<TypedDependency>>(goldDeps.size());
        this.ignorePunc = ignorePunc;
        for (Collection<TypedDependency> depCollection : goldDeps) {
            List<Set<TypedDependency>> sets = DependencyScoring.toSets(depCollection);
            this.goldDepsUnlabeled.add(sets.get(1));
            this.goldDeps.add(sets.get(0));
        }
        if (ignorePunc) {
            DependencyScoring.removeHeadsAssignedToPunc(this.goldDeps);
            DependencyScoring.removeHeadsAssignedToPunc(this.goldDepsUnlabeled);
        }
    }

    private static void removeHeadsAssignedToPunc(Set<TypedDependency> depSet) {
        ArrayList<TypedDependency> deps = new ArrayList<TypedDependency>(depSet);
        for (TypedDependency dep : deps) {
            if (!DependencyScoring.langIndependentPuncCheck(dep.dep().label().word())) continue;
            depSet.remove(dep);
        }
    }

    private static void removeHeadsAssignedToPunc(List<Set<TypedDependency>> depSets) {
        for (Set<TypedDependency> depSet : depSets) {
            DependencyScoring.removeHeadsAssignedToPunc(depSet);
        }
    }

    public static boolean langIndependentPuncCheck(String token) {
        int codepoint;
        boolean isNotWord = true;
        for (int offset = 0; offset < token.length(); offset += Character.charCount(codepoint)) {
            codepoint = token.codePointAt(offset);
            if (!Character.isLetterOrDigit(codepoint)) continue;
            isNotWord = false;
        }
        return isNotWord;
    }

    public static DependencyScoring newInstanceStringEquality(List<Collection<TypedDependency>> goldDeps, boolean ignorePunc) {
        return new DependencyScoring(DependencyScoring.convertStringEquality(goldDeps), ignorePunc);
    }

    public DependencyScoring(String filename, boolean CoNLLX, boolean ignorePunc) throws IOException {
        this(CoNLLX ? DependencyScoring.readDepsCoNLLX(filename) : DependencyScoring.readDeps(filename), ignorePunc);
    }

    public DependencyScoring(String filename) throws IOException {
        this(filename, false, false);
    }

    public static List<Collection<TypedDependency>> convertStringEquality(List<Collection<TypedDependency>> deps) {
        ArrayList<Collection<TypedDependency>> convertedDeps = new ArrayList<Collection<TypedDependency>>();
        for (Collection<TypedDependency> depSet : deps) {
            Set<TypedDependencyStringEquality> converted = Generics.newHashSet();
            for (TypedDependency dep : depSet) {
                converted.add(new TypedDependencyStringEquality(dep.reln(), dep.gov(), dep.dep()));
            }
            convertedDeps.add(converted);
        }
        return convertedDeps;
    }

    protected static String normalizeNumbers(String token) {
        String norm = token.replaceFirst("^([0-9]+)-([0-9]+)$", "<num>-$2");
        if (!norm.equals(token)) {
            System.err.printf("Normalized numbers in token: %s => %s\n", token, norm);
        }
        return token;
    }

    protected static List<Collection<TypedDependency>> readDepsCoNLLX(String filename) throws IOException {
        List<GrammaticalStructure> gss = GrammaticalStructure.readCoNLLXGrammaticStructureCollection(filename, new fakeShortNameToGRel(), new GraphLessGrammaticalStructureFactory());
        ArrayList<Collection<TypedDependency>> readDeps = new ArrayList<Collection<TypedDependency>>(gss.size());
        for (GrammaticalStructure gs : gss) {
            Collection<TypedDependency> deps = gs.typedDependencies();
            readDeps.add(deps);
        }
        return readDeps;
    }

    protected static List<Collection<TypedDependency>> readDeps(String filename) throws IOException {
        LineNumberReader breader = new LineNumberReader(new FileReader(filename));
        ArrayList<Collection<TypedDependency>> readDeps = new ArrayList<Collection<TypedDependency>>();
        ArrayList<TypedDependencyStringEquality> deps = new ArrayList<TypedDependencyStringEquality>();
        String line = breader.readLine();
        while (line != null) {
            if (line.equals("null(-0,-0)") || line.equals("null(-1,-1)")) {
                readDeps.add(deps);
                deps = new ArrayList();
            } else {
                try {
                    if (line.equals("")) {
                        if (deps.size() != 0) {
                            readDeps.add(deps);
                            deps = new ArrayList();
                        }
                    } else {
                        int firstParen = line.indexOf("(");
                        int commaSpace = line.indexOf(", ");
                        String depName = line.substring(0, firstParen);
                        String govName = line.substring(firstParen + 1, commaSpace);
                        String childName = line.substring(commaSpace + 2, line.length() - 1);
                        GrammaticalRelation grel = GrammaticalRelation.valueOf(depName);
                        if (depName.startsWith("prep_")) {
                            String prep = depName.substring(5);
                            grel = EnglishGrammaticalRelations.getPrep(prep);
                        }
                        if (depName.startsWith("prepc_")) {
                            String prepc = depName.substring(6);
                            grel = EnglishGrammaticalRelations.getPrepC(prepc);
                        }
                        if (depName.startsWith("conj_")) {
                            String conj = depName.substring(5);
                            grel = EnglishGrammaticalRelations.getConj(conj);
                        }
                        if (grel == null) {
                            throw new RuntimeException("Unknown grammatical relation '" + depName + "'");
                        }
                        Word govWord = new Word(DependencyScoring.normalizeNumbers(govName));
                        Word childWord = new Word(DependencyScoring.normalizeNumbers(childName));
                        TypedDependencyStringEquality dep = new TypedDependencyStringEquality(grel, new TreeGraphNode(govWord), new TreeGraphNode(childWord));
                        deps.add(dep);
                    }
                }
                catch (Exception e) {
                    breader.close();
                    throw new RuntimeException("Error on line " + breader.getLineNumber() + ":\n\n" + e);
                }
            }
            line = breader.readLine();
        }
        if (deps.size() != 0) {
            readDeps.add(deps);
        }
        breader.close();
        return readDeps;
    }

    public Score score(List<Collection<TypedDependency>> system) {
        int parserCnt = 0;
        int goldCnt = 0;
        int parserUnlabeledCnt = 0;
        int goldUnlabeledCnt = 0;
        int correctAttachment = 0;
        int correctUnlabeledAttachment = 0;
        int labelCnt = 0;
        int labelCorrect = 0;
        ClassicCounter<String> unlabeledErrorCounts = new ClassicCounter<String>();
        ClassicCounter<String> labeledErrorCounts = new ClassicCounter<String>();
        for (int i = 0; i < system.size(); ++i) {
            String sGov;
            String sChild;
            List<Set<TypedDependency>> l = DependencyScoring.toSets(system.get(i));
            if (this.ignorePunc) {
                DependencyScoring.removeHeadsAssignedToPunc(l.get(0));
                DependencyScoring.removeHeadsAssignedToPunc(l.get(1));
            }
            parserCnt += l.get(0).size();
            goldCnt += this.goldDeps.get(i).size();
            parserUnlabeledCnt += l.get(1).size();
            goldUnlabeledCnt += this.goldDepsUnlabeled.get(i).size();
            l.get(0).retainAll((Collection)this.goldDeps.get(i));
            l.get(1).retainAll((Collection)this.goldDepsUnlabeled.get(i));
            correctAttachment += l.get(0).size();
            correctUnlabeledAttachment += l.get(1).size();
            labelCnt += l.get(1).size();
            labelCorrect += l.get(0).size();
            List<Set<TypedDependency>> errl = DependencyScoring.toSets(system.get(i));
            errl.get(0).removeAll((Collection)this.goldDeps.get(i));
            errl.get(1).removeAll((Collection)this.goldDepsUnlabeled.get(i));
            Map<String, String> childCorrectWithLabel = Generics.newHashMap();
            Map<String, String> childCorrectWithOutLabel = Generics.newHashMap();
            for (TypedDependency goldDep : this.goldDeps.get(i)) {
                sChild = goldDep.dep().label().toString().replaceFirst("-[^-]*$", "");
                String prefixLabeled = "";
                String prefixUnlabeled = "";
                if (childCorrectWithLabel.containsKey(sChild)) {
                    prefixLabeled = (String)childCorrectWithLabel.get(sChild) + ", ";
                    prefixUnlabeled = (String)childCorrectWithOutLabel.get(sChild) + ", ";
                }
                childCorrectWithLabel.put(sChild, prefixLabeled + goldDep.reln() + "(" + goldDep.gov().label().toString().replaceFirst("-[^-]*$", "") + ", " + sChild + ")");
                childCorrectWithOutLabel.put(sChild, prefixUnlabeled + "dep(" + goldDep.gov().label().toString().replaceFirst("-[^-]*$", "") + ", " + sChild + ")");
            }
            for (TypedDependency labeledError : errl.get(0)) {
                sChild = labeledError.dep().label().toString().replaceFirst("-[^-]*$", "");
                sGov = labeledError.gov().label().toString().replaceFirst("-[^-]*$", "");
                labeledErrorCounts.incrementCount(labeledError.reln().toString() + "(" + sGov + ", " + sChild + ") <= " + (String)childCorrectWithLabel.get(sChild));
            }
            for (TypedDependency unlabeledError : errl.get(1)) {
                sChild = unlabeledError.dep().label().toString().replaceFirst("-[^-]*$", "");
                sGov = unlabeledError.gov().label().toString().replaceFirst("-[^-]*$", "");
                unlabeledErrorCounts.incrementCount("dep(" + sGov + ", " + sChild + ") <= " + (String)childCorrectWithOutLabel.get(sChild));
            }
        }
        return new Score(parserCnt, goldCnt, parserUnlabeledCnt, goldUnlabeledCnt, correctAttachment, correctUnlabeledAttachment, labelCnt, labelCorrect, labeledErrorCounts, unlabeledErrorCounts);
    }

    public static void main(String[] args) throws IOException {
        Properties props = StringUtils.argsToProperties(args);
        boolean verbose = Boolean.parseBoolean(props.getProperty("v", "False"));
        boolean conllx = Boolean.parseBoolean(props.getProperty("conllx", "False"));
        boolean jsonOutput = Boolean.parseBoolean(props.getProperty("jsonOutput", "False"));
        boolean ignorePunc = Boolean.parseBoolean(props.getProperty("nopunc", "False"));
        String goldFilename = props.getProperty("g");
        String systemFilename = props.getProperty("s");
        if (goldFilename == null || systemFilename == null) {
            System.err.println("Usage:\n\tjava ...DependencyScoring [-v True/False] [-conllx True/False] [-jsonOutput True/False] [-ignorePunc True/False] -g goldFile -s systemFile\n");
            System.err.println("\nOptions:\n\t-v verbose output");
            System.exit(-1);
        }
        DependencyScoring goldScorer = new DependencyScoring(goldFilename, conllx, ignorePunc);
        List<Collection<TypedDependency>> systemDeps = conllx ? DependencyScoring.readDepsCoNLLX(systemFilename) : DependencyScoring.readDeps(systemFilename);
        Score score = goldScorer.score(systemDeps);
        if (conllx) {
            System.out.println(score.toStringAttachmentScore(jsonOutput));
        } else {
            System.out.println(score.toStringFScore(verbose, jsonOutput));
        }
    }

    public class Score {
        final int parserCnt;
        final int goldCnt;
        final int parserUnlabeledCnt;
        final int goldUnlabeledCnt;
        final int correctAttachment;
        final int correctUnlabeledAttachment;
        final int labelCnt;
        final int labelCorrect;
        final ClassicCounter<String> unlabeledErrorCounts;
        final ClassicCounter<String> labeledErrorCounts;

        public Score(int parserCnt, int goldCnt, int parserUnlabeledCnt, int goldUnlabeledCnt, int correctAttachment, int correctUnlabeledAttachment, int labelCnt, int labelCorrect, ClassicCounter<String> labeledErrorCounts, ClassicCounter<String> unlabeledErrorCounts) {
            this.parserCnt = parserCnt;
            this.goldCnt = goldCnt;
            this.parserUnlabeledCnt = parserUnlabeledCnt;
            this.goldUnlabeledCnt = goldUnlabeledCnt;
            this.correctAttachment = correctAttachment;
            this.correctUnlabeledAttachment = correctUnlabeledAttachment;
            this.labelCnt = labelCnt;
            this.labelCorrect = labelCorrect;
            this.unlabeledErrorCounts = new ClassicCounter<String>(unlabeledErrorCounts);
            this.labeledErrorCounts = new ClassicCounter<String>(labeledErrorCounts);
        }

        public String toString() {
            return this.toStringFScore(false, false);
        }

        public String toStringAttachmentScore(boolean json) {
            if (this.parserCnt != this.goldCnt) {
                throw new RuntimeException(String.format("AttachmentScore cannot be used when count(gold deps:%d) != count(system deps:%d)", this.parserCnt, this.goldCnt));
            }
            double las = (double)this.correctAttachment / (double)this.goldCnt;
            double uas = (double)this.correctUnlabeledAttachment / (double)this.goldCnt;
            StringBuilder sbuild = new StringBuilder();
            if (json) {
                sbuild.append("{");
                sbuild.append(String.format("'LAS' : %.3f, ", las));
                sbuild.append(String.format("'UAS' : %.3f, ", uas));
                sbuild.append("}");
            } else {
                sbuild.append(String.format("|| Labeled Attachment Score   ||", new Object[0]));
                sbuild.append(String.format(" %.3f (%d/%d) ||\n", las, this.correctAttachment, this.goldCnt));
                sbuild.append(String.format("|| Unlabeled Attachment Score ||", new Object[0]));
                sbuild.append(String.format(" %.3f (%d/%d) ||\n", uas, this.correctUnlabeledAttachment, this.goldCnt));
            }
            return sbuild.toString();
        }

        public String toStringFScore(boolean verbose, boolean json) {
            double lp = (double)this.correctAttachment / (double)this.parserCnt;
            double lr = (double)this.correctAttachment / (double)this.goldCnt;
            double lf = 2.0 * (lp * lr) / (lp + lr);
            double ulp = (double)this.correctUnlabeledAttachment / (double)this.parserUnlabeledCnt;
            double ulr = (double)this.correctUnlabeledAttachment / (double)this.goldUnlabeledCnt;
            double ulf = 2.0 * (ulp * ulr) / (ulp + ulr);
            StringBuilder sbuild = new StringBuilder();
            if (json) {
                sbuild.append("{");
                sbuild.append(String.format("'LF1' : %.3f, ", lf));
                sbuild.append(String.format("'LP' : %.3f, ", lp));
                sbuild.append(String.format("'LR' : %.3f, ", lr));
                sbuild.append(String.format("'UF1' : %.3f, ", ulf));
                sbuild.append(String.format("'UP' : %.3f, ", ulp));
                sbuild.append(String.format("'UR' : %.3f, ", ulr));
                sbuild.append("}");
            } else {
                sbuild.append(String.format("|| Labeled Attachment   || F ||  P ||  R ||\n", new Object[0]));
                sbuild.append(String.format("||                      || %.3f || %.3f (%d/%d) || %.3f (%d/%d)||\n", lf, lp, this.correctAttachment, this.parserCnt, lr, this.correctAttachment, this.goldCnt));
                sbuild.append(String.format("|| Unlabeled Attachment || F ||  P ||  R ||\n", new Object[0]));
                sbuild.append(String.format("||                     || %.3f || %.3f (%d/%d) || %.3f (%d/%d)||\n", ulf, ulp, this.correctUnlabeledAttachment, this.parserCnt, ulr, this.correctUnlabeledAttachment, this.goldCnt));
                if (verbose) {
                    sbuild.append("\nLabeled Attachment Error Counts\n");
                    sbuild.append(Counters.toSortedString(this.labeledErrorCounts, Integer.MAX_VALUE, "\t%2$f\t%1$s", "\n"));
                    sbuild.append("\n");
                    sbuild.append("\nUnlabeled Attachment Error Counts\n");
                    sbuild.append(Counters.toSortedString(this.unlabeledErrorCounts, Integer.MAX_VALUE, "\t%2$f\t%1$s", "\n"));
                }
            }
            return sbuild.toString();
        }
    }

    private static class TypedDependencyStringEquality
    extends TypedDependency {
        private static final long serialVersionUID = 1L;

        public TypedDependencyStringEquality(GrammaticalRelation reln, TreeGraphNode gov, TreeGraphNode dep) {
            super(reln, gov, dep);
        }

        @Override
        public boolean equals(Object o) {
            return o.toString().toLowerCase().equals(this.toString().toLowerCase());
        }

        @Override
        public int hashCode() {
            return this.toString().toLowerCase().hashCode();
        }
    }
}

