package io.github.xinyangpan.wechat4j.controller;

import java.io.IOException;
import java.util.Optional;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.fasterxml.jackson.dataformat.xml.XmlMapper;

import io.github.xinyangpan.wechat4j.api.Api;
import io.github.xinyangpan.wechat4j.core.CoreUtils;
import io.github.xinyangpan.wechat4j.core.WechatExtProperties;
import io.github.xinyangpan.wechat4j.core.WechatExtService;
import io.github.xinyangpan.wechat4j.core.dto.PayInfo;
import io.github.xinyangpan.wechat4j.core.dto.WxConfig;
import io.github.xinyangpan.wechat4j.core.dto.xml.pay.PayCallbackInfo;
import io.github.xinyangpan.wechat4j.core.dto.xml.pay.PayCallbackResult;
import io.github.xinyangpan.wechat4j.core.listener.IncomeMessageListener;
import io.github.xinyangpan.wechat4j.core.oauth.Scope;
import io.github.xinyangpan.wechat4j.core.pay.WechatPayService;

@Controller
@RequestMapping(value = "/wechat")
public class WechatExtController {
	private static final Logger log = LoggerFactory.getLogger(WechatExtController.class);
	@Autowired
	private WechatExtProperties wechatExtProperties;
	@Autowired
	private WechatExtService wechatExtService;
	@Autowired
	private Api api;
	private XmlMapper xmlMapper = CoreUtils.defaultXmlMapper();
	// 
	@Autowired
	private IncomeMessageListener incomeMessageListener;
	// 
	@Autowired
	private WechatPayService wechatPayService;

	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public String token(
		@RequestParam("signature") String signature,
		@RequestParam("echostr") String echostr,
		@RequestParam("timestamp") String timestamp,
		@RequestParam("nonce") String nonce) {
		// 
		if (!wechatExtService.isSignatureValid(signature, wechatExtProperties.getToken(), timestamp, nonce)) {
			log.warn("invalid signature: " + signature);
			return "fail";
		}
		return echostr;
	}

	@RequestMapping(value = "/redirect", method = RequestMethod.GET)
	public void weixinRedirect(HttpServletRequest request, HttpServletResponse response, Optional<Scope> scope) throws IOException {
		wechatExtService.redirectToOAuth2Server(response, wechatExtProperties.getRedirectUri(), scope.orElse(Scope.snsapi_base));
		log.debug("Redirected for scope={}.", scope);
	}

	@RequestMapping(method = RequestMethod.POST, consumes = "text/xml")
	@ResponseBody
	public String business(
		@RequestParam("signature") String signature,
		@RequestParam("timestamp") String timestamp,
		@RequestParam("nonce") String nonce,
		@RequestBody String content) throws IOException {
		// 
		if (!wechatExtService.isSignatureValid(signature, wechatExtProperties.getToken(), timestamp, nonce)) {
			log.warn("invalid signature: " + signature);
			return "fail";
		}
		// 
		log.info("business method. content:\n{}", content);
		return incomeMessageListener.onMessage(content);
	}

	@GetMapping("/wxConfig")
	@ResponseBody
	public WxConfig wxConfig(@RequestParam String url) {
		log.info("requsting wx config for {}.", url);
		return api.core().generateWxConfig(url);
	}

	@PostMapping("/pay")
	@ResponseBody
	public PayInfo pay(HttpServletRequest request, HttpServletResponse response) {
		log.info("Pay: {}.", request.getParameter("billId"));
		return wechatExtService.sign(wechatPayService.pay(request));
	}

//	@PostMapping("/payCallback")
//	@ResponseBody
//	public PayCallbackResult payCallback(@RequestBody PayCallbackInfo payCallbackInfo) {
//		log.info("PayCallback info: {}.", payCallbackInfo);
//		wechatExtService.checkSign(payCallbackInfo);
//		return wechatPayService.payCallback(payCallbackInfo);
//	}

	@PostMapping("/payCallback")
	@ResponseBody
	public PayCallbackResult payCallback(@RequestBody String callbackInfo) throws IOException {
		log.info("CallbackInfo info: {}.", callbackInfo);
		PayCallbackInfo payCallbackInfo = xmlMapper.readValue(callbackInfo, PayCallbackInfo.class);
		log.info("PayCallback info: {}.", payCallbackInfo);
		wechatExtService.checkSign(payCallbackInfo);
		return wechatPayService.payCallback(payCallbackInfo);
	}

	// -----------------------------
	// ----- Get Set ToString HashCode Equals
	// -----------------------------

	public WechatExtProperties getWechatExtProperties() {
		return wechatExtProperties;
	}

	public void setWechatExtProperties(WechatExtProperties wechatExtProperties) {
		this.wechatExtProperties = wechatExtProperties;
	}

	public IncomeMessageListener getIncomeMessageListener() {
		return incomeMessageListener;
	}

	public void setIncomeMessageListener(IncomeMessageListener incomeMessageListener) {
		this.incomeMessageListener = incomeMessageListener;
	}

	public WechatExtService getWechatExtService() {
		return wechatExtService;
	}

	public void setWechatExtService(WechatExtService wechatExtService) {
		this.wechatExtService = wechatExtService;
	}

}
