package io.github.xinyangpan.wechat4j.core;

import static io.github.xinyangpan.wechat4j.core.CoreUtils.defaultObjectMapper;

import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import com.google.common.base.Joiner;

import io.github.xinyangpan.wechat4j.core.dto.json.UserBaseInfo;
import io.github.xinyangpan.wechat4j.core.dto.xml.pay.UnifiedOrder;
import io.github.xinyangpan.wechat4j.core.oauth.Result;
import io.github.xinyangpan.wechat4j.core.oauth.Scope;
import io.github.xinyangpan.wechat4j.core.pay.HasSign;
import okhttp3.HttpUrl;

public class WechatExtService {
	private static final Logger log = LoggerFactory.getLogger(WechatExtService.class);
	private static String BUSINESS_URL = "https://api.weixin.qq.com/cgi-bin";
	private static String CS_URL = "https://api.weixin.qq.com/customservice";
	private static String PAY_URL = "https://api.mch.weixin.qq.com/pay";
	// 
	private RestWrapper restWrapper;
	private WechatExtProperties wechatExtProperties;

	public HttpUrl.Builder commonBuilder() {
		return HttpUrl.parse(BUSINESS_URL).newBuilder();
	}
	
	public HttpUrl.Builder csBuilder() {
		return HttpUrl.parse(CS_URL).newBuilder();
	}
	
	public HttpUrl.Builder payBuilder() {
		return HttpUrl.parse(PAY_URL).newBuilder();
	}

	/**
	 * @param scope 
	 * 		snsapi_base - 不弹出授权页面，直接跳转，只能获取用户openid
	 * 		snsapi_userinfo - 弹出授权页面，可通过openid拿到昵称、性别、所在地。并且， 即使在未关注的情况下，只要用户授权，也能获取其信息
	 * @return authorize Url
	 */
	public String authorizeUrl(Scope scope) {
		return this.authorizeUrl(scope, wechatExtProperties.getRedirectUri());
	}

	public String authorizeUrl(Scope scope, String redirectUri) {
		String appId = wechatExtProperties.getAppId();
		String responseType = "code";
		return HttpUrl.parse("https://open.weixin.qq.com/connect/oauth2/authorize").newBuilder()//
			.addQueryParameter("appid", appId)//
			.addQueryParameter("redirect_uri", redirectUri)//
			.addQueryParameter("response_type", responseType)//
			.addQueryParameter("scope", scope.name())//
			.addQueryParameter("state", wechatExtProperties.getOauthState())//
			.encodedFragment("wechat_redirect").toString();
	}

	public UserBaseInfo userBaseInfo(String code) {
		String url = HttpUrl.parse("https://api.weixin.qq.com/sns/oauth2/access_token").newBuilder()//
			.addQueryParameter("appid", wechatExtProperties.getAppId())//
			.addQueryParameter("secret", wechatExtProperties.getAppSecret())//
			.addQueryParameter("code", code)//
			.addQueryParameter("grant_type", "authorization_code")//
			.toString();
		return restWrapper.getForObject(url, UserBaseInfo.class);
	}

	public boolean isSignatureValid(String signature, String token, String timestamp, String nonce) {
		log.debug("isSignatureValid: signature={}, token={}, timestamp={}, nonce={}", signature, token, timestamp, nonce);
		String[] strs = new String[] { token, timestamp, nonce };
		Arrays.sort(strs);
		String concat = Joiner.on("").join(strs);
		String calculateSignature = DigestUtils.sha1Hex(concat);
		log.debug("Calculated Signature: {}", calculateSignature);
		return Objects.equals(signature, calculateSignature);
	}

	public long now() {
		return System.currentTimeMillis();
	}

	public boolean isStateValid(String state) {
		if (Objects.equals(wechatExtProperties.getOauthState(), state)) {
			return true;
		} else {
			log.error("Illegal Wechat OAuth2 state. expected: {}, actual: {}", wechatExtProperties.getOauthState(), state);
			return false;
		}
	}

	public void validateState(HttpServletRequest request) {
		String state = request.getParameter("state");
		if (!this.isStateValid(state)) {
			throw new IllegalStateException("Illegal Wechat OAuth2 state Exception.");
		}
	}
	
	public UserBaseInfo getUserBaseInfo(HttpServletRequest request) {
		String code = request.getParameter("code");
		if (code == null) {
			throw new IllegalArgumentException("Code value not found.");
		}
		// validate WeChat OAuth state
		this.validateState(request);
		// 
		return this.userBaseInfo(code);
	}
	
	public void redirectToOAuth2Server(HttpServletResponse response, String redirectUri, Scope scope) throws IOException  {
		// 
		String url = this.authorizeUrl(scope, redirectUri);
		log.debug("redirect URL - {}", url);
		response.sendRedirect(url);
	}
	
	public Result intercept(HttpServletRequest request, HttpServletResponse response, String redirectUri, Scope scope) throws IOException {
		String code = request.getParameter("code");
		if (code != null) { // WeChat callback
			log.debug("WeChat OAuth Server Callback.");
			return new Result(this.getUserBaseInfo(request));
		} else { // Client request. Redirecting to WeChat OAuth Server
			log.debug("Client request.");
			redirectToOAuth2Server(response, redirectUri, scope);
			return new Result(null);
		}
	}
	
	public UnifiedOrder createUnifiedOrder() {
		UnifiedOrder unifiedOrder = new UnifiedOrder();
		unifiedOrder.setAppid(wechatExtProperties.getAppId());
		unifiedOrder.setMchId(wechatExtProperties.getMchId());
		unifiedOrder.setNonceStr(CoreUtils.nextNonceString(24));
		unifiedOrder.setNotifyUrl(wechatExtProperties.getPayNotifyUrl());
		unifiedOrder.setTradeType("JSAPI");
		return unifiedOrder;
	}
	
	public <T extends HasSign> T sign(T t) {
		t.setSign(this.getSign(t));
		return t;
	}
	
	public <T extends HasSign> T checkSign(T t) {
		if (t.getSign() == null) {
			// ignore check sign when incoming bean doesn't have sign
			return t;
		}
		String sign = getSign(t);
		Assert.isTrue(Objects.equals(t.getSign(), sign), "Sign DO NOT match.");
		return t;
	}

	private <T extends HasSign> String getSign(T t) {
		TreeMap<?, ?> sortedMap = new TreeMap<>((Map<?, ?>) defaultObjectMapper().convertValue(t, Map.class));
		t.removeKeys().forEach(sortedMap::remove);
		String stringSignTemp = sortedMap.entrySet()//
			.stream()//
			.map(e -> String.format("%s=%s", e.getKey(), e.getValue()))//
			.collect(Collectors.joining("&")) + "&key=" + wechatExtProperties.getPayKey();
		return DigestUtils.md5Hex(stringSignTemp).toUpperCase();
	}

	// -----------------------------
	// ----- Get Set ToString HashCode Equals
	// -----------------------------

	public WechatExtProperties getWechatExtProperties() {
		return wechatExtProperties;
	}

	public void setWechatExtProperties(WechatExtProperties wechatExtProperties) {
		this.wechatExtProperties = wechatExtProperties;
	}

	public RestWrapper getRestWrapper() {
		return restWrapper;
	}

	public void setRestWrapper(RestWrapper restWrapper) {
		this.restWrapper = restWrapper;
	}

}
