/*
 * Decompiled with CFR 0.152.
 */
package io.joynr.messaging.routing;

import com.google.inject.name.Named;
import io.joynr.exceptions.JoynrDelayMessageException;
import io.joynr.exceptions.JoynrMessageNotSentException;
import io.joynr.exceptions.JoynrSendBufferFullException;
import io.joynr.exceptions.JoynrShutdownException;
import io.joynr.messaging.FailureAction;
import io.joynr.messaging.IMessaging;
import io.joynr.messaging.routing.MessageRouter;
import io.joynr.messaging.routing.MessagingStubFactory;
import io.joynr.messaging.routing.RoutingTable;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.annotation.CheckForNull;
import javax.inject.Inject;
import javax.inject.Singleton;
import joynr.JoynrMessage;
import joynr.system.RoutingTypes.Address;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageRouterImpl
implements MessageRouter {
    private static final long TERMINATION_TIMEOUT = 5000L;
    private Logger logger = LoggerFactory.getLogger(MessageRouterImpl.class);
    private final RoutingTable routingTable;
    private static final int UUID_TAIL = 32;
    private static final DateFormat DateFormatter = new SimpleDateFormat("dd/MM HH:mm:ss:sss");
    private ScheduledExecutorService scheduler;
    private long sendMsgRetryIntervalMs;
    private MessagingStubFactory messagingStubFactory;

    @Inject
    @Singleton
    public MessageRouterImpl(RoutingTable routingTable, @Named(value="io.joynr.messaging.scheduledthreadpool") ScheduledExecutorService scheduler, @Named(value="joynr.messaging.sendmsgretryintervalms") long sendMsgRetryIntervalMs, MessagingStubFactory messagingStubFactory) {
        this.routingTable = routingTable;
        this.scheduler = scheduler;
        this.sendMsgRetryIntervalMs = sendMsgRetryIntervalMs;
        this.messagingStubFactory = messagingStubFactory;
    }

    @Override
    public void removeNextHop(String participantId) {
        this.routingTable.remove(participantId);
    }

    @Override
    public boolean resolveNextHop(String participantId) {
        return this.routingTable.containsKey(participantId);
    }

    @Override
    public void addNextHop(String participantId, Address address) {
        this.routingTable.put(participantId, address);
    }

    @CheckForNull
    protected Address getAddress(String toParticipantId) {
        Address address = null;
        if (toParticipantId != null && this.routingTable.containsKey(toParticipantId)) {
            address = this.routingTable.get(toParticipantId);
        }
        this.logger.trace("Participant with ID {} has address {}", new Object[]{toParticipantId, address});
        return address;
    }

    @Override
    public void route(JoynrMessage message) {
        this.checkExpiry(message);
        this.routeInternal(message, 0L, 0);
    }

    protected void schedule(Runnable runnable, String messageId, long delay, TimeUnit timeUnit) {
        if (this.scheduler.isShutdown()) {
            JoynrShutdownException joynrShutdownEx = new JoynrShutdownException("MessageScheduler is shutting down already. Unable to send message [messageId: " + messageId + "].");
            throw joynrShutdownEx;
        }
        this.scheduler.schedule(runnable, delay, timeUnit);
    }

    private void routeInternal(final JoynrMessage message, long delayMs, final int retriesCount) {
        try {
            this.logger.debug("Scheduling {} with delay {} and retries {}", new Object[]{message, delayMs, retriesCount});
            this.schedule(new Runnable(){

                @Override
                public void run() {
                    MessageRouterImpl.this.logger.debug("Staring processing of message {}", (Object)message);
                    try {
                        MessageRouterImpl.this.checkExpiry(message);
                        String toParticipantId = message.getTo();
                        Address address = MessageRouterImpl.this.getAddress(toParticipantId);
                        if (address == null) {
                            throw new JoynrMessageNotSentException("Failed to send Request: No route for given participantId: " + toParticipantId);
                        }
                        String messageId = message.getId().substring(32);
                        MessageRouterImpl.this.logger.info(">>>>> SEND  ID:{}:{} from: {} to: {} header: {}", (Object[])new String[]{messageId, message.getType(), message.getHeaderValue("from"), message.getHeaderValue("to"), message.getHeader().toString()});
                        MessageRouterImpl.this.logger.debug(">>>>> body  ID:{}:{}: {}", (Object[])new String[]{messageId, message.getType(), message.getPayload()});
                        IMessaging messagingStub = MessageRouterImpl.this.messagingStubFactory.create(address);
                        messagingStub.transmit(message, MessageRouterImpl.this.createFailureAction(message, retriesCount));
                    }
                    catch (Exception error) {
                        MessageRouterImpl.this.logger.error("error in scheduled message router thread: {}", (Object)error.getMessage());
                        FailureAction failureAction = MessageRouterImpl.this.createFailureAction(message, retriesCount);
                        failureAction.execute(error);
                    }
                }
            }, message.getId(), delayMs, TimeUnit.MILLISECONDS);
        }
        catch (RejectedExecutionException e) {
            this.logger.error("Execution rejected while scheduling SendSerializedMessageRequest ", (Throwable)e);
            throw new JoynrSendBufferFullException(e);
        }
    }

    private void checkExpiry(JoynrMessage message) {
        long currentTimeMillis = System.currentTimeMillis();
        long ttlExpirationDateMs = message.getExpiryDate();
        if (ttlExpirationDateMs <= currentTimeMillis) {
            String errorMessage = MessageFormat.format("ttl must be greater than 0 / ttl timestamp must be in the future: now: {0} abs_ttl: {1}", currentTimeMillis, ttlExpirationDateMs);
            this.logger.error(errorMessage);
            throw new JoynrMessageNotSentException(errorMessage);
        }
    }

    private FailureAction createFailureAction(final JoynrMessage message, final int retriesCount) {
        FailureAction failureAction = new FailureAction(){
            final String messageId;
            {
                this.messageId = message.getId();
            }

            @Override
            public void execute(Throwable error) {
                long delayMs;
                if (error instanceof JoynrShutdownException) {
                    MessageRouterImpl.this.logger.warn("{}", (Object)error.getMessage());
                    return;
                }
                if (error instanceof JoynrMessageNotSentException) {
                    MessageRouterImpl.this.logger.error(" ERROR SENDING:  aborting send of messageId: {} to Address: {}. Error: {}", new Object[]{this.messageId, MessageRouterImpl.this.getAddress(message.getTo()), error.getMessage()});
                    return;
                }
                MessageRouterImpl.this.logger.warn("PROBLEM SENDING, will retry. messageId: {} to Address: {}. Error: {} Message: {}", new Object[]{this.messageId, MessageRouterImpl.this.getAddress(message.getTo()), error.getClass().getName(), error.getMessage()});
                if (error instanceof JoynrDelayMessageException) {
                    delayMs = ((JoynrDelayMessageException)error).getDelayMs();
                } else {
                    delayMs = MessageRouterImpl.this.sendMsgRetryIntervalMs;
                    delayMs += MessageRouterImpl.this.exponentialBackoff(delayMs, retriesCount);
                }
                try {
                    MessageRouterImpl.this.logger.error("Rescheduling messageId: {} with delay " + delayMs + " ms, new TTL expiration date: {}", (Object)this.messageId, (Object)DateFormatter.format(message.getExpiryDate()));
                    MessageRouterImpl.this.routeInternal(message, delayMs, retriesCount + 1);
                    return;
                }
                catch (JoynrSendBufferFullException e) {
                    try {
                        MessageRouterImpl.this.logger.error("Rescheduling message: {} delayed {} ms because send buffer is full", (Object)delayMs, (Object)this.messageId);
                        Thread.sleep(delayMs);
                        this.execute(e);
                    }
                    catch (InterruptedException e1) {
                        Thread.currentThread().interrupt();
                        return;
                    }
                    return;
                }
            }
        };
        return failureAction;
    }

    @Override
    public void shutdown() {
        this.scheduler.shutdown();
        try {
            this.scheduler.awaitTermination(5000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.logger.error("Message Scheduler did not shut down in time: {}", (Object)e.getMessage());
        }
    }

    private long exponentialBackoff(long delayMs, int retries) {
        this.logger.debug("TRIES: " + retries);
        long millis = delayMs + (long)((double)((long)(2 ^ retries) * delayMs) * Math.random());
        this.logger.debug("MILLIS: " + millis);
        return millis;
    }
}

