/*
 * Copyright (c) 2008-2009, Stephen Colebourne & Michael Nascimento Santos
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  * Neither the name of JSR-310 nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package javax.time.calendar;

import java.io.ObjectStreamException;
import java.io.Serializable;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.Map.Entry;

/**
 * A set of date-time fields.
 * <p>
 * Instances of this class store a map of field-value pairs.
 * Together these specify constraints on the dates and times that match.
 * For example, if an instance stores 'DayOfMonth=13' and 'DayOfWeek=Friday'
 * then it represents and matches only dates of Friday the Thirteenth.
 * <p>
 * All the values will be within the valid range for the field.
 * However, there is no cross validation between fields.
 * Thus, it is possible for the date-time represented to never exist.
 * For example, if an instance stores 'DayOfMonth=31' and 'MonthOfYear=February'
 * then there will never be a matching date.
 *
 * @author Michael Nascimento Santos
 * @author Stephen Colebourne
 */
public final class DateTimeFields
        implements CalendricalProvider,
            DateMatcher, TimeMatcher, Iterable<DateTimeFieldRule>, Serializable {

    /** Serialization version. */
    private static final long serialVersionUID = 1L;
    /** A singleton empty field set, placing no restrictions on the date-time. */
    private static final DateTimeFields EMPTY = new DateTimeFields(createMap());

    /**
     * The date time map, never null, may be empty.
     */
    private final TreeMap<DateTimeFieldRule, Integer> fieldValueMap;

    /**
     * Obtains an empty instance of <code>DateTimeFields</code>.
     * <p>
     * This factory simply returns the <code>EMPTY</code> constant.
     *
     * @return an empty fields instance, never null
     */
    public static DateTimeFields fields() {
        return EMPTY;
    }

    /**
     * Obtains an instance of <code>DateTimeFields</code> from a field-value pair.
     * <p>
     * This factory allows the creation of a fields object with a single field-value pair.
     * The value must be within the valid range for the field.
     *
     * @param fieldRule  the rule, not null
     * @param value  the field value, may be invalid
     * @return the fields instance, never null
     * @throws NullPointerException if the field rule is null
     * @throws IllegalCalendarFieldValueException if the value is invalid
     */
    public static DateTimeFields fields(DateTimeFieldRule fieldRule, int value) {
        ISOChronology.checkNotNull(fieldRule, "DateTimeFieldRule must not be null");
        fieldRule.checkValue(value);
        TreeMap<DateTimeFieldRule, Integer> map = createMap();
        map.put(fieldRule, value);
        return new DateTimeFields(map);
    }

    /**
     * Obtains an instance of <code>DateTimeFields</code> from two field-value pairs.
     * <p>
     * This factory allows the creation of a fields object with two field-value pairs.
     * Each value must be within the valid range for that field.
     * <p>
     * The two fields are not cross-validated. Thus, you can specify MonthOfYear of June
     * and DayOfMonth of 31, which is a date that can never occur.
     *
     * @param fieldRule1  the first rule, not null
     * @param value1  the first field value
     * @param fieldRule2  the second rule, not null
     * @param value2  the second field value
     * @return the fields instance, never null
     * @throws NullPointerException if either field rule is null
     * @throws IllegalCalendarFieldValueException if either value is invalid
     */
    public static DateTimeFields fields(DateTimeFieldRule fieldRule1, int value1, DateTimeFieldRule fieldRule2, int value2) {
        ISOChronology.checkNotNull(fieldRule1, "First DateTimeFieldRule must not be null");
        ISOChronology.checkNotNull(fieldRule2, "Second DateTimeFieldRule must not be null");
        fieldRule1.checkValue(value1);
        fieldRule2.checkValue(value2);
        TreeMap<DateTimeFieldRule, Integer> map = createMap();
        map.put(fieldRule1, value1);
        map.put(fieldRule2, value2);
        return new DateTimeFields(map);
    }

    /**
     * Obtains an instance of <code>DateTimeFields</code> from a map of field-value pairs.
     * <p>
     * This factory allows the creation of a fields object from a map of field-value pairs.
     * Each value must be within the valid range for that field.
     * <p>
     * The fields are not cross-validated. Thus, you can specify MonthOfYear of June
     * and DayOfMonth of 31, which is a date that can never occur.
     *
     * @param fieldValueMap  a map of fields that will be used to create a field set,
     *  not updated by this factory, not null, contains no nulls
     * @return the fields instance, never null
     * @throws NullPointerException if the map contains null keys or values
     * @throws IllegalCalendarFieldValueException if any value is invalid
     */
    public static DateTimeFields fields(Map<DateTimeFieldRule, Integer> fieldValueMap) {
        ISOChronology.checkNotNull(fieldValueMap, "Field-value map must not be null");
        if (fieldValueMap.isEmpty()) {
            return EMPTY;
        }
        // don't use contains() as tree map and others can throw NPE
        TreeMap<DateTimeFieldRule, Integer> map = createMap();
        for (Entry<DateTimeFieldRule, Integer> entry : fieldValueMap.entrySet()) {
            DateTimeFieldRule fieldRule = entry.getKey();
            Integer value = entry.getValue();
            ISOChronology.checkNotNull(fieldRule, "Null keys are not permitted in field-value map");
            ISOChronology.checkNotNull(value, "Null values are not permitted in field-value map");
            fieldRule.checkValue(value);
            map.put(fieldRule, value);
        }
        return new DateTimeFields(map);
    }

    /**
     * Creates a new empty map.
     *
     * @return ordered representation of internal map
     */
    private static TreeMap<DateTimeFieldRule, Integer> createMap() {
        return new TreeMap<DateTimeFieldRule, Integer>(Collections.reverseOrder());
    }

    //-----------------------------------------------------------------------
    /**
     * Constructor.
     *
     * @param assignedMap  the map of fields, which is assigned, not null
     */
    private DateTimeFields(TreeMap<DateTimeFieldRule, Integer> assignedMap) {
        fieldValueMap = assignedMap;
    }

    /**
     * Ensure EMPTY singleton.
     *
     * @return the resolved instance
     * @throws ObjectStreamException if an error occurs
     */
    private Object readResolve() throws ObjectStreamException {
        return fieldValueMap.isEmpty() ? EMPTY : this;
    }

    //-----------------------------------------------------------------------
    /**
     * Returns the size of the map of fields to values.
     * <p>
     * This method returns the number of field-value pairs stored.
     *
     * @return number of field-value pairs, zero or greater
     */
    public int size() {
        return fieldValueMap.size();
    }

    /**
     * Iterates through all the field rules.
     * <p>
     * This method fulfills the {@link Iterable} interface and allows looping
     * around the fields using the for-each loop. The values can be obtained using
     * {@link #get(DateTimeFieldRule)}.
     *
     * @return an iterator over the fields in this object, never null
     */
    public Iterator<DateTimeFieldRule> iterator() {
        return fieldValueMap.keySet().iterator();
    }

    /**
     * Checks if this object contains a mapping for the specified field.
     * <p>
     * This method returns true if a value can be obtained for the specified field.
     *
     * @param fieldRule  the field to query, null returns false
     * @return true if the field is supported, false otherwise
     */
    public boolean contains(DateTimeFieldRule fieldRule) {
        return fieldRule != null && fieldValueMap.containsKey(fieldRule);
    }

    //-----------------------------------------------------------------------
    /**
     * Gets the value for the specified field throwing an exception if the
     * field is not in the field-value map.
     * <p>
     * The value will be within the valid range for the field.
     * <p>
     * No attempt is made to derive values - the result is simply based on
     * the contents of the stored field-value map. If you want to derive a
     * value then convert this object to a <code>Calendrical</code> and
     * use {@link Calendrical#deriveValue(DateTimeFieldRule)}.
     *
     * @param fieldRule  the rule to query from the map, not null
     * @return the value mapped to the specified field
     * @throws UnsupportedCalendarFieldException if the field is not in the map
     */
    public int get(DateTimeFieldRule fieldRule) {
        ISOChronology.checkNotNull(fieldRule, "DateTimeFieldRule must not be null");
        Integer value = fieldValueMap.get(fieldRule);
        if (value == null) {
            throw new UnsupportedCalendarFieldException(fieldRule, "DateTimeFields");
        }
        return value;
    }

    /**
     * Gets the value for the specified field quietly returning null
     * if the field is not in the field-value map.
     * <p>
     * The value will be within the valid range for the field.
     *
     * @param fieldRule  the rule to query from the map, null returns null
     * @return the value mapped to the specified field, null if not present
     */
    public Integer getQuiet(DateTimeFieldRule fieldRule) {
        return fieldRule == null ? null : fieldValueMap.get(fieldRule);
    }

    //-----------------------------------------------------------------------
    /**
     * Returns a copy of this DateTimeFields with the specified field value.
     * <p>
     * If this instance already has a value for the field then the value is replaced.
     * Otherwise the value is added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param fieldRule  the field to set in the returned object, not null
     * @param value  the value to set in the returned set of fields
     * @return a new, updated DateTimeFields, never null
     * @throws NullPointerException if DateTimeFieldRule is null
     * @throws IllegalCalendarFieldValueException if the value is invalid
     */
    public DateTimeFields with(DateTimeFieldRule fieldRule, int value) {
        ISOChronology.checkNotNull(fieldRule, "DateTimeFieldRule must not be null");
        fieldRule.checkValue(value);
        TreeMap<DateTimeFieldRule, Integer> clonedMap = clonedMap();
        clonedMap.put(fieldRule, value);
        return new DateTimeFields(clonedMap);
    }

//    /**
//     * Returns a copy of this DateTimeFields with the fields from the specified map added.
//     * <p>
//     * If this instance already has a value for any field then the value is replaced.
//     * Otherwise the value is added.
//     * <p>
//     * This instance is immutable and unaffected by this method call.
//     *
//     * @param fieldValueMap  the new map of fields, not null
//     * @return a new, updated DateTimeFields, never null
//     * @throws NullPointerException if the map contains null keys or values
//     * @throws IllegalCalendarFieldValueException if any value is invalid
//     */
//    public DateTimeFields with(Map<DateTimeFieldRule, Integer> fieldValueMap) {
//        ISOChronology.checkNotNull(fieldValueMap, "Field-value map must not be null");
//        if (fieldValueMap.isEmpty()) {
//            return this;
//        }
//        // don't use contains() as tree map and others can throw NPE
//        TreeMap<DateTimeFieldRule, Integer> clonedMap = clonedMap();
//        for (Entry<DateTimeFieldRule, Integer> entry : fieldValueMap.entrySet()) {
//            DateTimeFieldRule fieldRule = entry.getKey();
//            Integer value = entry.getValue();
//            ISOChronology.checkNotNull(fieldRule, "Null keys are not permitted in field-value map");
//            ISOChronology.checkNotNull(value, "Null values are not permitted in field-value map");
//            fieldRule.checkValue(value);
//            clonedMap.put(fieldRule, value);
//        }
//        return new DateTimeFields(clonedMap);
//    }

    /**
     * Returns a copy of this DateTimeFields with the specified fields added.
     * <p>
     * If this instance already has a value for the field then the value is replaced.
     * Otherwise the value is added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param fields  the fields to add to the returned object, not null
     * @return a new, updated DateTimeFields, never null
     */
    public DateTimeFields with(DateTimeFields fields) {
        ISOChronology.checkNotNull(fields, "DateTimeFields must not be null");
        if (fields.size() == 0 || fields == this) {
            return this;
        }
        TreeMap<DateTimeFieldRule, Integer> clonedMap = clonedMap();
        clonedMap.putAll(fields.fieldValueMap);
        return new DateTimeFields(clonedMap);
    }

    /**
     * Returns a copy of this object with the specified field removed.
     * <p>
     * If this instance does not contain the field then the returned instance
     * is the same as this one.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param fieldRule  the field to remove from the returned object, not null
     * @return a new, updated DateTimeFields, never null
     */
    public DateTimeFields withFieldRemoved(DateTimeFieldRule fieldRule) {
        ISOChronology.checkNotNull(fieldRule, "DateTimeFieldRule must not be null");
        TreeMap<DateTimeFieldRule, Integer> clonedMap = clonedMap();
        if (clonedMap.remove(fieldRule) == null) {
            return this;
        }
        return clonedMap.isEmpty() ? EMPTY : new DateTimeFields(clonedMap);
    }

    //-----------------------------------------------------------------------
    /**
     * Checks if the date fields in this object match the specified date.
     * <p>
     * This implementation checks that all date fields in this object match the input date.
     *
     * @param date  the date to match, not null
     * @return true if the date fields match, false otherwise
     */
    public boolean matchesDate(LocalDate date) {
        ISOChronology.checkNotNull(date, "LocalDate must not be null");
        for (Entry<DateTimeFieldRule, Integer> entry : fieldValueMap.entrySet()) {
            Integer dateValue = entry.getKey().getValueQuiet(date, null);
            if (dateValue != null && dateValue.equals(entry.getValue()) == false) {
                return false;
            }
        }
        return true;
    }

    /**
     * Checks if the time fields in this object match the specified time.
     * <p>
     * This implementation checks that all time fields in this object match the input time.
     *
     * @param time  the time to match, not null
     * @return true if the time fields match, false otherwise
     */
    public boolean matchesTime(LocalTime time) {
        ISOChronology.checkNotNull(time, "LocalTime must not be null");
        for (Entry<DateTimeFieldRule, Integer> entry : fieldValueMap.entrySet()) {
            Integer timeValue = entry.getKey().getValueQuiet(null, time);
            if (timeValue != null && timeValue.equals(entry.getValue()) == false) {
                return false;
            }
        }
        return true;
    }

    //-----------------------------------------------------------------------
    /**
     * Converts this object to a map of fields to values.
     * <p>
     * The returned map will never be null, however it may be empty.
     * It is independent of this object - changes will not be reflected back.
     *
     * @return an independent, modifiable copy of the field-value map, never null
     */
    public SortedMap<DateTimeFieldRule, Integer> toFieldValueMap() {
        return new TreeMap<DateTimeFieldRule, Integer>(fieldValueMap);
    }

    /**
     * Clones the field-value map.
     *
     * @return a clone of the field-value map, never null
     */
    private TreeMap<DateTimeFieldRule, Integer> clonedMap() {
        TreeMap<DateTimeFieldRule, Integer> cloned = createMap();
        cloned.putAll(fieldValueMap);
        return cloned;
    }

    /**
     * Copies the field-value map into the specified map.
     *
     * @param map  the map to copy into, not null
     */
    void copyInto(Map<DateTimeFieldRule, Integer> map) {
        map.putAll(fieldValueMap);
    }

    //-----------------------------------------------------------------------
    /**
     * Converts this object to a <code>Calendrical</code> with the same set of fields.
     * <p>
     * Methods on the <code>Calendrical</code> allow conversion to and from
     * other date/time objects.
     *
     * @return the calendrical with the same set of fields, never null
     */
    public Calendrical toCalendrical() {
        return new Calendrical(this);
    }

    //-----------------------------------------------------------------------
    /**
     * Is this object equal to the specified object.
     * <p>
     * This compares the map of field-value pairs.
     *
     * @param obj  the other fields to compare to, null returns false
     * @return true if this instance is equal to the specified field set
     */
    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj instanceof DateTimeFields) {
            DateTimeFields other = (DateTimeFields) obj;
            return fieldValueMap.equals(other.fieldValueMap);
        }
        return false;
    }

    /**
     * A hash code for these fields.
     *
     * @return a suitable hash code
     */
    @Override
    public int hashCode() {
        return fieldValueMap.hashCode();
    }

    //-----------------------------------------------------------------------
    /**
     * Outputs the fields as a <code>String</code>.
     * <p>
     * The output will consist of the field-value map in standard map format.
     *
     * @return the formatted date-time string, never null
     */
    @Override
    public String toString() {
        return fieldValueMap.toString();
    }

}
