package org.apache.maven.jar;

/* ====================================================================
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements.  See the NOTICE file distributed with
 *   this work for additional information regarding copyright ownership.
 *   The ASF licenses this file to You under the Apache License, Version 2.0
 *   (the "License"); you may not use this file except in compliance with
 *   the License.  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 * ====================================================================
 */


import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import java.util.Iterator;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;


/**
 * Utility class for maven-jar-plugin. Currently only used for
 * testing the manifest created by Maven's jar:jar goal.
 *
 * @author <a href="mailto:ltheussl@apache.org">Lukas Theussl</a>
 * @version $Id: JarUtils.java 531612 2007-04-23 21:28:38Z ltheussl $
 */
public class JarUtils
{

    /** The manifest. */
    private Manifest manifest;

    /** The Map of manifest entries. */
    private Map manifestEntries;

    /** The main Attributes of the manifest. */
    private Attributes mainAttributes;

    /** The section Attributes of the manifest. */
    private Attributes sectionAttributes;


    /**
     * Extracts the manifest from the given jar file.
     * The manifest entries can then be retrieved via
     * getManifestEntries() and getMainAttributes().
     *
     * @param theJarFile The jar file to extract the manifest from.
     * @throws IOException If there is an error opening the jar file.
     */
    public void extractManifestFromJar( File theJarFile ) throws IOException
    {
        JarFile jarfile = new JarFile( theJarFile );
        this.manifest = jarfile.getManifest();
        extractManifestEntries();
    }

    /**
     * Extracts the manifest from a given file.
     * The manifest entries can then be retrieved via
     * getManifestEntries() and getMainAttributes().
     *
     * @param manifestFile The manifest file.
     * @throws IOException If there is an error opening the file.
     */
    public void extractManifestFromFile( File manifestFile ) throws IOException
    {
        InputStream fis = new FileInputStream( manifestFile );
        this.manifest = new Manifest( fis );
        extractManifestEntries();
    }

    /**
     * Sets the manifest.
     *
     * @param mf The new manifest.
     */
    public void setManifest( Manifest mf )
    {
      this.manifest = mf;
    }

    /**
     * Gets the manifest.
     *
     * @return The manifest.
     */
    public Manifest getManifest()
    {
      return manifest;
    }

    /**
     * Returns a Map of manifest entries.
     *
     * @return Map of manifest entries.
     */
    public Map getManifestEntries()
    {
        return manifestEntries;
    }

    /**
     * Returns the main attributes of the manifest as Attributes.
     *
     * @return The main attributes.
     */
    public Attributes getMainAttributes()
    {
        return mainAttributes;
    }

    /**
     * Checks if the current manifest contains a section with the given key.
     *
     * @param key The section name.
     * @return true if the manifest contains a section with the given key.
     */
    public boolean containsSection( String key )
    {
        return manifestEntries.containsKey( key );
    }

    /**
     * Returns the value of the main attribute key.
     *
     * @param key The attribute name.
     * @return The attribute value, null if the attribute is not found.
     */
    public String getMainAttribute( String key )
    {
        return mainAttributes.getValue( key );
    }

    /**
     * Checks if the current manifest contains a main attribute
     * with the given key.
     *
     * @param key The main attribute name.
     * @return true if the manifest contains a main attribute
     * with the given key.
     */
    public boolean containsMainAttribute( String key )
    {
        boolean contains = false;
        if ( getMainAttribute( key ) != null )
        {
            contains = true;
        }
        return contains;
    }

    /**
     * Returns the value of the section attribute key.
     *
     * @param key The attribute name.
     * @return The attribute value, null if the attribute is not found.
     */
    public String getSectionAttribute( String key )
    {
        return sectionAttributes.getValue( key );
    }

    /**
     * Checks if the current manifest contains a section attribute
     * with the given key.
     *
     * @param key The section attribute name.
     * @return true if the manifest contains a  section attribute
     * with the given key.
     */
    public boolean containsSectionAttribute( String key )
    {
        boolean contains = false;
        if ( getSectionAttribute( key ) != null )
        {
            contains = true;
        }
        return contains;
    }

    /**
     * Extracts the manifest entries, main attributes and section attributes.
     */
    private void extractManifestEntries()
    {
        this.manifestEntries = manifest.getEntries();
        this.mainAttributes = manifest.getMainAttributes();

        for (Iterator it = manifestEntries.keySet().iterator(); it.hasNext(); )
        {
            String entryName = (String) it.next();
            this.sectionAttributes =
                (Attributes) manifestEntries.get( entryName );
        }
    }

}
