/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.transcoding.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoder;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoderRegistry;
import net.shibboleth.idp.attribute.transcoding.TranscodingRule;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.DeprecationSupport;
import net.shibboleth.shared.primitive.DeprecationSupport.ObjectType;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;

/** Service implementation of the {@link AttributeTranscoderRegistry} interface. */
@ThreadSafe
public class AttributeTranscoderRegistryImpl extends AbstractIdentifiableInitializableComponent
        implements AttributeTranscoderRegistry, ApplicationContextAware {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(AttributeTranscoderRegistryImpl.class);
    
    /** Registry of transcoding instructions for a given "name" and type of object. */
    @Nonnull private final Map<String,Multimap<Class<?>,TranscodingRule>> transcodingRegistry;
    
    /** Registry of display name mappings associated with internal attribute IDs. */
    @Nonnull private final Map<String,Map<Locale,String>> displayNameRegistry;
    
    /** Registry of description mappings associated with internal attribute IDs. */
    @Nonnull private final Map<String,Map<Locale,String>> descriptionRegistry;
    
    /** Registry of naming functions for supported object types. */
    @Nonnull private final Map<Class<?>,Function<?,String>> namingFunctionRegistry;
    
    /** Optional factory function for building extended activation conditions. */
    @Nullable private Function<Map<String,Object>,Predicate<ProfileRequestContext>> extendedConditionFactory;
    
    /** The context used to load this bean. */
    @Nullable private ApplicationContext applicationContext;
    
    /** Constructor. */
    public AttributeTranscoderRegistryImpl() {
        transcodingRegistry = new HashMap<>();
        namingFunctionRegistry = new HashMap<>();
        displayNameRegistry = new HashMap<>();
        descriptionRegistry = new HashMap<>();
    }

    /** {@inheritDoc} */
    @Override public void setApplicationContext(@Nullable final ApplicationContext context) {
        checkSetterPreconditions();
        applicationContext = context;
    }

    /**
     * Get the context used to load this bean.
     * 
     * @return the context.
     */
    @Nullable public ApplicationContext getApplicationContext() {
        return applicationContext;
    }


    /**
     * Installs registry of naming functions mapped against the types of objects they support.
     * 
     * @param registry collection of naming functions for indexing
     */
    public void setNamingRegistry(@Nullable final Collection<NamingFunction<?>> registry) {
        checkSetterPreconditions();
        
        namingFunctionRegistry.clear();
        
        if (registry != null) {
            registry.forEach(nf -> namingFunctionRegistry.put(nf.getType(), nf.getFunction()));
        }
    }

    /**
     * Installs the transcoder mappings en masse.
     * 
     * <p>Each map connects an {@link IdPAttribute} name to the rules for transcoding to/from it.</p>
     * 
     * <p>The rules MUST contain at least:</p>
     * <ul>
     *  <li>{@link #PROP_ID} - internal attribute ID to map to/from</li>
     *  <li>{@link #PROP_TRANSCODER} - an {@link AttributeTranscoder} instance supporting the type</li>
     * </ul>
     * 
     * Transcoders will generally require particular properties in their own right to function.
     * 
     * @param mappings transcoding rulesets
     */
    public void setTranscoderRegistry(@Nonnull final Collection<TranscodingRule> mappings) {
        checkSetterPreconditions();
        Constraint.isNotNull(mappings, "Mappings cannot be null");
        
        transcodingRegistry.clear();
        
        for (final TranscodingRule mapping : mappings) {
            
            final String internalId = StringSupport.trimOrNull(mapping.get(PROP_ID, String.class));
            if (internalId != null && !IdPAttribute.isInvalidId(internalId)) {
                
                if (IdPAttribute.isDeprecatedId(internalId)) {
                    DeprecationSupport.warn(ObjectType.CONFIGURATION,
                            "TranscodingRule",
                            "TranscodingRule id with special characters (\'\"%{})", null);
                }
                
                final Predicate<?> activationCondition = buildActivationCondition(mapping.getMap());
                if (activationCondition != null) {
                    mapping.getMap().put(PROP_CONDITION, activationCondition);
                } else {
                    mapping.getMap().remove(PROP_CONDITION);
                }
                
                final Collection<AttributeTranscoder<?>> transcoders = getAttributeTranscoders(mapping);
                for (final AttributeTranscoder<?> transcoder : transcoders) {
                    assert transcoder != null;
                    addMapping(internalId, transcoder, mapping.getMap());
                }
            } else {
                log.warn("Ignoring TranscodingRule with invalid id property: {}", internalId);
            }
        }
    }
    
    /**
     * Set a factory function to call to consume a rule set and produce an optional "extended"
     * activation condition {@link Predicate}.
     * 
     * <p>This allows deployments to define extended property names that represent condition types not
     * known to this layer of the code base.</p>
     * 
     * @param factory factory function
     */
    public void setExtendedConditionFactory(
            @Nullable final Function<Map<String,Object>,Predicate<ProfileRequestContext>> factory) {
        checkSetterPreconditions();
        extendedConditionFactory = factory;
    }
    
    /** {@inheritDoc} */
    @Nonnull @NotLive @Unmodifiable public Map<Locale,String> getDisplayNames(@Nonnull final IdPAttribute attribute) {
        checkComponentActive();
        Constraint.isNotNull(attribute, "IdPAttribute cannot be null");
     
        final Map<Locale, String> map = displayNameRegistry.get(attribute.getId()); 
        if (map != null) {
            return map;
        }
        return CollectionSupport.emptyMap();
    }

    /** {@inheritDoc} */
    @Nonnull @NotLive @Unmodifiable public Map<Locale,String> getDescriptions(@Nonnull final IdPAttribute attribute) {
        checkComponentActive();
        Constraint.isNotNull(attribute, "IdPAttribute cannot be null");
        
        final Map<Locale, String> map = descriptionRegistry.get(attribute.getId()); 
        if (map != null) {
            return map;
        }
        return CollectionSupport.emptyMap();
    }
    
    /** {@inheritDoc} */
    @Nonnull @Unmodifiable @NotLive public Collection<TranscodingRule> getTranscodingRules(
            @Nonnull final IdPAttribute from, @Nonnull final Class<?> to) {
        checkComponentActive();
        Constraint.isNotNull(from, "IdPAttribute cannot be null");
        Constraint.isNotNull(to, "Target type cannot be null");
        
        final Multimap<Class<?>,TranscodingRule> propertyCollections = transcodingRegistry.get(from.getId());
        if (propertyCollections == null) {
            return CollectionSupport.emptyList();
        }
        
        final Class<?> effectiveType = getEffectiveType(to);
        if (effectiveType == null) {
            log.warn("Unsupported object type: {}", to.getClass().getName());
            return CollectionSupport.emptyList();
        }
        
        log.trace("Using rules for effective type {}", effectiveType.getName());
        
        final Collection<TranscodingRule> result = propertyCollections.get(effectiveType);
        assert result != null;
        return CollectionSupport.copyToList(result);
    }

    /** {@inheritDoc} */
    @Nonnull @Unmodifiable @NotLive public <T> Collection<TranscodingRule> getTranscodingRules(@Nonnull final T from) {
        checkComponentActive();
        Constraint.isNotNull(from, "Input object cannot be null");
        
        final Class<?> effectiveType = getEffectiveType(from.getClass());
        if (effectiveType == null) {
            log.warn("Unsupported object type: {}", from.getClass().getName());
            return CollectionSupport.emptyList();
        }
        
        log.trace("Using rules for effective type {}", effectiveType.getName());
        
        final Function<?,String> namingFunction = namingFunctionRegistry.get(effectiveType);
        
        // Don't know if we can work around this cast or not.
        @SuppressWarnings("unchecked")
        final String id = ((Function<? super T,String>) namingFunction).apply(from);
        if (id != null) {
            final Multimap<Class<?>,TranscodingRule> propertyCollections = transcodingRegistry.get(id);

            if (propertyCollections == null) {
                return CollectionSupport.emptyList();
            }
            final Collection<TranscodingRule> result = propertyCollections.get(effectiveType);
            assert result != null;
            return CollectionSupport.copyToList(result);
        }
        log.warn("Object of type {} did not have a canonical name", from.getClass().getName());
        
        return CollectionSupport.emptyList();
    }
    
    /**
     * Get the appropriate {@link AttributeTranscoder} objects to use.
     * 
     * @param rule transcoding rule
     * 
     * @return transcoders to install under a copy of each ruleset's {@link #PROP_TRANSCODER} property
     */
    @Nonnull private Collection<AttributeTranscoder<?>> getAttributeTranscoders(@Nonnull final TranscodingRule rule) {
        
        AttributeTranscoder<?> transcoder = rule.get(PROP_TRANSCODER, AttributeTranscoder.class);
        if (transcoder != null) {
            return CollectionSupport.singletonList(transcoder);
        }
        
        final String beanNames = rule.get(PROP_TRANSCODER, String.class);
        if (beanNames == null) {
            log.error("{} property is missing or of incorrect type", PROP_TRANSCODER);
            return CollectionSupport.emptyList();
        }

        final List<AttributeTranscoder<?>> transcoders = new ArrayList<>();
        
        final ApplicationContext appContext = getApplicationContext();
        if (appContext != null) {
            for (final String id : StringSupport.stringToList(beanNames, " ")) {
                try {
                    assert id != null;
                    transcoder = appContext.getBean(id, AttributeTranscoder.class);
                    transcoder.initialize();
                    transcoders.add(transcoder);
                } catch (final Exception e) {
                    log.error("Unable to locate AttributeTranscoder bean named {}", id, e);
                }
            }
        } else {
            log.error("Unable to locate AttributeTranscoder beans, ApplicationContext was null");
        }
        
        return transcoders;
    }
    
// Checkstyle: CyclomaticComplexity OFF
    /**
     * Add a mapping between an {@link IdPAttribute} name and a set of transcoding rules.
     * 
     * @param id name of the {@link IdPAttribute} to map to/from
     * @param transcoder the transcoder for this rule
     * @param ruleset transcoding rules
     */
    private void addMapping(@Nonnull @NotEmpty final String id, @Nonnull final AttributeTranscoder<?> transcoder,
            @Nonnull final Map<String,Object> ruleset) {
        
        final TranscodingRule copy = new TranscodingRule(ruleset);
        copy.getMap().put(PROP_TRANSCODER, transcoder);

        final Class<?> type = transcoder.getEncodedType();
        final String targetName = transcoder.getEncodedName(copy);
        if (targetName != null) {
            
            Boolean encoder = copy.getOrDefault(PROP_ENCODER, Boolean.class, true);
            if (encoder == null) {
                encoder = true;
            }
            Boolean decoder = copy.getOrDefault(PROP_DECODER, Boolean.class, true);
            if (decoder == null) {
                decoder = true;
            }
            if (!encoder && !decoder) {
                log.warn("Transcoding rule for {} and type {} was disabled in both directions, ignoring",
                        id, type.getName());
                return;
            }
            
            log.debug("Attribute mapping: {} {}-{} {} via {}", id, decoder ? "<" : "", encoder ? ">" : "", targetName,
                    transcoder.getClass().getSimpleName());
            
            // Install mapping back to IdPAttribute's trimmed name.
            copy.getMap().put(PROP_ID, id);

            if (encoder) {
                Multimap<Class<?>,TranscodingRule> rulesetsForIdPName = transcodingRegistry.get(id);
                if (rulesetsForIdPName == null) {
                    rulesetsForIdPName = ArrayListMultimap.create();
                    transcodingRegistry.put(id, rulesetsForIdPName);
                }
                rulesetsForIdPName.put(type, copy);
            }
            
            if (decoder) {
                Multimap<Class<?>,TranscodingRule> rulesetsForEncodedName = transcodingRegistry.get(targetName);
                if (rulesetsForEncodedName == null) {
                    rulesetsForEncodedName = ArrayListMultimap.create();
                    transcodingRegistry.put(targetName, rulesetsForEncodedName);
                }
                rulesetsForEncodedName.put(type, copy);
            }
            
            if (displayNameRegistry.containsKey(id)) {
                displayNameRegistry.get(id).putAll(copy.getDisplayNames());
            } else {
                displayNameRegistry.put(id, new HashMap<>(copy.getDisplayNames()));
            }

            if (descriptionRegistry.containsKey(id)) {
                descriptionRegistry.get(id).putAll(copy.getDescriptions());
            } else {
                descriptionRegistry.put(id, new HashMap<>(copy.getDescriptions()));
            }
            
        } else {
            log.warn("Transcoding rule for {} and type {} did not produce an encoded name", id, type.getName());
        }
    }
// Checkstyle: CyclomaticComplexity ON
    
    /**
     * Build an appropriate {@link Predicate} to use as an activation condition within the ruleset.
     * 
     * @param ruleset transcoding rules
     * 
     * @return a predicate to install under the ruleset's {@link #PROP_CONDITION}
     */
    @SuppressWarnings("unchecked")
    @Nullable private Predicate<ProfileRequestContext> buildActivationCondition(
            @Nonnull final Map<String,Object> ruleset) {
        
        Predicate<ProfileRequestContext> effectiveCondition = null;
        
        final Object baseCondition = ruleset.get(PROP_CONDITION);
        if (baseCondition instanceof Predicate) {
            effectiveCondition = (Predicate<ProfileRequestContext>) baseCondition;
        } else if (baseCondition instanceof String) {
            try {
                final ApplicationContext appContext = getApplicationContext();
                if (appContext != null) {
                    effectiveCondition = appContext.getBean((String) baseCondition, Predicate.class);
                } else {
                    log.error("Unable to locate Predicate bean named {}, ApplicationContext was null", baseCondition);
                }
            } catch (final Exception e) {
                log.error("Unable to locate Predicate bean named {}", baseCondition, e);
            }
        } else if (baseCondition != null) {
            log.error("{} property did not contain a Predicate object, ignored", PROP_CONDITION);
        }

        final Predicate<ProfileRequestContext> extendedCondition = extendedConditionFactory != null ?
                extendedConditionFactory.apply(ruleset) : null;

        if (effectiveCondition == null) {
            return extendedCondition;
        } else if (extendedCondition != null) {
            return effectiveCondition.and(extendedCondition);
        } else {
            return effectiveCondition;
        }
    }

    /**
     * Convert an input type into the appropriate type (possibly itself) to use in looking up
     * rules in the registry.
     * 
     * @param inputType the type passed into the registry operation
     * 
     * @return the appropriate type to use subsequently or null if not found
     */
    @Nullable private Class<?> getEffectiveType(@Nonnull final Class<?> inputType) {
        
        // Check for explicit support.
        if (namingFunctionRegistry.containsKey(inputType)) {
            return inputType;
        }
        
        // Try each map entry for a match. Optimized around the assumption the
        // map will be fairly small.
        for (final Class<?> candidate : namingFunctionRegistry.keySet()) {
            if (candidate.isAssignableFrom(inputType)) {
                return candidate;
            }
        }
        
        return null;
    }
    
}