/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.httpclient.factory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.hc.client5.http.auth.UsernamePasswordCredentials;
import org.springframework.beans.factory.BeanCreationException;

import net.shibboleth.shared.spring.factory.AbstractComponentAwareFactoryBean;

/**
 * Factory bean for {@link UsernamePasswordCredentials}.
 * 
 * <p>Mainly for insulation from Apache client API.</p>
 * 
 * @since 9.0.0
 */
public class UsernamePasswoordCredentialsFactoryBean
        extends AbstractComponentAwareFactoryBean<UsernamePasswordCredentials> {

    /** The username. */
    @Nullable private String username;

    /** The password. */
    @Nullable private String password;

    /** {@inheritDoc} */
    @Override
    @Nonnull public Class<?> getObjectType() {
        return UsernamePasswordCredentials.class;
    }

    /**
     * Set the username.
     * 
     * @param what value to set
     */
    public void setUsername(@Nullable final String what) {
        username = what;
    }

    /**
     * Set the password.
     * 
     * @param what value to set
     */
    public void setPassword(@Nullable final String what) {
        password = what;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull protected UsernamePasswordCredentials doCreateInstance() throws Exception {
        if (username == null || password == null) {
            throw new BeanCreationException("Username and password cannot be null");
        }
        
        assert password != null;
        return new UsernamePasswordCredentials(username, password.toCharArray());
    }

}