/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.httpclient.resource;

import java.io.IOException;
import java.io.InputStream;

import javax.annotation.Nonnull;

import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.HttpResponse;

/**
 * A wrapper around the entity content {@link InputStream} represented by an {@link HttpResponse}
 * that closes the stream and the HttpResponse when {@link #close()} is invoked.
 */
 public class ConnectionClosingInputStream extends InputStream {

    /** HTTP response that is being wrapped. */
    @Nonnull private final ClassicHttpResponse response;

    /** Stream owned by the given HTTP response. */
    @Nonnull private final InputStream stream;

    /**
     * Constructor.
     *
     * @param httpResponse HTTP method that was invoked
     * @throws IOException if there is a problem getting the entity content input stream from the response
     */
    public ConnectionClosingInputStream(@Nonnull final ClassicHttpResponse httpResponse) throws IOException {
        response = httpResponse;
        final InputStream str = response.getEntity().getContent();
        assert str!=null;
        stream = str;
    }

    /** {@inheritDoc} */
    @Override
    public int available() throws IOException {
        return stream.available();
    }

    /** {@inheritDoc} */
    @Override
    public void close() throws IOException {
        stream.close();
        response.close();
    }

    /** {@inheritDoc} */
    @Override
    public synchronized void mark(final int readLimit) {
        stream.mark(readLimit);
    }

    /** {@inheritDoc} */
    @Override
    public boolean markSupported() {
        return stream.markSupported();
    }

    /** {@inheritDoc} */
    @Override
    public int read() throws IOException {
        return stream.read();
    }

    /** {@inheritDoc} */
    @Override
    public int read(final byte[] b) throws IOException {
        return stream.read(b);
    }

    /** {@inheritDoc} */
    @Override
    public int read(final byte[] b, final int off, final int len) throws IOException {
        return stream.read(b, off, len);
    }

    /** {@inheritDoc} */
    @Override
    public synchronized void reset() throws IOException {
        stream.reset();
    }

    /** {@inheritDoc} */
    @Override
    public long skip(final long n) throws IOException {
        return stream.skip(n);
    }
}