/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.servlet;

import org.springframework.core.Ordered;

import jakarta.servlet.Filter;

/**
 * Marker interface for filters to be automatically installed via implementation-specific means.
 */
public interface ChainableFilter extends Filter, Ordered {

    /**
     * An abstraction for modeling filter order.
     */
    enum FilterOrder {
    
        /** Run as early as possible. */
        EARLIEST(Integer.MIN_VALUE),
        
        /** Run early but not the earliest. */
        EARLY(Integer.MIN_VALUE + 1000),
        
        /** Run whenever. */
        NEUTRAL(0),
        
        /** Run later but not the latest. */
        LATE(Integer.MAX_VALUE - 1000),
        
        /** Run as late as possible. */
        LATEST(Integer.MAX_VALUE);
    
        /** Store integer value. */
        private final int intValue;
        
        /**
         * Constructor.
         *
         * @param value value to use
         */
        private FilterOrder(final int value) {
            intValue = value;
        }
        
        /**
         * Get the integral equivalent of the enum.
         * 
         * @return integral value
         */
        public int getValue() {
            return intValue;
        }
    }

    /** {@inheritDoc} */
    default int getOrder() {
        return FilterOrder.NEUTRAL.getValue();
    }

}