/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.servlet.impl;

import java.io.IOException;
import java.util.function.Predicate;

import javax.annotation.Nonnull;

import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.logic.PredicateSupport;
import net.shibboleth.shared.spring.servlet.ChainableFilter;

/**
 * Adapts Spring filter into a chainable/conditional one.
 */
public class CharacterEncodingFilter extends org.springframework.web.filter.CharacterEncodingFilter
        implements ChainableFilter {

    /** Whether filter should run or not. */
    @Nonnull private Predicate<ServletRequest> activationCondition;

    /** Constructor. */
    public CharacterEncodingFilter() {
        activationCondition = PredicateSupport.alwaysTrue();
    }

    /**
     * Get the condition to control activation of this filter.
     * 
     * @return condition
     */
    @Nonnull public Predicate<ServletRequest> getActivationCondition() {
        return activationCondition;
    }
    
    /**
     * Set the condition to control activation of this filter.
     * 
     * @param condition run condition
     */
    public void setActivationCondition(@Nonnull final Predicate<ServletRequest> condition) {
        activationCondition = Constraint.isNotNull(condition, "Run condition cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    public int getOrder() {
        return FilterOrder.NEUTRAL.getValue();
    }

    /** {@inheritDoc} */
    @Override
    protected void doFilterInternal(@Nonnull final HttpServletRequest request,
            @Nonnull final HttpServletResponse response, @Nonnull final FilterChain filterChain)
                    throws ServletException, IOException {
        
        if (!activationCondition.test(request)) {
            filterChain.doFilter(request, response);
            return;
        }
        
        super.doFilterInternal(request, response, filterChain);
    }

}