/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.servlet.impl;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import jakarta.servlet.GenericServlet;
import jakarta.servlet.Servlet;
import jakarta.servlet.ServletContext;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.StringSupport;

import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
import org.springframework.web.filter.DelegatingFilterProxy;

/**
 * Proxy for a standard Servlet, delegating to a Spring-managed bean that
 * implements the Servlet interface. Supports a "targetBeanName" init-param
 * in {@code web.xml}, specifying the name of the target bean in the Spring
 * application context.
 *
 * <p>This class was originally inspired by Spring's {@link DelegatingFilterProxy}
 * class and has the same general behavior.</p>
 */
public class DelegatingServletProxy extends GenericServlet {

    /** For serialization. */
    private static final long serialVersionUID = 5470308159110258401L;

    /** The name of the ServletContext attribute which should be used to retrieve the
     * {@link WebApplicationContext} from which to load the delegate {@link Servlet} bean.
     */
    @Nullable @NotEmpty private String contextAttribute;

    /** The application context from which the target filter will be retrieved. */
    @Nullable private WebApplicationContext webApplicationContext;

    /** The name of the target bean in the Spring application context. */
    @Nullable @NotEmpty private String targetBeanName;

    /** Whether to invoke the Servlet lifecycle methods on the target bean. */
    private boolean targetFilterLifecycle;

    /** The {@code HttpServlet} instance that this proxy will delegate to and
     * manage the lifecycle for.
     */
    @Nullable private volatile Servlet delegate;

    /** Object used to syncrhonize access where needed. */
    @Nonnull private Object delegateMonitor;

    /**
     * Constructor for traditional use in {@code web.xml}.
     */
    public DelegatingServletProxy() {
        delegateMonitor = new Object();
    }

    /**
     * Constructor using the given {@link Servlet} delegate.
     * 
     * <p>Bypasses entirely the need for interacting with a Spring application context,
     * specifying the {@linkplain #setTargetBeanName target bean name}, etc.</p>
     * 
     * <p>For use with instance-based registration of filters.</p>
     * 
     * @param del the {@code HttpServlet} instance that this proxy will delegate to and
     * manage the lifecycle for
     */
    public DelegatingServletProxy(@Nonnull final Servlet del) {
        delegateMonitor = new Object();
        delegate = Constraint.isNotNull(del, "Delegate servlet cannot be null");
    }

    /**
     * Constructor that will retrieve the named target
     * bean from the Spring {@code WebApplicationContext} found in the {@code ServletContext}
     * (either the 'root' application context or the context named by
     * {@link #setContextAttribute}).
     * 
     * <p>For use with instance-based registration of filters.</p>
     * 
     * <p>The target bean must implement the standard {@link Servlet} interface.</p>
     * 
     * @param targetBeanName name of the target servlet bean to look up in the Spring
     * application context
     */
    public DelegatingServletProxy(@Nonnull @NotEmpty final String targetBeanName) {
        this(targetBeanName, null);
    }

    /**
     * Constructor that will retrieve the named target bean from the given Spring {@code WebApplicationContext}.
     * 
     * <p>For use with instance-based registration of filters.</p>
     * 
     * <p>The target bean must implement the standard {@link Servlet} interface.</p>
     * 
     * <p>The given {@code WebApplicationContext} may or may not be refreshed when passed
     * in. If it has not, and if the context implements {@link ConfigurableApplicationContext},
     * a {@link ConfigurableApplicationContext#refresh() refresh()} will be attempted before
     * retrieving the named target bean.</p>
     * 
     * @param targetBeanName name of the target filter bean in the Spring application context
     * @param wac the application context from which the target filter will be retrieved;
     *     if {@code null}, an application context will be looked up from {@code ServletContext}
     *     as a fallback
     */
    public DelegatingServletProxy(@Nonnull @NotEmpty final String targetBeanName,
            @Nullable final WebApplicationContext wac) {
        delegateMonitor = new Object();
        
        setTargetBeanName(targetBeanName);
        webApplicationContext = wac;
    }

    /**
     * Set the name of the ServletContext attribute which should be used to retrieve the
     * {@link WebApplicationContext} from which to load the delegate {@link Servlet} bean.
     * 
     * @param name of attribute
     */
    public void setContextAttribute(@Nullable @NotEmpty final String name) {
        contextAttribute = StringSupport.trimOrNull(name);
    }

    /**
     * Return the name of the ServletContext attribute which should be used to retrieve the
     * {@link WebApplicationContext} from which to load the delegate {@link Servlet} bean.
     * 
     * @return name of attribute
     */
    @Nullable @NotEmpty public String getContextAttribute() {
        return contextAttribute;
    }

    /**
     * Set the name of the target bean in the Spring application context.
     * 
     * <p>The target bean must implement the standard {@link Servlet} interface.</p>
     * 
     * <p>By default, the {@code servlet-name} as specified for the
     * DelegatingFilterProxy in {@code web.xml} will be used.</p>
     * 
     * @param name bean name
     */
    public void setTargetBeanName(@Nullable @NotEmpty final String name) {
        targetBeanName = StringSupport.trimOrNull(name);
    }

    /**
     * Return the name of the target bean in the Spring application context.
     * 
     * @return bean name
     */
    @Nullable @NotEmpty protected String getTargetBeanName() {
        return targetBeanName;
    }

    /**
     * Set whether to invoke the {@link Servlet#init(jakarta.servlet.ServletConfig)} and
     * {@link Servlet#destroy()} lifecycle methods on the target bean.
     * 
     * <p>Default is "false"; target beans usually rely on the Spring application
     * context for managing their lifecycle. Setting this flag to "true" means
     * that the servlet container will control the lifecycle of the target
     * bean, with this proxy delegating the corresponding calls.</p>
     * 
     * @param flag flag to set
     */
    public void setTargetFilterLifecycle(final boolean flag) {
        targetFilterLifecycle = flag;
    }

    /**
     * Return whether to invoke the Servlet lifecycle methods on the target bean.
     * 
     * @return whether to invoke the Servlet lifecycle methods on the target bean
     */
    protected boolean isTargetFilterLifecycle() {
        return targetFilterLifecycle;
    }

    /** {@inheritDoc} */
    @Override
    public void init() throws ServletException {
        synchronized (delegateMonitor) {
            if (delegate == null) {
                // If no target bean name specified, use servlet name.
                if (targetBeanName == null) {
                    targetBeanName = getServletName();
                }
                
                // Fetch Spring root application context and initialize the delegate early,
                // if possible. If the root application context will be started after this
                // filter proxy, we'll have to resort to lazy initialization.
                final WebApplicationContext wac = findWebApplicationContext();
                if (wac != null) {
                    delegate = initDelegate(wac);
                }
            }
        }
    }

    /** {@inheritDoc} */
    @Override
    public void service(final ServletRequest request, final ServletResponse response)
            throws ServletException, IOException {

        // Lazily initialize the delegate if necessary.
        Servlet delegateToUse = delegate;
        if (delegateToUse == null) {
            synchronized (delegateMonitor) {
                delegateToUse = delegate;
                if (delegateToUse == null) {
                    final WebApplicationContext wac = findWebApplicationContext();
                    if (wac == null) {
                        throw new IllegalStateException("No WebApplicationContext found: " +
                                "no ContextLoaderListener or DispatcherServlet registered?");
                    }
                    delegateToUse = initDelegate(wac);
                }
                delegate = delegateToUse;
            }
        }

        // Let the delegate perform the actual service operation.
        invokeDelegate(delegateToUse, request, response);
    }

    /** {@inheritDoc} */
    @Override
    public void destroy() {
        final Servlet delegateToUse = delegate;
        if (delegateToUse != null) {
            destroyDelegate(delegateToUse);
        }
    }


    /**
     * Return the {@link WebApplicationContext} passed in at construction time, if available.
     * 
     * <p>Otherwise, attempt to retrieve a {@link WebApplicationContext} from the
     * {@link ServletContext} attribute with the {@linkplain #setContextAttribute
     * configured name} if set. Otherwise look up a {@link WebApplicationContext} under
     * the well-known "root" application context attribute.</p>
     * 
     * <p>The {@link WebApplicationContext} must have already been loaded and stored in the
     * {@link ServletContext} before this filter gets initialized (or invoked).</p>
     * 
     * <p>Subclasses may override this method to provide a different
     * {@link WebApplicationContext} retrieval strategy.</p>
     * 
     * @return the {@link WebApplicationContext} for this proxy, or {@code null} if not found
     */
    @Nullable protected WebApplicationContext findWebApplicationContext() {
        if (webApplicationContext != null) {
            // The user has injected a context at construction time -> use it...
            if (webApplicationContext instanceof ConfigurableApplicationContext cac && !cac.isActive()) {
                // The context has not yet been refreshed -> do so before returning it...
                cac.refresh();
            }
            return webApplicationContext;
        }
        final String attrName = getContextAttribute();
        if (attrName != null) {
            return WebApplicationContextUtils.getWebApplicationContext(getServletContext(), attrName);
        }
        else {
            return WebApplicationContextUtils.findWebApplicationContext(getServletContext());
        }
    }

    /**
     * Initialize the Servlet delegate, defined as a bean in the given Spring application context.
     * 
     * <p>The default implementation fetches the bean from the application context
     * and calls the standard {@code Servlet.init} method on it, passing
     * in the ServletConfig of this Servlet proxy.</p>
     * 
     * @param wac the root application context
     * 
     * @return the initialized delegate Filter
     * 
     * @throws ServletException if thrown by the servlet or if no bean name can be found
     */
    @Nonnull protected Servlet initDelegate(@Nonnull final WebApplicationContext wac) throws ServletException {
        final String targetBean = getTargetBeanName();
        if (targetBean == null) {
            throw new ServletException("No target bean name set.");
        }
        final Servlet del = wac.getBean(targetBean, Servlet.class);
        if (isTargetFilterLifecycle()) {
            del.init(getServletConfig());
        }
        return del;
    }

    /**
     * Actually invoke the delegate Servlet with the given request and response.
     * 
     * @param del the delegate Servlet
     * @param request the current request
     * @param response the current response
     * 
     * @throws ServletException if thrown by the Servlet
     * @throws IOException if thrown by the Servlet
     */
    protected void invokeDelegate(@Nonnull final Servlet del, @Nonnull final ServletRequest request,
            @Nonnull final ServletResponse response) throws ServletException, IOException {

        del.service(request, response);
    }

    /**
     * Destroy the Servlet delegate.
     * 
     * @param del the Servlet delegate
     */
    protected void destroyDelegate(@Nonnull final Servlet del) {
        if (isTargetFilterLifecycle()) {
            del.destroy();
        }
    }

}