/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.factory;

import java.io.InputStream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.scripting.EvaluableScript;

import org.slf4j.Logger;

import org.springframework.beans.factory.BeanCreationException;
import org.springframework.core.io.Resource;

/**
 * A factory bean to summon up an {@link EvaluableScript} from either inline data or from a resource.
 */
public class EvaluableScriptFactoryBean extends AbstractComponentAwareFactoryBean<EvaluableScript> {

    /** log. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(EvaluableScript.class);

    /** The resource which locates the script. */
    @Nullable private Resource resource;

    /** The script. */
    @Nullable private String script;

    /** The JSR223 engine name. */
    @Nullable private String engineName;

    /** The source Id. */
    @Nullable private String sourceId;

    /**
     * Get the resource which locates the script.
     * 
     * @return Returns the resource.
     */
    @Nullable public Resource getResource() {
        return resource;
    }

    /**
     * Set the resource which locates the script.
     * 
     * @param what the resource to set.
     */
    public void setResource(@Nullable final Resource what) {
        resource = what;
    }

    /**
     * Get the script.
     * 
     * @return Returns the script as text.
     */
    @Nullable public String getScript() {
        return script;
    }

    /**
     * Set the script.
     * 
     * @param what the script to set.
     */
    public void setScript(@Nullable final String what) {
        script = what;
    }

    /**
     * Get the source ID.
     * 
     * @return Returns the sourceID.
     */
    @Nullable public String getSourceId() {
        return sourceId;
    }

    /**
     * Set the source Id.
     * 
     * @param what the Id to set.
     */
    public void setSourceId(@Nullable final String what) {
        sourceId = what;
    }

    /**
     * Get the engine Name.
     * 
     * @return Returns the engine name.
     */
    @Nullable public String getEngineName() {
        return engineName;
    }

    /**
     * Set the engine name.
     * 
     * @param what the engine name to set.
     */
    public void setEngineName(@Nullable final String what) {
        engineName = what;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public Class<?> getObjectType() {
        return EvaluableScript.class;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull protected EvaluableScript doCreateInstance() throws Exception {

        if (null == script && null == resource) {
            log.error("{} A script or a resource must be supplied", sourceId);
            throw new BeanCreationException("A script or a resource must be supplied");
        }
        if (null != script && null != resource) {
            log.error("{} Only one of script or resource should be supplied", sourceId);
            throw new BeanCreationException("Only one of script or resource should be supplied");
        }

        final String toExecute;
        
        if (null != resource) {
            try (final InputStream is = resource.getInputStream()) {
                toExecute = StringSupport.inputStreamToString(is, null);
            }
        } else {
            toExecute = script;
        }

        log.debug("{} Language: {} Script: {} ", sourceId, engineName==null ? "<default>" : engineName, script);

        if (toExecute != null) {
            final EvaluableScript evaluableScript = new EvaluableScript();
            evaluableScript.setScript(toExecute);
    
            if (engineName != null) {
                evaluableScript.setEngineName(engineName);
            }
            //
            // Initialize for compatibility reasons
            //
            evaluableScript.initialize();
            return evaluableScript;
        }
        
        throw new BeanCreationException("Unable to load script");
    }

}