/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.factory;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.servlet.ViewResolver;

import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * Class used for auto-wiring {@link ViewResolver} beans for Spring Web Flow.
 * 
 * @since 9.1.0
 */
public class ViewResolversFactoryBean extends AbstractFactoryBean<List<ViewResolver>> {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ViewResolversFactoryBean.class);

    /** Underlying collection. */
    @Nonnull private List<ViewResolver> resolvers;
    
    /**
     * Auto-wiring point for resolvers.
     * 
     * @param freeObjects free-standing objects
     */
    @Autowired
    public ViewResolversFactoryBean(@Nullable final List<ViewResolver> freeObjects) {
        if (freeObjects != null) {
            resolvers = CollectionSupport.copyToList(freeObjects);
        } else {
            resolvers = CollectionSupport.emptyList();
        }
        
        log.debug("Registered ViewResolvers: {}",
                resolvers.stream().map(Object::getClass).map(Class::getName).collect(Collectors.toUnmodifiableList()));
    }
    
    /** {@inheritDoc} */
    @Override
    @SuppressWarnings("rawtypes")
    @Nullable public Class<List> getObjectType() {
        return List.class;
    }

    /** {@inheritDoc} */
    @Override
    @Nullable protected List<ViewResolver> createInstance() throws Exception {
        return CollectionSupport.copyToList(resolvers);
    }
    
}