/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.testing;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.sql.DataSource;

import org.hsqldb.jdbc.JDBCDataSource;
import org.slf4j.Logger;

import com.google.common.io.CharStreams;

import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * Helper class for testing database-backed classes using custom DDL and SQL.
 */
public final class DatabaseTestingSupport {

    /** Logger. */
    @Nonnull private static Logger log = LoggerFactory.getLogger(DatabaseTestingSupport.class);
    
    /** Hidden constructor. */
    private DatabaseTestingSupport() {}
    
    /**
     * Initialize SQL database.
     * 
     * @param initializingSQLFile path to SQL to run at init time
     * @param source the data source
     */
    public static void InitializeDataSource(final @Nullable String initializingSQLFile,
            @Nonnull final DataSource source) {

        final String sql = ReadSqlFromFile(initializingSQLFile);
        if (sql == null) {
            return;
        }
        ExecuteUpdate(sql, source);
    }

    /**
     * Read some SQL from a file.
     * 
     * @param initializingSQLFile the file name
     * @return the data
     */
    @Nullable protected static String ReadSqlFromFile(final @Nullable String initializingSQLFile) {

        final String file = StringSupport.trimOrNull(initializingSQLFile);

        if (null == file) {
            return null;
        }

        final InputStream is = DatabaseTestingSupport.class.getResourceAsStream(file);

        if (null == is) {
            log.warn("Could not locate SQL file called {} ", file);
            return null;
        }
        final String sql;
        try {
            sql = StringSupport.trimOrNull(CharStreams.toString(new InputStreamReader(is)));
        } catch (final IOException e) {
            log.warn("Could not read SQL file called {}.", file);
            return null;
        }

        if (null == sql) {
            log.warn("SQL file called {} was empty.", file);
            return null;
        }

        return sql;
    }

    /**
     * Execute an SQP update.
     * 
     * @param sql what to execute
     * @param source what to executer it on
     */
    protected static void ExecuteUpdate(final @Nullable String sql, final DataSource source) {

        log.debug("Applying SQL: \n {}", sql);

        try {
            final Connection dbConn = source.getConnection();
            final Statement statement = dbConn.createStatement();

            statement.executeUpdate(sql);
        } catch (final SQLException e) {
            log.warn("Could not contact data source {} or execute commands", source, e);
            return;
        }
    }

    /**
     * Summons up an in memory database with the provided identifier. The contents of the resource stream (if any) are
     * then submitted to the database (so as to allow initializing to a known state.
     * 
     * @param initializingSQLFile a file in the classpath with SQL files
     * @param identifier a name to uniquify this database.
     * @return a DataSource which can then be used for testing.
     */
    @Nonnull public static DataSource GetMockDataSource(final @Nonnull String initializingSQLFile,
            final @Nonnull String identifier) {
        return GetDataSourceFromUrl(initializingSQLFile, "jdbc:hsqldb:mem:" + identifier);
    }

    /**
     * Summons up a database connection to  an hsqldb server running somewhere.
     * @param initializingSQLFile a file in the classpath with SQL files
     * @param server the server name and database name.  For instance "//localhost/testdb"
     * @return a DataSource which can then be used for testing
     */
    @Nonnull public static DataSource GetDataSourceFromHsqlServer(final @Nonnull String initializingSQLFile,
            final @Nonnull String server) {
        return GetDataSourceFromUrl(initializingSQLFile, "jdbc:hsqldb:hsql:" + server);
    }

    /**
     * Initialize SQL database.
     * 
     * @param sqlFile path to file containing multiple SQL statements separated by semicolons.
     * @param source data source
     */
    public static void InitializeDataSourceFromFile(final String sqlFile, final DataSource source) {
        final String sql = ReadSqlFromFile(sqlFile);
        assert sql != null;
        final String[] statements = sql.split(";");
        for (final String statement : statements) {
            ExecuteUpdate(statement.trim(), source);
        }
    }

    /**
     * Createa data source from a provided URL.
     * 
     * @param initializingSQLFile what to initialize with
     * @param JdbcUri the URI
     * 
     * @return the {@link DataSource}
     */
    @Nonnull protected static DataSource GetDataSourceFromUrl(@Nonnull final String initializingSQLFile,
            @Nonnull final String JdbcUri) {
        final JDBCDataSource jdbcSource = new JDBCDataSource();

        jdbcSource.setUrl(JdbcUri);
        jdbcSource.setUser("SA");
        jdbcSource.setPassword("");

        InitializeDataSource(initializingSQLFile, jdbcSource);

        return jdbcSource;
    }

}
