/*
 * Decompiled with CFR 0.152.
 */
package alluxio;

import alluxio.Client;
import alluxio.Configuration;
import alluxio.PropertyKey;
import alluxio.RuntimeConstants;
import alluxio.exception.AlluxioException;
import alluxio.exception.ConnectionFailedException;
import alluxio.exception.ExceptionMessage;
import alluxio.exception.PreconditionMessage;
import alluxio.retry.ExponentialBackoffRetry;
import alluxio.security.authentication.TransportProvider;
import alluxio.thrift.AlluxioService;
import alluxio.thrift.AlluxioTException;
import alluxio.thrift.ThriftIOException;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketTimeoutException;
import java.util.regex.Pattern;
import javax.annotation.concurrent.ThreadSafe;
import javax.security.auth.Subject;
import org.apache.thrift.TException;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TMultiplexedProtocol;
import org.apache.thrift.protocol.TProtocol;
import org.apache.thrift.transport.TTransportException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public abstract class AbstractClient
implements Client {
    private static final Logger LOG = LoggerFactory.getLogger((String)"alluxio.logger.type");
    private static final Pattern FRAME_SIZE_EXCEPTION_PATTERN = Pattern.compile("Frame size \\((\\d+)\\) larger than max length");
    protected static final int RPC_MAX_NUM_RETRY = 30;
    protected final String mMode;
    protected InetSocketAddress mAddress = null;
    protected TProtocol mProtocol = null;
    protected boolean mConnected = false;
    protected boolean mClosed = false;
    protected long mServiceVersion;
    protected final TransportProvider mTransportProvider;
    private final Subject mParentSubject;

    public AbstractClient(Subject subject, InetSocketAddress address, String mode) {
        this.mAddress = (InetSocketAddress)Preconditions.checkNotNull((Object)address);
        this.mMode = mode;
        this.mServiceVersion = -1L;
        this.mTransportProvider = TransportProvider.Factory.create();
        this.mParentSubject = subject;
    }

    protected abstract AlluxioService.Client getClient();

    protected abstract String getServiceName();

    protected abstract long getServiceVersion();

    protected void checkVersion(AlluxioService.Client client, long version) throws IOException {
        if (this.mServiceVersion == -1L) {
            try {
                this.mServiceVersion = client.getServiceVersion();
            }
            catch (TException e) {
                throw new IOException(e);
            }
            if (this.mServiceVersion != version) {
                throw new IOException(ExceptionMessage.INCOMPATIBLE_VERSION.getMessage(this.getServiceName(), version, this.mServiceVersion));
            }
        }
    }

    protected void afterConnect() throws IOException {
    }

    protected void afterDisconnect() {
    }

    protected void beforeDisconnect() {
    }

    @Override
    public synchronized void connect() throws IOException, ConnectionFailedException {
        if (this.mConnected) {
            return;
        }
        this.disconnect();
        Preconditions.checkState((!this.mClosed ? 1 : 0) != 0, (Object)"Client is closed, will not try to connect.");
        int maxConnectsTry = Configuration.getInt(PropertyKey.MASTER_RETRY);
        int BASE_SLEEP_MS = 50;
        ExponentialBackoffRetry retry = new ExponentialBackoffRetry(50, 1000, maxConnectsTry);
        while (!this.mClosed) {
            this.mAddress = this.getAddress();
            LOG.info("Alluxio client (version {}) is trying to connect with {} {} @ {}", new Object[]{RuntimeConstants.VERSION, this.getServiceName(), this.mMode, this.mAddress});
            TBinaryProtocol binaryProtocol = new TBinaryProtocol(this.mTransportProvider.getClientTransport(this.mParentSubject, this.mAddress));
            this.mProtocol = new TMultiplexedProtocol((TProtocol)binaryProtocol, this.getServiceName());
            try {
                this.mProtocol.getTransport().open();
                LOG.info("Client registered with {} {} @ {}", new Object[]{this.getServiceName(), this.mMode, this.mAddress});
                this.mConnected = true;
                this.afterConnect();
                this.checkVersion(this.getClient(), this.getServiceVersion());
                return;
            }
            catch (IOException e) {
                if (e.getMessage() != null && FRAME_SIZE_EXCEPTION_PATTERN.matcher(e.getMessage()).find()) {
                    String message = String.format("Failed to connect to %s %s @ %s: %s. This exception may be caused by incorrect network configuration. Please consult %s for common solutions to address this problem.", this.getServiceName(), this.mMode, this.mAddress, e.getMessage(), RuntimeConstants.ALLUXIO_DEBUG_DOCS_URL);
                    throw new IOException(message, e);
                }
                throw e;
            }
            catch (TTransportException e) {
                LOG.error("Failed to connect (" + retry.getRetryCount() + ") to " + this.getServiceName() + " " + this.mMode + " @ " + this.mAddress + " : " + e.getMessage());
                if (e.getCause() instanceof SocketTimeoutException) {
                    String message = "Thrift transport open times out. Please check whether the authentication types match between client and server. Note that NOSASL client is not able to connect to servers with SIMPLE security mode.";
                    throw new IOException(message, e);
                }
                if (retry.attemptRetry()) continue;
                break;
            }
        }
        throw new ConnectionFailedException("Failed to connect to " + this.getServiceName() + " " + this.mMode + " @ " + this.mAddress + " after " + retry.getRetryCount() + " attempts");
    }

    public synchronized void disconnect() {
        if (this.mConnected) {
            Preconditions.checkNotNull((Object)this.mProtocol, (Object)((Object)PreconditionMessage.PROTOCOL_NULL_WHEN_CONNECTED));
            LOG.debug("Disconnecting from the {} {} {}", new Object[]{this.getServiceName(), this.mMode, this.mAddress});
            this.beforeDisconnect();
            this.mProtocol.getTransport().close();
            this.mConnected = false;
            this.afterDisconnect();
        }
    }

    @Override
    public synchronized boolean isConnected() {
        return this.mConnected;
    }

    @Override
    public synchronized void close() {
        this.disconnect();
        this.mClosed = true;
    }

    @Override
    public synchronized void resetConnection() {
        this.disconnect();
        this.mAddress = this.getAddress();
    }

    protected synchronized InetSocketAddress getAddress() {
        return this.mAddress;
    }

    protected synchronized <V> V retryRPC(RpcCallable<V> rpc) throws IOException, ConnectionFailedException {
        int retry = 0;
        while (!this.mClosed && retry++ <= 30) {
            this.connect();
            try {
                return rpc.call();
            }
            catch (ThriftIOException e) {
                throw new IOException((Throwable)((Object)e));
            }
            catch (AlluxioTException e) {
                throw Throwables.propagate((Throwable)AlluxioException.fromThrift(e));
            }
            catch (TException e) {
                LOG.error(e.getMessage(), (Throwable)e);
                this.disconnect();
            }
        }
        throw new IOException("Failed after " + retry + " retries.");
    }

    protected synchronized <V> V retryRPC(RpcCallableThrowsAlluxioTException<V> rpc) throws AlluxioException, IOException {
        int retry = 0;
        while (!this.mClosed && retry++ <= 30) {
            this.connect();
            try {
                return rpc.call();
            }
            catch (AlluxioTException e) {
                throw AlluxioException.fromThrift(e);
            }
            catch (ThriftIOException e) {
                throw new IOException((Throwable)((Object)e));
            }
            catch (TException e) {
                LOG.error(e.getMessage(), (Throwable)e);
                this.disconnect();
            }
        }
        throw new IOException("Failed after " + retry + " retries.");
    }

    protected static interface RpcCallableThrowsAlluxioTException<V> {
        public V call() throws AlluxioTException, TException;
    }

    protected static interface RpcCallable<V> {
        public V call() throws TException;
    }
}

