/*
 * Decompiled with CFR 0.152.
 */
package alluxio.util.network;

import alluxio.AlluxioURI;
import alluxio.Configuration;
import alluxio.MasterInquireClient;
import alluxio.PropertyKey;
import alluxio.exception.PreconditionMessage;
import alluxio.util.OSUtils;
import alluxio.wire.WorkerNetAddress;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.thrift.transport.TServerSocket;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class NetworkAddressUtils {
    public static final String WILDCARD_ADDRESS = "0.0.0.0";
    public static final boolean WINDOWS = OSUtils.isWindows();
    private static final Logger LOG = LoggerFactory.getLogger((String)"alluxio.logger.type");
    private static String sLocalHost;
    private static String sLocalIP;

    private NetworkAddressUtils() {
    }

    public static void assertValidPort(int port) {
        Preconditions.checkArgument((port < 65536 ? 1 : 0) != 0, (Object)"Port must be less than 65536");
        Preconditions.checkArgument((port >= 0 ? 1 : 0) != 0, (Object)"Port must be non-negative");
    }

    public static void assertValidPort(InetSocketAddress address) {
        NetworkAddressUtils.assertValidPort(address.getPort());
    }

    public static InetSocketAddress getConnectAddress(ServiceType service) {
        return new InetSocketAddress(NetworkAddressUtils.getConnectHost(service), NetworkAddressUtils.getPort(service));
    }

    public static String getConnectHost(ServiceType service) {
        String bindHost;
        String connectHost;
        if (Configuration.containsKey(service.mHostNameKey) && !(connectHost = Configuration.get(service.mHostNameKey)).isEmpty() && !connectHost.equals(WILDCARD_ADDRESS)) {
            return connectHost;
        }
        if (Configuration.containsKey(service.mBindHostKey) && !(bindHost = Configuration.get(service.mBindHostKey)).isEmpty() && !bindHost.equals(WILDCARD_ADDRESS)) {
            return bindHost;
        }
        return NetworkAddressUtils.getLocalHostName();
    }

    public static int getPort(ServiceType service) {
        return Configuration.getInt(service.mPortKey);
    }

    public static InetSocketAddress getBindAddress(ServiceType service) {
        int port = NetworkAddressUtils.getPort(service);
        NetworkAddressUtils.assertValidPort(port);
        String host = Configuration.containsKey(service.mBindHostKey) && !Configuration.get(service.mBindHostKey).isEmpty() ? Configuration.get(service.mBindHostKey) : NetworkAddressUtils.getLocalHostName();
        return new InetSocketAddress(host, port);
    }

    public static synchronized String getLocalHostName() {
        if (sLocalHost != null) {
            return sLocalHost;
        }
        int hostResolutionTimeout = Configuration.getInt(PropertyKey.NETWORK_HOST_RESOLUTION_TIMEOUT_MS);
        return NetworkAddressUtils.getLocalHostName(hostResolutionTimeout);
    }

    public static synchronized String getLocalHostName(int timeoutMs) {
        if (sLocalHost != null) {
            return sLocalHost;
        }
        try {
            sLocalHost = InetAddress.getByName(NetworkAddressUtils.getLocalIpAddress(timeoutMs)).getCanonicalHostName();
            return sLocalHost;
        }
        catch (UnknownHostException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static synchronized String getLocalIpAddress() {
        if (sLocalIP != null) {
            return sLocalIP;
        }
        int hostResolutionTimeout = Configuration.getInt(PropertyKey.NETWORK_HOST_RESOLUTION_TIMEOUT_MS);
        return NetworkAddressUtils.getLocalIpAddress(hostResolutionTimeout);
    }

    public static synchronized String getLocalIpAddress(int timeoutMs) {
        if (sLocalIP != null) {
            return sLocalIP;
        }
        try {
            InetAddress address = InetAddress.getLocalHost();
            LOG.debug("address: {} isLoopbackAddress: {}, with host {} {}", new Object[]{address, address.isLoopbackAddress(), address.getHostAddress(), address.getHostName()});
            if (!NetworkAddressUtils.isValidAddress(address, timeoutMs)) {
                Enumeration<NetworkInterface> networkInterfaces = NetworkInterface.getNetworkInterfaces();
                if (!WINDOWS) {
                    ArrayList<NetworkInterface> netIFs = Collections.list(networkInterfaces);
                    Collections.reverse(netIFs);
                    networkInterfaces = Collections.enumeration(netIFs);
                }
                while (networkInterfaces.hasMoreElements()) {
                    NetworkInterface ni = networkInterfaces.nextElement();
                    Enumeration<InetAddress> addresses = ni.getInetAddresses();
                    while (addresses.hasMoreElements()) {
                        address = addresses.nextElement();
                        if (!NetworkAddressUtils.isValidAddress(address, timeoutMs)) continue;
                        sLocalIP = address.getHostAddress();
                        return sLocalIP;
                    }
                }
                LOG.warn("Your hostname, " + InetAddress.getLocalHost().getHostName() + " resolves to" + " a loopback/non-reachable address: " + address.getHostAddress() + ", but we couldn't find any external IP address!");
            }
            sLocalIP = address.getHostAddress();
            return sLocalIP;
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static boolean isServing(String host, int port) {
        if (port < 0) {
            return false;
        }
        try {
            Socket socket = new Socket(host, port);
            socket.close();
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    private static boolean isValidAddress(InetAddress address, int timeoutMs) throws IOException {
        return !address.isAnyLocalAddress() && !address.isLinkLocalAddress() && !address.isLoopbackAddress() && address.isReachable(timeoutMs) && address instanceof Inet4Address;
    }

    public static AlluxioURI replaceHostName(AlluxioURI path) throws UnknownHostException {
        if (path == null) {
            return null;
        }
        if (path.hasAuthority()) {
            String authority = NetworkAddressUtils.resolveHostName(path.getHost());
            if (path.getPort() != -1) {
                authority = authority + ":" + path.getPort();
            }
            return new AlluxioURI(path.getScheme(), authority, path.getPath(), path.getQueryMap());
        }
        return path;
    }

    public static String resolveHostName(String hostname) throws UnknownHostException {
        if (hostname == null || hostname.isEmpty()) {
            return null;
        }
        return InetAddress.getByName(hostname).getCanonicalHostName();
    }

    public static String getFqdnHost(InetSocketAddress addr) {
        Preconditions.checkNotNull((Object)addr.getAddress(), (Object)("the address of " + addr + " is invalid."));
        return addr.getAddress().getCanonicalHostName();
    }

    public static String getFqdnHost(WorkerNetAddress addr) throws UnknownHostException {
        return NetworkAddressUtils.resolveHostName(addr.getHost());
    }

    public static int getThriftPort(TServerSocket thriftSocket) {
        return NetworkAddressUtils.getThriftSocket(thriftSocket).getLocalPort();
    }

    public static ServerSocket getThriftSocket(TServerSocket thriftSocket) {
        try {
            Field field = TServerSocket.class.getDeclaredField("serverSocket_");
            field.setAccessible(true);
            return (ServerSocket)field.get(thriftSocket);
        }
        catch (IllegalAccessException | NoSuchFieldException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static InetSocketAddress parseInetSocketAddress(String address) throws IOException {
        if (address == null) {
            return null;
        }
        String[] strArr = address.split(":");
        if (strArr.length != 2) {
            throw new IOException("Invalid InetSocketAddress " + address);
        }
        return new InetSocketAddress(strArr[0], Integer.parseInt(strArr[1]));
    }

    public static InetSocketAddress getRpcPortSocketAddress(WorkerNetAddress netAddress) {
        try {
            String host = NetworkAddressUtils.getFqdnHost(netAddress);
            int port = netAddress.getRpcPort();
            return new InetSocketAddress(host, port);
        }
        catch (UnknownHostException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static InetSocketAddress getDataPortSocketAddress(WorkerNetAddress netAddress) {
        try {
            String host = NetworkAddressUtils.getFqdnHost(netAddress);
            int port = netAddress.getDataPort();
            return new InetSocketAddress(host, port);
        }
        catch (UnknownHostException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static InetSocketAddress getLeaderAddressFromZK(String zkLeaderPath) {
        Preconditions.checkState((boolean)Configuration.containsKey(PropertyKey.ZOOKEEPER_ADDRESS), (String)PreconditionMessage.ERR_ZK_ADDRESS_NOT_SET.toString(), (Object[])new Object[]{PropertyKey.ZOOKEEPER_ADDRESS.toString()});
        Preconditions.checkState((boolean)Configuration.containsKey(PropertyKey.ZOOKEEPER_ELECTION_PATH), (Object)PropertyKey.ZOOKEEPER_ELECTION_PATH.toString());
        MasterInquireClient masterInquireClient = MasterInquireClient.getClient(Configuration.get(PropertyKey.ZOOKEEPER_ADDRESS), Configuration.get(PropertyKey.ZOOKEEPER_ELECTION_PATH), zkLeaderPath);
        try {
            String temp = masterInquireClient.getLeaderAddress();
            return NetworkAddressUtils.parseInetSocketAddress(temp);
        }
        catch (IOException e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static List<InetSocketAddress> getMasterAddressesFromZK() {
        Preconditions.checkState((boolean)Configuration.containsKey(PropertyKey.ZOOKEEPER_ADDRESS));
        Preconditions.checkState((boolean)Configuration.containsKey(PropertyKey.ZOOKEEPER_ELECTION_PATH));
        Preconditions.checkState((boolean)Configuration.containsKey(PropertyKey.ZOOKEEPER_LEADER_PATH));
        MasterInquireClient masterInquireClient = MasterInquireClient.getClient(Configuration.get(PropertyKey.ZOOKEEPER_ADDRESS), Configuration.get(PropertyKey.ZOOKEEPER_ELECTION_PATH), Configuration.get(PropertyKey.ZOOKEEPER_LEADER_PATH));
        List<String> addresses = masterInquireClient.getMasterAddresses();
        if (addresses == null) {
            throw new RuntimeException(String.format("Failed to get the master addresses from zookeeper, zookeeper address: %s", Configuration.get(PropertyKey.ZOOKEEPER_ADDRESS)));
        }
        ArrayList<InetSocketAddress> ret = new ArrayList<InetSocketAddress>(addresses.size());
        try {
            for (String address : addresses) {
                ret.add(NetworkAddressUtils.parseInetSocketAddress(address));
            }
            return ret;
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static enum ServiceType {
        MASTER_RPC("Alluxio Master RPC service", PropertyKey.MASTER_HOSTNAME, PropertyKey.MASTER_BIND_HOST, PropertyKey.MASTER_RPC_PORT),
        MASTER_WEB("Alluxio Master Web service", PropertyKey.MASTER_WEB_HOSTNAME, PropertyKey.MASTER_WEB_BIND_HOST, PropertyKey.MASTER_WEB_PORT),
        WORKER_RPC("Alluxio Worker RPC service", PropertyKey.WORKER_HOSTNAME, PropertyKey.WORKER_BIND_HOST, PropertyKey.WORKER_RPC_PORT),
        WORKER_DATA("Alluxio Worker data service", PropertyKey.WORKER_DATA_HOSTNAME, PropertyKey.WORKER_DATA_BIND_HOST, PropertyKey.WORKER_DATA_PORT),
        WORKER_WEB("Alluxio Worker Web service", PropertyKey.WORKER_WEB_HOSTNAME, PropertyKey.WORKER_WEB_BIND_HOST, PropertyKey.WORKER_WEB_PORT),
        PROXY_WEB("Alluxio Proxy Web service", PropertyKey.PROXY_WEB_HOSTNAME, PropertyKey.PROXY_WEB_BIND_HOST, PropertyKey.PROXY_WEB_PORT);

        private final String mServiceName;
        private final PropertyKey mHostNameKey;
        private final PropertyKey mBindHostKey;
        private final PropertyKey mPortKey;

        private ServiceType(String serviceName, PropertyKey hostNameKey, PropertyKey bindHostKey, PropertyKey portKey) {
            this.mServiceName = serviceName;
            this.mHostNameKey = hostNameKey;
            this.mBindHostKey = bindHostKey;
            this.mPortKey = portKey;
        }

        public String getServiceName() {
            return this.mServiceName;
        }

        public PropertyKey getHostNameKey() {
            return this.mHostNameKey;
        }

        public PropertyKey getBindHostKey() {
            return this.mBindHostKey;
        }

        public PropertyKey getPortKey() {
            return this.mPortKey;
        }

        public int getDefaultPort() {
            return Integer.parseInt(this.mPortKey.getDefaultValue());
        }
    }
}

