/*
 * Decompiled with CFR 0.152.
 */
package org.apache.archiva.metadata.repository.jcr;

import com.google.common.collect.ImmutableMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import javax.jcr.Credentials;
import javax.jcr.InvalidItemStateException;
import javax.jcr.NamespaceRegistry;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.jcr.ValueFactory;
import javax.jcr.Workspace;
import javax.jcr.nodetype.NodeTypeDefinition;
import javax.jcr.nodetype.NodeTypeManager;
import javax.jcr.nodetype.NodeTypeTemplate;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;
import javax.jcr.query.Row;
import javax.jcr.query.RowIterator;
import org.apache.archiva.metadata.model.ArtifactMetadata;
import org.apache.archiva.metadata.model.CiManagement;
import org.apache.archiva.metadata.model.Dependency;
import org.apache.archiva.metadata.model.IssueManagement;
import org.apache.archiva.metadata.model.License;
import org.apache.archiva.metadata.model.MailingList;
import org.apache.archiva.metadata.model.MetadataFacet;
import org.apache.archiva.metadata.model.MetadataFacetFactory;
import org.apache.archiva.metadata.model.Organization;
import org.apache.archiva.metadata.model.ProjectMetadata;
import org.apache.archiva.metadata.model.ProjectVersionMetadata;
import org.apache.archiva.metadata.model.ProjectVersionReference;
import org.apache.archiva.metadata.model.Scm;
import org.apache.archiva.metadata.repository.MetadataRepository;
import org.apache.archiva.metadata.repository.MetadataRepositoryException;
import org.apache.archiva.metadata.repository.MetadataResolutionException;
import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.commons.JcrUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JcrMetadataRepository
implements MetadataRepository {
    private static final String JCR_LAST_MODIFIED = "jcr:lastModified";
    static final String NAMESPACE_NODE_TYPE = "archiva:namespace";
    static final String PROJECT_NODE_TYPE = "archiva:project";
    static final String PROJECT_VERSION_NODE_TYPE = "archiva:projectVersion";
    static final String ARTIFACT_NODE_TYPE = "archiva:artifact";
    static final String FACET_NODE_TYPE = "archiva:facet";
    private static final String DEPENDENCY_NODE_TYPE = "archiva:dependency";
    private final Map<String, MetadataFacetFactory> metadataFacetFactories;
    private Logger log = LoggerFactory.getLogger(JcrMetadataRepository.class);
    private Repository repository;
    private Session jcrSession;

    public JcrMetadataRepository(Map<String, MetadataFacetFactory> metadataFacetFactories, Repository repository) throws RepositoryException {
        this.metadataFacetFactories = metadataFacetFactories;
        this.repository = repository;
    }

    static void initialize(Session session) throws RepositoryException {
        Workspace workspace = session.getWorkspace();
        NamespaceRegistry registry = workspace.getNamespaceRegistry();
        if (!Arrays.asList(registry.getPrefixes()).contains("archiva")) {
            registry.registerNamespace("archiva", "http://archiva.apache.org/jcr/");
        }
        NodeTypeManager nodeTypeManager = workspace.getNodeTypeManager();
        JcrMetadataRepository.registerMixinNodeType(nodeTypeManager, NAMESPACE_NODE_TYPE);
        JcrMetadataRepository.registerMixinNodeType(nodeTypeManager, PROJECT_NODE_TYPE);
        JcrMetadataRepository.registerMixinNodeType(nodeTypeManager, PROJECT_VERSION_NODE_TYPE);
        JcrMetadataRepository.registerMixinNodeType(nodeTypeManager, ARTIFACT_NODE_TYPE);
        JcrMetadataRepository.registerMixinNodeType(nodeTypeManager, FACET_NODE_TYPE);
        JcrMetadataRepository.registerMixinNodeType(nodeTypeManager, DEPENDENCY_NODE_TYPE);
    }

    private static void registerMixinNodeType(NodeTypeManager nodeTypeManager, String name) throws RepositoryException {
        NodeTypeTemplate nodeType = nodeTypeManager.createNodeTypeTemplate();
        nodeType.setMixin(true);
        nodeType.setName(name);
        if (!nodeTypeManager.hasNodeType(name)) {
            nodeTypeManager.registerNodeType((NodeTypeDefinition)nodeType, false);
        }
    }

    public void updateProject(String repositoryId, ProjectMetadata project) throws MetadataRepositoryException {
        this.updateProject(repositoryId, project.getNamespace(), project.getId());
    }

    private void updateProject(String repositoryId, String namespace, String projectId) throws MetadataRepositoryException {
        this.updateNamespace(repositoryId, namespace);
        try {
            this.getOrAddProjectNode(repositoryId, namespace, projectId);
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void updateArtifact(String repositoryId, String namespace, String projectId, String projectVersion, ArtifactMetadata artifactMeta) throws MetadataRepositoryException {
        this.updateNamespace(repositoryId, namespace);
        try {
            Node node = this.getOrAddArtifactNode(repositoryId, namespace, projectId, projectVersion, artifactMeta.getId());
            Calendar cal = Calendar.getInstance();
            cal.setTime(artifactMeta.getFileLastModified());
            node.setProperty(JCR_LAST_MODIFIED, cal);
            cal = Calendar.getInstance();
            cal.setTime(artifactMeta.getWhenGathered());
            node.setProperty("whenGathered", cal);
            node.setProperty("size", artifactMeta.getSize());
            node.setProperty("md5", artifactMeta.getMd5());
            node.setProperty("sha1", artifactMeta.getSha1());
            node.setProperty("version", artifactMeta.getVersion());
            for (String facetId : this.metadataFacetFactories.keySet()) {
                MetadataFacet metadataFacet = artifactMeta.getFacet(facetId);
                if (metadataFacet == null) continue;
                if (node.hasNode(facetId)) {
                    node.getNode(facetId).remove();
                }
                if (metadataFacet == null) continue;
                Node n = node.addNode(facetId);
                n.addMixin(FACET_NODE_TYPE);
                for (Map.Entry entry : metadataFacet.toProperties().entrySet()) {
                    n.setProperty((String)entry.getKey(), (String)entry.getValue());
                }
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void updateProjectVersion(String repositoryId, String namespace, String projectId, ProjectVersionMetadata versionMetadata) throws MetadataRepositoryException {
        this.updateProject(repositoryId, namespace, projectId);
        try {
            Node versionNode = this.getOrAddProjectVersionNode(repositoryId, namespace, projectId, versionMetadata.getId());
            versionNode.setProperty("name", versionMetadata.getName());
            versionNode.setProperty("description", versionMetadata.getDescription());
            versionNode.setProperty("url", versionMetadata.getUrl());
            versionNode.setProperty("incomplete", versionMetadata.isIncomplete());
            if (versionMetadata.getScm() != null) {
                versionNode.setProperty("scm.connection", versionMetadata.getScm().getConnection());
                versionNode.setProperty("scm.developerConnection", versionMetadata.getScm().getDeveloperConnection());
                versionNode.setProperty("scm.url", versionMetadata.getScm().getUrl());
            }
            if (versionMetadata.getCiManagement() != null) {
                versionNode.setProperty("ci.system", versionMetadata.getCiManagement().getSystem());
                versionNode.setProperty("ci.url", versionMetadata.getCiManagement().getUrl());
            }
            if (versionMetadata.getIssueManagement() != null) {
                versionNode.setProperty("issue.system", versionMetadata.getIssueManagement().getSystem());
                versionNode.setProperty("issue.url", versionMetadata.getIssueManagement().getUrl());
            }
            if (versionMetadata.getOrganization() != null) {
                versionNode.setProperty("org.name", versionMetadata.getOrganization().getName());
                versionNode.setProperty("org.url", versionMetadata.getOrganization().getUrl());
            }
            int i = 0;
            for (License license : versionMetadata.getLicenses()) {
                versionNode.setProperty("license." + i + ".name", license.getName());
                versionNode.setProperty("license." + i + ".url", license.getUrl());
                ++i;
            }
            i = 0;
            for (Object mailingList : versionMetadata.getMailingLists()) {
                versionNode.setProperty("mailingList." + i + ".archive", mailingList.getMainArchiveUrl());
                versionNode.setProperty("mailingList." + i + ".name", mailingList.getName());
                versionNode.setProperty("mailingList." + i + ".post", mailingList.getPostAddress());
                versionNode.setProperty("mailingList." + i + ".unsubscribe", mailingList.getUnsubscribeAddress());
                versionNode.setProperty("mailingList." + i + ".subscribe", mailingList.getSubscribeAddress());
                versionNode.setProperty("mailingList." + i + ".otherArchives", this.join(mailingList.getOtherArchives()));
                ++i;
            }
            if (!versionMetadata.getDependencies().isEmpty()) {
                Node dependenciesNode = JcrUtils.getOrAddNode((Node)versionNode, (String)"dependencies");
                for (Dependency dependency : versionMetadata.getDependencies()) {
                    String id = dependency.getGroupId() + ";" + dependency.getArtifactId() + "-" + dependency.getVersion();
                    if (dependency.getClassifier() != null) {
                        id = id + "-" + dependency.getClassifier();
                    }
                    id = id + "." + dependency.getType();
                    Node n = JcrUtils.getOrAddNode((Node)dependenciesNode, (String)id);
                    n.addMixin(DEPENDENCY_NODE_TYPE);
                    n.setProperty("groupId", dependency.getGroupId());
                    n.setProperty("artifactId", dependency.getArtifactId());
                    n.setProperty("version", dependency.getVersion());
                    n.setProperty("type", dependency.getType());
                    n.setProperty("classifier", dependency.getClassifier());
                    n.setProperty("scope", dependency.getScope());
                    n.setProperty("systemPath", dependency.getSystemPath());
                    n.setProperty("optional", dependency.isOptional());
                }
            }
            for (MetadataFacet facet : versionMetadata.getFacetList()) {
                if (versionNode.hasNode(facet.getFacetId())) {
                    versionNode.getNode(facet.getFacetId()).remove();
                }
                Node n = versionNode.addNode(facet.getFacetId());
                n.addMixin(FACET_NODE_TYPE);
                for (Map.Entry entry : facet.toProperties().entrySet()) {
                    n.setProperty((String)entry.getKey(), (String)entry.getValue());
                }
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void updateNamespace(String repositoryId, String namespace) throws MetadataRepositoryException {
        try {
            Node node = this.getOrAddNamespaceNode(repositoryId, namespace);
            node.setProperty("namespace", namespace);
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void removeProject(String repositoryId, String namespace, String projectId) throws MetadataRepositoryException {
        try {
            Node root = this.getJcrSession().getRootNode();
            String namespacePath = JcrMetadataRepository.getNamespacePath(repositoryId, namespace);
            if (root.hasNode(namespacePath)) {
                for (Node node : JcrUtils.getChildNodes((Node)root.getNode(namespacePath))) {
                    if (!node.isNodeType(PROJECT_NODE_TYPE) || !projectId.equals(node.getName())) continue;
                    node.remove();
                }
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public boolean hasMetadataFacet(String repositoryId, String facetId) throws MetadataRepositoryException {
        try {
            Node node = this.getJcrSession().getRootNode().getNode(JcrMetadataRepository.getFacetPath(repositoryId, facetId));
            return node.getNodes().hasNext();
        }
        catch (PathNotFoundException e) {
            return false;
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public List<String> getMetadataFacets(String repositoryId, String facetId) throws MetadataRepositoryException {
        ArrayList<String> facets = new ArrayList<String>();
        try {
            Node node = this.getJcrSession().getRootNode().getNode(JcrMetadataRepository.getFacetPath(repositoryId, facetId));
            this.recurse(facets, "", node);
        }
        catch (PathNotFoundException node) {
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
        return facets;
    }

    private void recurse(List<String> facets, String prefix, Node node) throws RepositoryException {
        for (Node n : JcrUtils.getChildNodes((Node)node)) {
            String name = prefix + "/" + n.getName();
            if (n.hasNodes()) {
                this.recurse(facets, name, n);
                continue;
            }
            facets.add(name.substring(1));
        }
    }

    public MetadataFacet getMetadataFacet(String repositoryId, String facetId, String name) throws MetadataRepositoryException {
        MetadataFacet metadataFacet = null;
        try {
            Node root = this.getJcrSession().getRootNode();
            Node node = root.getNode(JcrMetadataRepository.getFacetPath(repositoryId, facetId, name));
            if (this.metadataFacetFactories == null) {
                return metadataFacet;
            }
            MetadataFacetFactory metadataFacetFactory = this.metadataFacetFactories.get(facetId);
            if (metadataFacetFactory != null) {
                metadataFacet = metadataFacetFactory.createMetadataFacet(repositoryId, name);
                HashMap<String, String> map = new HashMap<String, String>();
                for (Property property : JcrUtils.getProperties((Node)node)) {
                    String p = property.getName();
                    if (p.startsWith("jcr:")) continue;
                    map.put(p, property.getString());
                }
                metadataFacet.fromProperties(map);
            }
        }
        catch (PathNotFoundException root) {
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
        return metadataFacet;
    }

    public void addMetadataFacet(String repositoryId, MetadataFacet metadataFacet) throws MetadataRepositoryException {
        try {
            Node repo = this.getOrAddRepositoryNode(repositoryId);
            Node facets = JcrUtils.getOrAddNode((Node)repo, (String)"facets");
            String id = metadataFacet.getFacetId();
            Node facetNode = JcrUtils.getOrAddNode((Node)facets, (String)id);
            Node node = this.getOrAddNodeByPath(facetNode, metadataFacet.getName());
            for (Map.Entry entry : metadataFacet.toProperties().entrySet()) {
                node.setProperty((String)entry.getKey(), (String)entry.getValue());
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void removeNamespace(String repositoryId, String projectId) throws MetadataRepositoryException {
        try {
            Node node;
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getNamespacePath(repositoryId, projectId);
            if (root.hasNode(path) && (node = root.getNode(path)).isNodeType(NAMESPACE_NODE_TYPE)) {
                node.remove();
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void removeMetadataFacets(String repositoryId, String facetId) throws MetadataRepositoryException {
        try {
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getFacetPath(repositoryId, facetId);
            if (root.hasNode(path)) {
                root.getNode(path).remove();
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void removeMetadataFacet(String repositoryId, String facetId, String name) throws MetadataRepositoryException {
        try {
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getFacetPath(repositoryId, facetId, name);
            if (root.hasNode(path)) {
                Node parent;
                Node node = root.getNode(path);
                do {
                    parent = node.getParent();
                    node.remove();
                } while (!(node = parent).hasNodes());
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public List<ArtifactMetadata> getArtifactsByDateRange(String repoId, Date startTime, Date endTime) throws MetadataRepositoryException {
        ArrayList<ArtifactMetadata> artifacts;
        String q = JcrMetadataRepository.getArtifactQuery(repoId);
        if (startTime != null) {
            q = q + " AND [whenGathered] >= $start";
        }
        if (endTime != null) {
            q = q + " AND [whenGathered] <= $end";
        }
        try {
            Query query = this.getJcrSession().getWorkspace().getQueryManager().createQuery(q, "JCR-SQL2");
            ValueFactory valueFactory = this.getJcrSession().getValueFactory();
            if (startTime != null) {
                query.bindValue("start", valueFactory.createValue(JcrMetadataRepository.createCalendar(startTime)));
            }
            if (endTime != null) {
                query.bindValue("end", valueFactory.createValue(JcrMetadataRepository.createCalendar(endTime)));
            }
            QueryResult result = query.execute();
            artifacts = new ArrayList<ArtifactMetadata>();
            for (Node n : JcrUtils.getNodes((QueryResult)result)) {
                artifacts.add(this.getArtifactFromNode(repoId, n));
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
        return artifacts;
    }

    public Collection<String> getRepositories() throws MetadataRepositoryException {
        List<String> repositories;
        try {
            Node root = this.getJcrSession().getRootNode();
            if (root.hasNode("repositories")) {
                Node node = root.getNode("repositories");
                repositories = new ArrayList<String>();
                NodeIterator i = node.getNodes();
                while (i.hasNext()) {
                    Node n = i.nextNode();
                    repositories.add(n.getName());
                }
            } else {
                repositories = Collections.emptyList();
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
        return repositories;
    }

    public List<ArtifactMetadata> getArtifactsByChecksum(String repositoryId, String checksum) throws MetadataRepositoryException {
        ArrayList<ArtifactMetadata> artifacts;
        String q = JcrMetadataRepository.getArtifactQuery(repositoryId) + " AND ([sha1] = $checksum OR [md5] = $checksum)";
        try {
            Query query = this.getJcrSession().getWorkspace().getQueryManager().createQuery(q, "JCR-SQL2");
            ValueFactory valueFactory = this.getJcrSession().getValueFactory();
            query.bindValue("checksum", valueFactory.createValue(checksum));
            QueryResult result = query.execute();
            artifacts = new ArrayList<ArtifactMetadata>();
            for (Node n : JcrUtils.getNodes((QueryResult)result)) {
                artifacts.add(this.getArtifactFromNode(repositoryId, n));
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
        return artifacts;
    }

    private List<ArtifactMetadata> runJcrQuery(String repositoryId, String q, Map<String, String> bindings) throws MetadataRepositoryException {
        ArrayList<ArtifactMetadata> artifacts;
        if (repositoryId != null) {
            q = q + " AND ISDESCENDANTNODE(artifact,'/" + JcrMetadataRepository.getRepositoryContentPath(repositoryId) + "')";
        }
        this.log.info("Running JCR Query: {}", (Object)q);
        try {
            Query query = this.getJcrSession().getWorkspace().getQueryManager().createQuery(q, "JCR-SQL2");
            ValueFactory valueFactory = this.getJcrSession().getValueFactory();
            for (Map.Entry<String, String> entry : bindings.entrySet()) {
                query.bindValue(entry.getKey(), valueFactory.createValue(entry.getValue()));
            }
            long start = Calendar.getInstance().getTimeInMillis();
            QueryResult result = query.execute();
            long end = Calendar.getInstance().getTimeInMillis();
            this.log.info("JCR Query ran in {} milliseconds: {}", (Object)(end - start), (Object)q);
            artifacts = new ArrayList<ArtifactMetadata>();
            RowIterator rows = result.getRows();
            while (rows.hasNext()) {
                Row row = rows.nextRow();
                Node node = row.getNode("artifact");
                artifacts.add(this.getArtifactFromNode(repositoryId, node));
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
        return artifacts;
    }

    public List<ArtifactMetadata> getArtifactsByProjectVersionMetadata(String key, String value, String repositoryId) throws MetadataRepositoryException {
        String q = "SELECT * FROM [archiva:projectVersion] AS projectVersion INNER JOIN [archiva:artifact] AS artifact ON ISCHILDNODE(artifact, projectVersion) INNER JOIN [archiva:facet] AS facet ON ISCHILDNODE(facet, projectVersion) WHERE ([facet].[" + key + "] = $value)";
        return this.runJcrQuery(repositoryId, q, (Map<String, String>)ImmutableMap.of((Object)"value", (Object)value));
    }

    public List<ArtifactMetadata> getArtifactsByMetadata(String key, String value, String repositoryId) throws MetadataRepositoryException {
        String q = "SELECT * FROM [archiva:artifact] AS artifact INNER JOIN [archiva:facet] AS facet ON ISCHILDNODE(facet, artifact) WHERE ([facet].[" + key + "] = $value)";
        return this.runJcrQuery(repositoryId, q, (Map<String, String>)ImmutableMap.of((Object)"value", (Object)value));
    }

    public List<ArtifactMetadata> getArtifactsByProperty(String key, String value, String repositoryId) throws MetadataRepositoryException {
        String q = "SELECT * FROM [archiva:projectVersion] AS projectVersion INNER JOIN [archiva:artifact] AS artifact ON ISCHILDNODE(artifact, projectVersion) WHERE ([projectVersion].[" + key + "] = $value)";
        return this.runJcrQuery(repositoryId, q, (Map<String, String>)ImmutableMap.of((Object)"value", (Object)value));
    }

    public void removeRepository(String repositoryId) throws MetadataRepositoryException {
        try {
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getRepositoryPath(repositoryId);
            if (root.hasNode(path)) {
                root.getNode(path).remove();
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public List<ArtifactMetadata> getArtifacts(String repositoryId) throws MetadataRepositoryException {
        ArrayList<ArtifactMetadata> artifacts;
        String q = JcrMetadataRepository.getArtifactQuery(repositoryId);
        try {
            Query query = this.getJcrSession().getWorkspace().getQueryManager().createQuery(q, "JCR-SQL2");
            QueryResult result = query.execute();
            artifacts = new ArrayList<ArtifactMetadata>();
            for (Node n : JcrUtils.getNodes((QueryResult)result)) {
                if (!n.isNodeType(ARTIFACT_NODE_TYPE)) continue;
                artifacts.add(this.getArtifactFromNode(repositoryId, n));
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
        return artifacts;
    }

    private static String getArtifactQuery(String repositoryId) {
        return "SELECT * FROM [archiva:artifact] AS artifact WHERE ISDESCENDANTNODE(artifact,'/" + JcrMetadataRepository.getRepositoryContentPath(repositoryId) + "')";
    }

    public ProjectMetadata getProject(String repositoryId, String namespace, String projectId) throws MetadataResolutionException {
        ProjectMetadata metadata = null;
        try {
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getProjectPath(repositoryId, namespace, projectId);
            if (root.hasNode(path)) {
                metadata = new ProjectMetadata();
                metadata.setId(projectId);
                metadata.setNamespace(namespace);
            }
        }
        catch (RepositoryException e) {
            throw new MetadataResolutionException(e.getMessage(), (Throwable)e);
        }
        return metadata;
    }

    public ProjectVersionMetadata getProjectVersion(String repositoryId, String namespace, String projectId, String projectVersion) throws MetadataResolutionException {
        ProjectVersionMetadata versionMetadata;
        try {
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getProjectVersionPath(repositoryId, namespace, projectId, projectVersion);
            if (!root.hasNode(path)) {
                return null;
            }
            Node node = root.getNode(path);
            versionMetadata = new ProjectVersionMetadata();
            versionMetadata.setId(projectVersion);
            versionMetadata.setName(JcrMetadataRepository.getPropertyString(node, "name"));
            versionMetadata.setDescription(JcrMetadataRepository.getPropertyString(node, "description"));
            versionMetadata.setUrl(JcrMetadataRepository.getPropertyString(node, "url"));
            versionMetadata.setIncomplete(node.hasProperty("incomplete") && node.getProperty("incomplete").getBoolean());
            String scmConnection = JcrMetadataRepository.getPropertyString(node, "scm.connection");
            String scmDeveloperConnection = JcrMetadataRepository.getPropertyString(node, "scm.developerConnection");
            String scmUrl = JcrMetadataRepository.getPropertyString(node, "scm.url");
            if (scmConnection != null || scmDeveloperConnection != null || scmUrl != null) {
                Scm scm = new Scm();
                scm.setConnection(scmConnection);
                scm.setDeveloperConnection(scmDeveloperConnection);
                scm.setUrl(scmUrl);
                versionMetadata.setScm(scm);
            }
            String ciSystem = JcrMetadataRepository.getPropertyString(node, "ci.system");
            String ciUrl = JcrMetadataRepository.getPropertyString(node, "ci.url");
            if (ciSystem != null || ciUrl != null) {
                CiManagement ci = new CiManagement();
                ci.setSystem(ciSystem);
                ci.setUrl(ciUrl);
                versionMetadata.setCiManagement(ci);
            }
            String issueSystem = JcrMetadataRepository.getPropertyString(node, "issue.system");
            String issueUrl = JcrMetadataRepository.getPropertyString(node, "issue.url");
            if (issueSystem != null || issueUrl != null) {
                IssueManagement issueManagement = new IssueManagement();
                issueManagement.setSystem(issueSystem);
                issueManagement.setUrl(issueUrl);
                versionMetadata.setIssueManagement(issueManagement);
            }
            String orgName = JcrMetadataRepository.getPropertyString(node, "org.name");
            String orgUrl = JcrMetadataRepository.getPropertyString(node, "org.url");
            if (orgName != null || orgUrl != null) {
                Organization org = new Organization();
                org.setName(orgName);
                org.setUrl(orgUrl);
                versionMetadata.setOrganization(org);
            }
            boolean done = false;
            int i = 0;
            while (!done) {
                String licenseName = JcrMetadataRepository.getPropertyString(node, "license." + i + ".name");
                String licenseUrl = JcrMetadataRepository.getPropertyString(node, "license." + i + ".url");
                if (licenseName != null || licenseUrl != null) {
                    License license = new License();
                    license.setName(licenseName);
                    license.setUrl(licenseUrl);
                    versionMetadata.addLicense(license);
                } else {
                    done = true;
                }
                ++i;
            }
            done = false;
            i = 0;
            while (!done) {
                String mailingListName = JcrMetadataRepository.getPropertyString(node, "mailingList." + i + ".name");
                if (mailingListName != null) {
                    MailingList mailingList = new MailingList();
                    mailingList.setName(mailingListName);
                    mailingList.setMainArchiveUrl(JcrMetadataRepository.getPropertyString(node, "mailingList." + i + ".archive"));
                    Object n = "mailingList." + i + ".otherArchives";
                    if (node.hasProperty((String)n)) {
                        mailingList.setOtherArchives(Arrays.asList(JcrMetadataRepository.getPropertyString(node, (String)n).split(",")));
                    } else {
                        mailingList.setOtherArchives(Collections.emptyList());
                    }
                    mailingList.setPostAddress(JcrMetadataRepository.getPropertyString(node, "mailingList." + i + ".post"));
                    mailingList.setSubscribeAddress(JcrMetadataRepository.getPropertyString(node, "mailingList." + i + ".subscribe"));
                    mailingList.setUnsubscribeAddress(JcrMetadataRepository.getPropertyString(node, "mailingList." + i + ".unsubscribe"));
                    versionMetadata.addMailingList(mailingList);
                } else {
                    done = true;
                }
                ++i;
            }
            if (node.hasNode("dependencies")) {
                Node dependenciesNode = node.getNode("dependencies");
                for (Object n : JcrUtils.getChildNodes((Node)dependenciesNode)) {
                    if (!n.isNodeType(DEPENDENCY_NODE_TYPE)) continue;
                    Dependency dependency = new Dependency();
                    dependency.setArtifactId(JcrMetadataRepository.getPropertyString(n, "artifactId"));
                    dependency.setGroupId(JcrMetadataRepository.getPropertyString(n, "groupId"));
                    dependency.setClassifier(JcrMetadataRepository.getPropertyString(n, "classifier"));
                    dependency.setOptional(Boolean.valueOf(JcrMetadataRepository.getPropertyString(n, "optional")).booleanValue());
                    dependency.setScope(JcrMetadataRepository.getPropertyString(n, "scope"));
                    dependency.setSystemPath(JcrMetadataRepository.getPropertyString(n, "systemPath"));
                    dependency.setType(JcrMetadataRepository.getPropertyString(n, "type"));
                    dependency.setVersion(JcrMetadataRepository.getPropertyString(n, "version"));
                    versionMetadata.addDependency(dependency);
                }
            }
            for (Node n : JcrUtils.getChildNodes((Node)node)) {
                if (!n.isNodeType(FACET_NODE_TYPE)) continue;
                String name = n.getName();
                MetadataFacetFactory factory = this.metadataFacetFactories.get(name);
                if (factory == null) {
                    this.log.error("Attempted to load unknown project version metadata facet: {}", (Object)name);
                    continue;
                }
                MetadataFacet facet = factory.createMetadataFacet();
                HashMap<String, String> map = new HashMap<String, String>();
                for (Property property : JcrUtils.getProperties((Node)n)) {
                    String p = property.getName();
                    if (p.startsWith("jcr:")) continue;
                    map.put(p, property.getString());
                }
                facet.fromProperties(map);
                versionMetadata.addFacet(facet);
            }
        }
        catch (RepositoryException e) {
            throw new MetadataResolutionException(e.getMessage(), (Throwable)e);
        }
        return versionMetadata;
    }

    public Collection<String> getArtifactVersions(String repositoryId, String namespace, String projectId, String projectVersion) throws MetadataResolutionException {
        LinkedHashSet<String> versions = new LinkedHashSet<String>();
        try {
            Node root = this.getJcrSession().getRootNode();
            Node node = root.getNode(JcrMetadataRepository.getProjectVersionPath(repositoryId, namespace, projectId, projectVersion));
            for (Node n : JcrUtils.getChildNodes((Node)node)) {
                versions.add(n.getProperty("version").getString());
            }
        }
        catch (PathNotFoundException root) {
        }
        catch (RepositoryException e) {
            throw new MetadataResolutionException(e.getMessage(), (Throwable)e);
        }
        return versions;
    }

    public Collection<ProjectVersionReference> getProjectReferences(String repositoryId, String namespace, String projectId, String projectVersion) throws MetadataResolutionException {
        ArrayList<ProjectVersionReference> references = new ArrayList<ProjectVersionReference>();
        String q = "SELECT * FROM [archiva:dependency] WHERE ISDESCENDANTNODE([/repositories/" + repositoryId + "/content]) AND [groupId]='" + namespace + "' AND [artifactId]='" + projectId + "'";
        if (projectVersion != null) {
            q = q + " AND [version]='" + projectVersion + "'";
        }
        try {
            Query query = this.getJcrSession().getWorkspace().getQueryManager().createQuery(q, "JCR-SQL2");
            QueryResult result = query.execute();
            for (Node n : JcrUtils.getNodes((QueryResult)result)) {
                n = n.getParent();
                n = n.getParent();
                String usedByProjectVersion = n.getName();
                n = n.getParent();
                String usedByProject = n.getName();
                n = n.getParent();
                String usedByNamespace = n.getProperty("namespace").getString();
                ProjectVersionReference ref = new ProjectVersionReference();
                ref.setNamespace(usedByNamespace);
                ref.setProjectId(usedByProject);
                ref.setProjectVersion(usedByProjectVersion);
                ref.setReferenceType(ProjectVersionReference.ReferenceType.DEPENDENCY);
                references.add(ref);
            }
        }
        catch (RepositoryException e) {
            throw new MetadataResolutionException(e.getMessage(), (Throwable)e);
        }
        return references;
    }

    public Collection<String> getRootNamespaces(String repositoryId) throws MetadataResolutionException {
        return this.getNamespaces(repositoryId, null);
    }

    public Collection<String> getNamespaces(String repositoryId, String baseNamespace) throws MetadataResolutionException {
        String path = baseNamespace != null ? JcrMetadataRepository.getNamespacePath(repositoryId, baseNamespace) : JcrMetadataRepository.getRepositoryContentPath(repositoryId);
        return this.getNodeNames(path, NAMESPACE_NODE_TYPE);
    }

    public Collection<String> getProjects(String repositoryId, String namespace) throws MetadataResolutionException {
        return this.getNodeNames(JcrMetadataRepository.getNamespacePath(repositoryId, namespace), PROJECT_NODE_TYPE);
    }

    public Collection<String> getProjectVersions(String repositoryId, String namespace, String projectId) throws MetadataResolutionException {
        return this.getNodeNames(JcrMetadataRepository.getProjectPath(repositoryId, namespace, projectId), PROJECT_VERSION_NODE_TYPE);
    }

    public void removeArtifact(ArtifactMetadata artifactMetadata, String baseVersion) throws MetadataRepositoryException {
        String repositoryId = artifactMetadata.getRepositoryId();
        try {
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getProjectVersionPath(repositoryId, artifactMetadata.getNamespace(), artifactMetadata.getProject(), baseVersion);
            if (root.hasNode(path)) {
                Node node = root.getNode(path);
                for (Node n : JcrUtils.getChildNodes((Node)node)) {
                    String version;
                    if (!n.isNodeType(ARTIFACT_NODE_TYPE) || !n.hasProperty("version") || !StringUtils.equals((String)(version = n.getProperty("version").getString()), (String)artifactMetadata.getVersion())) continue;
                    n.remove();
                }
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void removeProjectVersion(String repoId, String namespace, String projectId, String projectVersion) throws MetadataRepositoryException {
        try {
            String path = JcrMetadataRepository.getProjectPath(repoId, namespace, projectId);
            Node root = this.getJcrSession().getRootNode();
            Node nodeAtPath = root.getNode(path);
            for (Node node : JcrUtils.getChildNodes((Node)nodeAtPath)) {
                if (!node.isNodeType(PROJECT_VERSION_NODE_TYPE) || !StringUtils.equals((String)projectVersion, (String)node.getName())) continue;
                node.remove();
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void removeArtifact(String repositoryId, String namespace, String projectId, String projectVersion, String id) throws MetadataRepositoryException {
        try {
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getArtifactPath(repositoryId, namespace, projectId, projectVersion, id);
            if (root.hasNode(path)) {
                root.getNode(path).remove();
            }
            path = JcrMetadataRepository.getProjectPath(repositoryId, namespace, projectId);
            Node nodeAtPath = root.getNode(path);
            for (Node node : JcrUtils.getChildNodes((Node)nodeAtPath)) {
                if (!node.isNodeType(PROJECT_VERSION_NODE_TYPE) || !StringUtils.equals((String)node.getName(), (String)projectVersion)) continue;
                node.remove();
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void removeArtifact(String repositoryId, String namespace, String project, String projectVersion, MetadataFacet metadataFacet) throws MetadataRepositoryException {
        try {
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getProjectVersionPath(repositoryId, namespace, project, projectVersion);
            if (root.hasNode(path)) {
                Node node = root.getNode(path);
                for (Node n : JcrUtils.getChildNodes((Node)node)) {
                    if (!n.isNodeType(ARTIFACT_NODE_TYPE)) continue;
                    ArtifactMetadata artifactMetadata = this.getArtifactFromNode(repositoryId, n);
                    this.log.debug("artifactMetadata: {}", (Object)artifactMetadata);
                    MetadataFacet metadataFacetToRemove = artifactMetadata.getFacet(metadataFacet.getFacetId());
                    if (metadataFacetToRemove == null || !metadataFacet.equals(metadataFacetToRemove)) continue;
                    n.remove();
                }
            }
        }
        catch (RepositoryException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public Collection<ArtifactMetadata> getArtifacts(String repositoryId, String namespace, String projectId, String projectVersion) throws MetadataResolutionException {
        ArrayList<ArtifactMetadata> artifacts = new ArrayList<ArtifactMetadata>();
        try {
            Node root = this.getJcrSession().getRootNode();
            String path = JcrMetadataRepository.getProjectVersionPath(repositoryId, namespace, projectId, projectVersion);
            if (root.hasNode(path)) {
                Node node = root.getNode(path);
                for (Node n : JcrUtils.getChildNodes((Node)node)) {
                    if (!n.isNodeType(ARTIFACT_NODE_TYPE)) continue;
                    artifacts.add(this.getArtifactFromNode(repositoryId, n));
                }
            }
        }
        catch (RepositoryException e) {
            throw new MetadataResolutionException(e.getMessage(), (Throwable)e);
        }
        return artifacts;
    }

    public void save() {
        try {
            this.getJcrSession().save();
        }
        catch (InvalidItemStateException e) {
            this.log.warn("skip InvalidItemStateException:" + e.getMessage(), (Throwable)e);
        }
        catch (RepositoryException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public void revert() {
        try {
            this.getJcrSession().refresh(false);
        }
        catch (RepositoryException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public boolean canObtainAccess(Class<?> aClass) {
        return aClass == Session.class;
    }

    public <T> T obtainAccess(Class<T> aClass) throws MetadataRepositoryException {
        if (aClass == Session.class) {
            try {
                return (T)this.getJcrSession();
            }
            catch (RepositoryException e) {
                this.log.error(e.getMessage(), (Throwable)e);
                throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
            }
        }
        throw new IllegalArgumentException("Access using " + aClass + " is not supported on the JCR metadata storage");
    }

    public void close() throws MetadataRepositoryException {
        if (this.jcrSession != null && this.jcrSession.isLive()) {
            this.jcrSession.logout();
        }
    }

    public List<ArtifactMetadata> searchArtifacts(String text, String repositoryId, boolean exact) throws MetadataRepositoryException {
        return this.searchArtifacts(null, text, repositoryId, exact);
    }

    public List<ArtifactMetadata> searchArtifacts(String key, String text, String repositoryId, boolean exact) throws MetadataRepositoryException {
        boolean e = exact && key != null;
        String theKey = key == null ? "*" : "[" + key + "]";
        String projectVersionCondition = e ? "(projectVersion." + theKey + " = $value)" : "contains([projectVersion]." + theKey + ", $value)";
        String facetCondition = e ? "(facet." + theKey + " = $value)" : "contains([facet]." + theKey + ", $value)";
        String q = "SELECT * FROM [archiva:projectVersion] AS projectVersion LEFT OUTER JOIN [archiva:artifact] AS artifact ON ISCHILDNODE(artifact, projectVersion) LEFT OUTER JOIN [archiva:facet] AS facet ON ISCHILDNODE(facet, projectVersion) WHERE (" + projectVersionCondition + " OR " + facetCondition + ")";
        return this.runJcrQuery(repositoryId, q, (Map<String, String>)ImmutableMap.of((Object)"value", (Object)text));
    }

    private ArtifactMetadata getArtifactFromNode(String repositoryId, Node artifactNode) throws RepositoryException {
        String id = artifactNode.getName();
        ArtifactMetadata artifact = new ArtifactMetadata();
        artifact.setId(id);
        artifact.setRepositoryId(repositoryId == null ? artifactNode.getAncestor(2).getName() : repositoryId);
        Node projectVersionNode = artifactNode.getParent();
        Node projectNode = projectVersionNode.getParent();
        Node namespaceNode = projectNode.getParent();
        artifact.setNamespace(namespaceNode.getProperty("namespace").getString());
        artifact.setProject(projectNode.getName());
        artifact.setProjectVersion(projectVersionNode.getName());
        artifact.setVersion(artifactNode.hasProperty("version") ? artifactNode.getProperty("version").getString() : projectVersionNode.getName());
        if (artifactNode.hasProperty(JCR_LAST_MODIFIED)) {
            artifact.setFileLastModified(artifactNode.getProperty(JCR_LAST_MODIFIED).getDate().getTimeInMillis());
        }
        if (artifactNode.hasProperty("whenGathered")) {
            artifact.setWhenGathered(artifactNode.getProperty("whenGathered").getDate().getTime());
        }
        if (artifactNode.hasProperty("size")) {
            artifact.setSize(artifactNode.getProperty("size").getLong());
        }
        if (artifactNode.hasProperty("md5")) {
            artifact.setMd5(artifactNode.getProperty("md5").getString());
        }
        if (artifactNode.hasProperty("sha1")) {
            artifact.setSha1(artifactNode.getProperty("sha1").getString());
        }
        for (Node n : JcrUtils.getChildNodes((Node)artifactNode)) {
            if (!n.isNodeType(FACET_NODE_TYPE)) continue;
            String name = n.getName();
            MetadataFacetFactory factory = this.metadataFacetFactories.get(name);
            if (factory == null) {
                this.log.error("Attempted to load unknown project version metadata facet: " + name);
                continue;
            }
            MetadataFacet facet = factory.createMetadataFacet();
            HashMap<String, String> map = new HashMap<String, String>();
            for (Property p : JcrUtils.getProperties((Node)n)) {
                String property = p.getName();
                if (property.startsWith("jcr:")) continue;
                map.put(property, p.getString());
            }
            facet.fromProperties(map);
            artifact.addFacet(facet);
        }
        return artifact;
    }

    private static String getPropertyString(Node node, String name) throws RepositoryException {
        return node.hasProperty(name) ? node.getProperty(name).getString() : null;
    }

    private Collection<String> getNodeNames(String path, String nodeType) throws MetadataResolutionException {
        ArrayList<String> names = new ArrayList<String>();
        try {
            Node root = this.getJcrSession().getRootNode();
            Node nodeAtPath = root.getNode(path);
            for (Node node : JcrUtils.getChildNodes((Node)nodeAtPath)) {
                if (!node.isNodeType(nodeType)) continue;
                names.add(node.getName());
            }
        }
        catch (PathNotFoundException root) {
        }
        catch (RepositoryException e) {
            throw new MetadataResolutionException(e.getMessage(), (Throwable)e);
        }
        return names;
    }

    private static String getRepositoryPath(String repositoryId) {
        return "repositories/" + repositoryId;
    }

    private static String getRepositoryContentPath(String repositoryId) {
        return JcrMetadataRepository.getRepositoryPath(repositoryId) + "/content/";
    }

    private static String getFacetPath(String repositoryId, String facetId) {
        return JcrMetadataRepository.getRepositoryPath(repositoryId) + "/facets/" + facetId;
    }

    private static String getNamespacePath(String repositoryId, String namespace) {
        return JcrMetadataRepository.getRepositoryContentPath(repositoryId) + namespace.replace('.', '/');
    }

    private static String getProjectPath(String repositoryId, String namespace, String projectId) {
        return JcrMetadataRepository.getNamespacePath(repositoryId, namespace) + "/" + projectId;
    }

    private static String getProjectVersionPath(String repositoryId, String namespace, String projectId, String projectVersion) {
        return JcrMetadataRepository.getProjectPath(repositoryId, namespace, projectId) + "/" + projectVersion;
    }

    private static String getArtifactPath(String repositoryId, String namespace, String projectId, String projectVersion, String id) {
        return JcrMetadataRepository.getProjectVersionPath(repositoryId, namespace, projectId, projectVersion) + "/" + id;
    }

    private Node getOrAddNodeByPath(Node baseNode, String name) throws RepositoryException {
        return this.getOrAddNodeByPath(baseNode, name, null);
    }

    private Node getOrAddNodeByPath(Node baseNode, String name, String nodeType) throws RepositoryException {
        Node node = baseNode;
        for (String n : name.split("/")) {
            node = JcrUtils.getOrAddNode((Node)node, (String)n);
            if (nodeType == null) continue;
            node.addMixin(nodeType);
        }
        return node;
    }

    private static String getFacetPath(String repositoryId, String facetId, String name) {
        return JcrMetadataRepository.getFacetPath(repositoryId, facetId) + "/" + name;
    }

    private Node getOrAddRepositoryNode(String repositoryId) throws RepositoryException {
        Node root = this.getJcrSession().getRootNode();
        Node node = JcrUtils.getOrAddNode((Node)root, (String)"repositories");
        node = JcrUtils.getOrAddNode((Node)node, (String)repositoryId);
        return node;
    }

    private Node getOrAddRepositoryContentNode(String repositoryId) throws RepositoryException {
        Node node = this.getOrAddRepositoryNode(repositoryId);
        return JcrUtils.getOrAddNode((Node)node, (String)"content");
    }

    private Node getOrAddNamespaceNode(String repositoryId, String namespace) throws RepositoryException {
        Node repo = this.getOrAddRepositoryContentNode(repositoryId);
        return this.getOrAddNodeByPath(repo, namespace.replace('.', '/'), NAMESPACE_NODE_TYPE);
    }

    private Node getOrAddProjectNode(String repositoryId, String namespace, String projectId) throws RepositoryException {
        Node namespaceNode = this.getOrAddNamespaceNode(repositoryId, namespace);
        Node node = JcrUtils.getOrAddNode((Node)namespaceNode, (String)projectId);
        node.addMixin(PROJECT_NODE_TYPE);
        return node;
    }

    private Node getOrAddProjectVersionNode(String repositoryId, String namespace, String projectId, String projectVersion) throws RepositoryException {
        Node projectNode = this.getOrAddProjectNode(repositoryId, namespace, projectId);
        Node node = JcrUtils.getOrAddNode((Node)projectNode, (String)projectVersion);
        node.addMixin(PROJECT_VERSION_NODE_TYPE);
        return node;
    }

    private Node getOrAddArtifactNode(String repositoryId, String namespace, String projectId, String projectVersion, String id) throws RepositoryException {
        Node versionNode = this.getOrAddProjectVersionNode(repositoryId, namespace, projectId, projectVersion);
        Node node = JcrUtils.getOrAddNode((Node)versionNode, (String)id);
        node.addMixin(ARTIFACT_NODE_TYPE);
        return node;
    }

    private static Calendar createCalendar(Date time) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(time);
        return cal;
    }

    private String join(Collection<String> ids) {
        if (ids != null && !ids.isEmpty()) {
            StringBuilder s = new StringBuilder();
            for (String id : ids) {
                s.append(id);
                s.append(",");
            }
            return s.substring(0, s.length() - 1);
        }
        return null;
    }

    public Session getJcrSession() throws RepositoryException {
        if (this.jcrSession == null || !this.jcrSession.isLive()) {
            this.jcrSession = this.repository.login((Credentials)new SimpleCredentials("admin", "admin".toCharArray()));
        }
        return this.jcrSession;
    }
}

