/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.cql;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;
import org.antlr.runtime.ANTLRStringStream;
import org.antlr.runtime.CharStream;
import org.antlr.runtime.CommonTokenStream;
import org.antlr.runtime.RecognitionException;
import org.antlr.runtime.TokenSource;
import org.antlr.runtime.TokenStream;
import org.apache.cassandra.auth.Permission;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ColumnDefinition;
import org.apache.cassandra.config.ConfigurationException;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.KSMetaData;
import org.apache.cassandra.cql.BatchUpdateStatement;
import org.apache.cassandra.cql.CQLStatement;
import org.apache.cassandra.cql.CqlLexer;
import org.apache.cassandra.cql.CqlParser;
import org.apache.cassandra.cql.CreateColumnFamilyStatement;
import org.apache.cassandra.cql.CreateIndexStatement;
import org.apache.cassandra.cql.CreateKeyspaceStatement;
import org.apache.cassandra.cql.DeleteStatement;
import org.apache.cassandra.cql.Relation;
import org.apache.cassandra.cql.RelationType;
import org.apache.cassandra.cql.SelectStatement;
import org.apache.cassandra.cql.StatementType;
import org.apache.cassandra.cql.Term;
import org.apache.cassandra.cql.UpdateStatement;
import org.apache.cassandra.db.IColumn;
import org.apache.cassandra.db.RangeSliceCommand;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.SliceByNamesReadCommand;
import org.apache.cassandra.db.SliceFromReadCommand;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.db.filter.QueryPath;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.MarshalException;
import org.apache.cassandra.db.migration.AddColumnFamily;
import org.apache.cassandra.db.migration.AddKeyspace;
import org.apache.cassandra.db.migration.DropColumnFamily;
import org.apache.cassandra.db.migration.DropKeyspace;
import org.apache.cassandra.db.migration.Migration;
import org.apache.cassandra.db.migration.UpdateColumnFamily;
import org.apache.cassandra.db.migration.avro.CfDef;
import org.apache.cassandra.dht.Bounds;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.locator.AbstractReplicationStrategy;
import org.apache.cassandra.service.ClientState;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.thrift.Column;
import org.apache.cassandra.thrift.ConsistencyLevel;
import org.apache.cassandra.thrift.CqlResult;
import org.apache.cassandra.thrift.CqlResultType;
import org.apache.cassandra.thrift.CqlRow;
import org.apache.cassandra.thrift.IndexClause;
import org.apache.cassandra.thrift.IndexExpression;
import org.apache.cassandra.thrift.IndexOperator;
import org.apache.cassandra.thrift.IndexType;
import org.apache.cassandra.thrift.InvalidRequestException;
import org.apache.cassandra.thrift.SchemaDisagreementException;
import org.apache.cassandra.thrift.SlicePredicate;
import org.apache.cassandra.thrift.SliceRange;
import org.apache.cassandra.thrift.ThriftValidation;
import org.apache.cassandra.thrift.TimedOutException;
import org.apache.cassandra.thrift.UnavailableException;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueryProcessor {
    private static final Logger logger = LoggerFactory.getLogger(QueryProcessor.class);

    private static List<Row> getSlice(String keyspace, SelectStatement select) throws InvalidRequestException, TimedOutException, UnavailableException {
        List<Row> rows = null;
        QueryPath queryPath = new QueryPath(select.getColumnFamily());
        AbstractType comparator = select.getComparator(keyspace);
        ArrayList<ReadCommand> commands = new ArrayList<ReadCommand>();
        assert (select.getKeys().size() == 1);
        CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, select.getColumnFamily(), false);
        ByteBuffer key = select.getKeys().get(0).getByteBuffer(metadata.getKeyValidator());
        QueryProcessor.validateKey(key);
        if (!select.isColumnRange()) {
            ArrayList<ByteBuffer> columnNames = new ArrayList<ByteBuffer>();
            for (Term column : select.getColumnNames()) {
                columnNames.add(column.getByteBuffer(comparator));
            }
            QueryProcessor.validateColumnNames(columnNames);
            commands.add(new SliceByNamesReadCommand(keyspace, key, queryPath, columnNames));
        } else {
            ByteBuffer start = select.getColumnStart().getByteBuffer(comparator);
            ByteBuffer finish = select.getColumnFinish().getByteBuffer(comparator);
            QueryProcessor.validateSliceRange(metadata, start, finish, select.isColumnsReversed());
            commands.add(new SliceFromReadCommand(keyspace, key, queryPath, start, finish, select.isColumnsReversed(), select.getColumnsLimit()));
        }
        try {
            rows = StorageProxy.read(commands, select.getConsistencyLevel());
        }
        catch (TimeoutException e) {
            throw new TimedOutException();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return rows;
    }

    private static List<Row> multiRangeSlice(String keyspace, SelectStatement select) throws TimedOutException, UnavailableException, InvalidRequestException {
        List<Row> rows = null;
        AbstractType keyType = DatabaseDescriptor.getCFMetaData(keyspace, select.getColumnFamily()).getKeyValidator();
        ByteBuffer startKey = select.getKeyStart() != null ? select.getKeyStart().getByteBuffer(keyType) : new Term().getByteBuffer();
        ByteBuffer finishKey = select.getKeyFinish() != null ? select.getKeyFinish().getByteBuffer(keyType) : new Term().getByteBuffer();
        IPartitioner p = StorageService.getPartitioner();
        Bounds bounds = new Bounds((Token)p.getToken(startKey), (Token)p.getToken(finishKey));
        CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, select.getColumnFamily(), false);
        AbstractType comparator = metadata.getComparatorFor(null);
        SlicePredicate thriftSlicePredicate = QueryProcessor.slicePredicateFromSelect(select, comparator);
        QueryProcessor.validateSlicePredicate(metadata, thriftSlicePredicate);
        try {
            rows = StorageProxy.getRangeSlice(new RangeSliceCommand(keyspace, select.getColumnFamily(), null, thriftSlicePredicate, bounds, select.getNumRecords()), select.getConsistencyLevel());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        catch (UnavailableException e) {
            throw new UnavailableException();
        }
        catch (TimeoutException e) {
            throw new TimedOutException();
        }
        return rows;
    }

    private static List<Row> getIndexedSlices(String keyspace, SelectStatement select) throws TimedOutException, UnavailableException, InvalidRequestException {
        List<Row> rows;
        CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, select.getColumnFamily(), false);
        AbstractType comparator = metadata.getComparatorFor(null);
        SlicePredicate thriftSlicePredicate = QueryProcessor.slicePredicateFromSelect(select, comparator);
        QueryProcessor.validateSlicePredicate(metadata, thriftSlicePredicate);
        ArrayList<IndexExpression> expressions = new ArrayList<IndexExpression>();
        for (Relation columnRelation : select.getColumnRelations()) {
            ByteBuffer entity = columnRelation.getEntity().getByteBuffer(comparator);
            ByteBuffer value = columnRelation.getValue().getByteBuffer(select.getValueValidator(keyspace, entity));
            expressions.add(new IndexExpression(entity, IndexOperator.valueOf((String)columnRelation.operator().toString()), value));
        }
        AbstractType keyType = DatabaseDescriptor.getCFMetaData(keyspace, select.getColumnFamily()).getKeyValidator();
        ByteBuffer startKey = !select.isKeyRange() ? new Term().getByteBuffer() : select.getKeyStart().getByteBuffer(keyType);
        IndexClause thriftIndexClause = new IndexClause(expressions, startKey, select.getNumRecords());
        try {
            rows = StorageProxy.scan(keyspace, select.getColumnFamily(), thriftIndexClause, thriftSlicePredicate, select.getConsistencyLevel());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        catch (TimeoutException e) {
            throw new TimedOutException();
        }
        return rows;
    }

    private static void batchUpdate(ClientState clientState, List<UpdateStatement> updateStatements, ConsistencyLevel consistency) throws InvalidRequestException, UnavailableException, TimedOutException {
        String keyspace = clientState.getKeyspace();
        ArrayList<RowMutation> rowMutations = new ArrayList<RowMutation>();
        ArrayList<String> cfamsSeen = new ArrayList<String>();
        for (UpdateStatement update : updateStatements) {
            CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, update.getColumnFamily(), false);
            if (!cfamsSeen.contains(update.getColumnFamily())) {
                clientState.hasColumnFamilyAccess(update.getColumnFamily(), Permission.WRITE);
                cfamsSeen.add(update.getColumnFamily());
            }
            ByteBuffer key = update.getKey().getByteBuffer(update.getKeyType(keyspace));
            QueryProcessor.validateKey(key);
            AbstractType<?> comparator = update.getComparator(keyspace);
            RowMutation rm = new RowMutation(keyspace, key);
            for (Map.Entry<Term, Term> column : update.getColumns().entrySet()) {
                ByteBuffer colName = column.getKey().getByteBuffer(comparator);
                ByteBuffer colValue = column.getValue().getByteBuffer(update.getValueValidator(keyspace, colName));
                QueryProcessor.validateColumn(metadata, colName, colValue);
                rm.add(new QueryPath(update.getColumnFamily(), null, colName), colValue, System.currentTimeMillis());
            }
            rowMutations.add(rm);
        }
        try {
            StorageProxy.mutate(rowMutations, consistency);
        }
        catch (UnavailableException e) {
            throw new UnavailableException();
        }
        catch (TimeoutException e) {
            throw new TimedOutException();
        }
    }

    private static SlicePredicate slicePredicateFromSelect(SelectStatement select, AbstractType<?> comparator) throws InvalidRequestException {
        SlicePredicate thriftSlicePredicate = new SlicePredicate();
        if (select.isColumnRange() || select.getColumnNames().size() == 0) {
            SliceRange sliceRange = new SliceRange();
            sliceRange.start = select.getColumnStart().getByteBuffer(comparator);
            sliceRange.finish = select.getColumnFinish().getByteBuffer(comparator);
            sliceRange.reversed = select.isColumnsReversed();
            sliceRange.count = select.getColumnsLimit();
            thriftSlicePredicate.slice_range = sliceRange;
        } else {
            ArrayList<ByteBuffer> columnNames = new ArrayList<ByteBuffer>();
            for (Term column : select.getColumnNames()) {
                columnNames.add(column.getByteBuffer(comparator));
            }
            thriftSlicePredicate.column_names = columnNames;
        }
        return thriftSlicePredicate;
    }

    private static void validateSelect(String keyspace, SelectStatement select) throws InvalidRequestException {
        if (select.isCountOperation() && (select.isKeyRange() || select.getKeys().size() < 1)) {
            throw new InvalidRequestException("Counts can only be performed for a single record (Hint: KEY=term)");
        }
        if (!select.isKeyRange() && select.getKeyFinish() != null) {
            throw new InvalidRequestException("Key range clauses must include a start key (i.e. KEY > term)");
        }
        if (select.isKeyRange() && select.getKeys().size() > 0) {
            throw new InvalidRequestException("You cannot combine key range and by-key clauses in a SELECT");
        }
        if (select.isKeyRange() && select.getKeyFinish() != null && select.getColumnRelations().size() > 0) {
            throw new InvalidRequestException("You cannot combine key range and by-column clauses in a SELECT");
        }
        if (select.getKeys().size() > 1) {
            throw new InvalidRequestException("SELECTs can contain only by by-key clause");
        }
        AbstractType comparator = select.getComparator(keyspace);
        if (select.getColumnRelations().size() > 0) {
            SortedSet<ByteBuffer> indexed = Table.open(keyspace).getColumnFamilyStore(select.getColumnFamily()).getIndexedColumns();
            for (Relation relation : select.getColumnRelations()) {
                if (!relation.operator().equals((Object)RelationType.EQ) || !indexed.contains(relation.getEntity().getByteBuffer(comparator))) continue;
                return;
            }
            throw new InvalidRequestException("No indexed columns present in by-columns clause with \"equals\" operator");
        }
    }

    private static void applyMigrationOnStage(final Migration m) throws InvalidRequestException {
        Future<Object> f = StageManager.getStage(Stage.MIGRATION).submit(new Callable<Object>(){

            @Override
            public Object call() throws Exception {
                m.apply();
                m.announce();
                return null;
            }
        });
        try {
            f.get();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            if (e.getCause() != null) {
                InvalidRequestException ex = new InvalidRequestException(e.getCause().getMessage());
                ex.initCause(e.getCause());
                throw ex;
            }
            InvalidRequestException ex = new InvalidRequestException(e.getMessage());
            ex.initCause((Throwable)e);
            throw ex;
        }
    }

    private static void validateKey(ByteBuffer key) throws InvalidRequestException {
        if (key == null || key.remaining() == 0) {
            throw new InvalidRequestException("Key may not be empty");
        }
        if (key.remaining() > 65535) {
            throw new InvalidRequestException("Key length of " + key.remaining() + " is longer than maximum of " + 65535);
        }
    }

    private static void validateColumnNames(Iterable<ByteBuffer> columns) throws InvalidRequestException {
        for (ByteBuffer name : columns) {
            if (name.remaining() > 65535) {
                throw new InvalidRequestException(String.format("column name is too long (%s > %s)", name.remaining(), 65535));
            }
            if (name.remaining() != 0) continue;
            throw new InvalidRequestException("zero-length column name");
        }
    }

    private static void validateColumnName(ByteBuffer column) throws InvalidRequestException {
        QueryProcessor.validateColumnNames(Arrays.asList(column));
    }

    private static void validateColumn(CFMetaData metadata, ByteBuffer name, ByteBuffer value) throws InvalidRequestException {
        QueryProcessor.validateColumnName(name);
        AbstractType validator = metadata.getValueValidator(name);
        try {
            if (validator != null) {
                validator.validate(value);
            }
        }
        catch (MarshalException me) {
            throw new InvalidRequestException(String.format("Invalid column value for column (name=%s); %s", ByteBufferUtil.bytesToHex(name), me.getMessage()));
        }
    }

    private static void validateSlicePredicate(CFMetaData metadata, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.slice_range != null) {
            QueryProcessor.validateSliceRange(metadata, predicate.slice_range);
        } else {
            QueryProcessor.validateColumnNames(predicate.column_names);
        }
    }

    private static void validateSliceRange(CFMetaData metadata, SliceRange range) throws InvalidRequestException {
        QueryProcessor.validateSliceRange(metadata, range.start, range.finish, range.reversed);
    }

    private static void validateSliceRange(CFMetaData metadata, ByteBuffer start, ByteBuffer finish, boolean reversed) throws InvalidRequestException {
        Comparator<ByteBuffer> orderedComparator;
        AbstractType comparator = metadata.getComparatorFor(null);
        Comparator<ByteBuffer> comparator2 = orderedComparator = reversed ? comparator.reverseComparator : comparator;
        if (start.remaining() > 0 && finish.remaining() > 0 && orderedComparator.compare(start, finish) > 0) {
            throw new InvalidRequestException("range finish must come after start in traversal order");
        }
    }

    private static void validateSchemaAgreement() throws SchemaDisagreementException {
        Map versions = Maps.filterKeys(StorageProxy.describeSchemaVersions(), (Predicate)Predicates.not((Predicate)Predicates.equalTo((Object)"UNREACHABLE")));
        if (versions.size() > 1) {
            throw new SchemaDisagreementException();
        }
    }

    public static CqlResult process(String queryString, ClientState clientState) throws RecognitionException, UnavailableException, InvalidRequestException, TimedOutException, SchemaDisagreementException {
        logger.trace("CQL QUERY: {}", (Object)queryString);
        CQLStatement statement = QueryProcessor.getStatement(queryString);
        String keyspace = null;
        if (StatementType.requiresKeyspace.contains((Object)statement.type)) {
            keyspace = clientState.getKeyspace();
        }
        CqlResult result = new CqlResult();
        logger.debug("CQL statement type: {}", (Object)statement.type.toString());
        switch (statement.type) {
            case SELECT: {
                SelectStatement select = (SelectStatement)statement.statement;
                clientState.hasColumnFamilyAccess(select.getColumnFamily(), Permission.READ);
                CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, select.getColumnFamily(), false);
                AbstractType comparator = metadata.getComparatorFor(null);
                QueryProcessor.validateSelect(keyspace, select);
                List<Row> rows = null;
                if (!select.isKeyRange() && select.getKeys().size() > 0) {
                    rows = QueryProcessor.getSlice(keyspace, select);
                    if (select.isCountOperation()) {
                        result.type = CqlResultType.INT;
                        if (rows.size() > 0) {
                            result.setNum(rows.get((int)0).cf != null ? rows.get((int)0).cf.getSortedColumns().size() : 0);
                        } else {
                            result.setNum(0);
                        }
                        return result;
                    }
                } else {
                    rows = select.getKeyFinish() != null || select.getColumnRelations().size() == 0 ? QueryProcessor.multiRangeSlice(keyspace, select) : QueryProcessor.getIndexedSlices(keyspace, select);
                }
                ArrayList<CqlRow> cqlRows = new ArrayList<CqlRow>();
                result.type = CqlResultType.ROWS;
                for (Row row : rows) {
                    if (row.cf == null) continue;
                    List<Column> thriftColumns = QueryProcessor.extractThriftColumns(select, comparator, row);
                    CqlRow cqlRow = new CqlRow();
                    cqlRow.key = row.key.key;
                    cqlRow.columns = thriftColumns;
                    if (select.isColumnsReversed()) {
                        Collections.reverse(cqlRow.columns);
                    }
                    cqlRows.add(cqlRow);
                }
                result.rows = cqlRows;
                return result;
            }
            case INSERT: 
            case UPDATE: {
                UpdateStatement update = (UpdateStatement)statement.statement;
                QueryProcessor.batchUpdate(clientState, Collections.singletonList(update), update.getConsistencyLevel());
                result.type = CqlResultType.VOID;
                return result;
            }
            case BATCH_UPDATE: {
                BatchUpdateStatement batch = (BatchUpdateStatement)statement.statement;
                for (UpdateStatement up : batch.getUpdates()) {
                    if (!up.isSetConsistencyLevel()) continue;
                    throw new InvalidRequestException("Consistency level must be set on the BATCH, not individual UPDATE statements");
                }
                QueryProcessor.batchUpdate(clientState, batch.getUpdates(), batch.getConsistencyLevel());
                result.type = CqlResultType.VOID;
                return result;
            }
            case USE: {
                clientState.setKeyspace((String)statement.statement);
                result.type = CqlResultType.VOID;
                return result;
            }
            case TRUNCATE: {
                String columnFamily = (String)statement.statement;
                clientState.hasColumnFamilyAccess(columnFamily, Permission.WRITE);
                try {
                    StorageProxy.truncateBlocking(keyspace, columnFamily);
                }
                catch (TimeoutException e) {
                    throw (UnavailableException)new UnavailableException().initCause((Throwable)e);
                }
                catch (IOException e) {
                    throw (UnavailableException)new UnavailableException().initCause((Throwable)e);
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DELETE: {
                DeleteStatement delete = (DeleteStatement)statement.statement;
                clientState.hasColumnFamilyAccess(delete.getColumnFamily(), Permission.WRITE);
                CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, delete.getColumnFamily(), false);
                AbstractType comparator = metadata.getComparatorFor(null);
                AbstractType keyType = DatabaseDescriptor.getCFMetaData(keyspace, delete.getColumnFamily()).getKeyValidator();
                ArrayList<RowMutation> rowMutations = new ArrayList<RowMutation>();
                for (Term key : delete.getKeys()) {
                    RowMutation rm = new RowMutation(keyspace, key.getByteBuffer(keyType));
                    if (delete.getColumns().size() < 1) {
                        rm.delete(new QueryPath(delete.getColumnFamily()), System.currentTimeMillis());
                    } else {
                        for (Term column : delete.getColumns()) {
                            ByteBuffer columnName = column.getByteBuffer(comparator);
                            QueryProcessor.validateColumnName(columnName);
                            rm.delete(new QueryPath(delete.getColumnFamily(), null, columnName), System.currentTimeMillis());
                        }
                    }
                    rowMutations.add(rm);
                }
                try {
                    StorageProxy.mutate(rowMutations, delete.getConsistencyLevel());
                }
                catch (TimeoutException e) {
                    throw new TimedOutException();
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case CREATE_KEYSPACE: {
                CreateKeyspaceStatement create = (CreateKeyspaceStatement)statement.statement;
                create.validate();
                clientState.hasKeyspaceListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    KSMetaData ksm = new KSMetaData(create.getName(), AbstractReplicationStrategy.getClass(create.getStrategyClass()), create.getStrategyOptions(), new CFMetaData[0]);
                    QueryProcessor.applyMigrationOnStage(new AddKeyspace(ksm));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case CREATE_COLUMNFAMILY: {
                CreateColumnFamilyStatement createCf = (CreateColumnFamilyStatement)statement.statement;
                clientState.hasColumnFamilyListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    QueryProcessor.applyMigrationOnStage(new AddColumnFamily(createCf.getCFMetaData(keyspace)));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case CREATE_INDEX: {
                CreateIndexStatement createIdx = (CreateIndexStatement)statement.statement;
                clientState.hasColumnFamilyListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                CFMetaData oldCfm = DatabaseDescriptor.getCFMetaData(CFMetaData.getId(keyspace, createIdx.getColumnFamily()));
                if (oldCfm == null) {
                    throw new InvalidRequestException("No such column family: " + createIdx.getColumnFamily());
                }
                ByteBuffer columnName = createIdx.getColumnName().getByteBuffer();
                ColumnDefinition columnDef = oldCfm.getColumn_metadata().get(columnName);
                if (columnDef != null) {
                    if (columnDef.getIndexType() != null) {
                        throw new InvalidRequestException("Index exists");
                    }
                    columnDef.setIndexName(createIdx.getIndexName());
                    columnDef.setIndexType(IndexType.KEYS);
                } else {
                    columnDef = new ColumnDefinition(columnName, DatabaseDescriptor.getValueValidator(keyspace, createIdx.getColumnFamily(), columnName), IndexType.KEYS, createIdx.getIndexName());
                }
                CfDef cfamilyDef = CFMetaData.convertToAvro(oldCfm);
                cfamilyDef.column_metadata.add(columnDef.deflate());
                try {
                    QueryProcessor.applyMigrationOnStage(new UpdateColumnFamily(cfamilyDef));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DROP_KEYSPACE: {
                String deleteKeyspace = (String)statement.statement;
                clientState.hasKeyspaceListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    QueryProcessor.applyMigrationOnStage(new DropKeyspace(deleteKeyspace));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DROP_COLUMNFAMILY: {
                String deleteColumnFamily = (String)statement.statement;
                clientState.hasColumnFamilyListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    QueryProcessor.applyMigrationOnStage(new DropColumnFamily(keyspace, deleteColumnFamily));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
        }
        return null;
    }

    private static List<Column> extractThriftColumns(SelectStatement select, AbstractType<?> comparator, Row row) {
        ArrayList<Column> thriftColumns = new ArrayList<Column>();
        if (select.isColumnRange()) {
            for (IColumn c : row.cf.getSortedColumns()) {
                if (c.isMarkedForDelete()) continue;
                thriftColumns.add(new Column(c.name()).setValue(c.value()).setTimestamp(c.timestamp()));
            }
        } else {
            for (Term term : select.getColumnNames()) {
                ByteBuffer name;
                try {
                    name = term.getByteBuffer(comparator);
                }
                catch (InvalidRequestException e) {
                    throw new AssertionError((Object)e);
                }
                IColumn c = row.cf.getColumn(name);
                if (c == null || c.isMarkedForDelete()) {
                    thriftColumns.add(new Column().setName(name));
                    continue;
                }
                thriftColumns.add(new Column(c.name()).setValue(c.value()).setTimestamp(c.timestamp()));
            }
        }
        return thriftColumns;
    }

    private static CQLStatement getStatement(String queryStr) throws InvalidRequestException, RecognitionException {
        ANTLRStringStream stream = new ANTLRStringStream(queryStr);
        CqlLexer lexer = new CqlLexer((CharStream)stream);
        CommonTokenStream tokenStream = new CommonTokenStream((TokenSource)lexer);
        CqlParser parser = new CqlParser((TokenStream)tokenStream);
        CQLStatement statement = parser.query();
        lexer.throwLastRecognitionError();
        parser.throwLastRecognitionError();
        return statement;
    }
}

