/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ConfigurationException;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.KSMetaData;
import org.apache.cassandra.db.Column;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.ExpiringColumn;
import org.apache.cassandra.db.IColumn;
import org.apache.cassandra.db.Memtable;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.filter.QueryFilter;
import org.apache.cassandra.db.filter.QueryPath;
import org.apache.cassandra.dht.LocalToken;
import org.apache.cassandra.io.CompactionInfo;
import org.apache.cassandra.io.CompactionType;
import org.apache.cassandra.io.sstable.ReducingKeyIterator;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.locator.AbstractReplicationStrategy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.NodeId;
import org.cliffc.high_scale_lib.NonBlockingHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Table {
    public static final String SYSTEM_TABLE = "system";
    private static final Logger logger = LoggerFactory.getLogger(Table.class);
    private static final String SNAPSHOT_SUBDIR_NAME = "snapshots";
    static final ReentrantReadWriteLock switchLock = new ReentrantReadWriteLock();
    private static final Map<String, Table> instances;
    public final String name;
    private final Map<Integer, ColumnFamilyStore> columnFamilyStores = new ConcurrentHashMap<Integer, ColumnFamilyStore>();
    private final Object[] indexLocks;
    private ScheduledFuture<?> flushTask;
    private volatile AbstractReplicationStrategy replicationStrategy;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static Table open(String table) {
        Table tableInstance = instances.get(table);
        if (tableInstance != null) return tableInstance;
        Class<Table> clazz = Table.class;
        synchronized (Table.class) {
            tableInstance = instances.get(table);
            if (tableInstance != null) return tableInstance;
            tableInstance = new Table(table);
            instances.put(table, tableInstance);
            for (ColumnFamilyStore cfs : tableInstance.getColumnFamilyStores()) {
                cfs.initCaches();
            }
            // ** MonitorExit[var2_2] (shouldn't be in output)
            return tableInstance;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Table clear(String table) throws IOException {
        Class<Table> clazz = Table.class;
        synchronized (Table.class) {
            Table t = instances.remove(table);
            if (t != null) {
                t.flushTask.cancel(false);
                for (ColumnFamilyStore cfs : t.getColumnFamilyStores()) {
                    t.unloadCf(cfs);
                }
            }
            // ** MonitorExit[var1_1] (shouldn't be in output)
            return t;
        }
    }

    public Collection<ColumnFamilyStore> getColumnFamilyStores() {
        return Collections.unmodifiableCollection(this.columnFamilyStores.values());
    }

    public ColumnFamilyStore getColumnFamilyStore(String cfName) {
        Integer id = CFMetaData.getId(this.name, cfName);
        if (id == null) {
            throw new IllegalArgumentException(String.format("Unknown table/cf pair (%s.%s)", this.name, cfName));
        }
        return this.getColumnFamilyStore(id);
    }

    public ColumnFamilyStore getColumnFamilyStore(Integer id) {
        ColumnFamilyStore cfs = this.columnFamilyStores.get(id);
        if (cfs == null) {
            throw new IllegalArgumentException("Unknown CF " + id);
        }
        return cfs;
    }

    public void forceCleanup(NodeId.OneShotRenewer renewer) throws IOException, ExecutionException, InterruptedException {
        if (this.name.equals(SYSTEM_TABLE)) {
            throw new UnsupportedOperationException("Cleanup of the system table is neither necessary nor wise");
        }
        ArrayList<ColumnFamilyStore> sortedColumnFamilies = new ArrayList<ColumnFamilyStore>(this.columnFamilyStores.values());
        Collections.sort(sortedColumnFamilies, new Comparator<ColumnFamilyStore>(){

            @Override
            public int compare(ColumnFamilyStore cf1, ColumnFamilyStore cf2) {
                long diff = cf1.getTotalDiskSpaceUsed() - cf2.getTotalDiskSpaceUsed();
                if (diff > 0L) {
                    return 1;
                }
                if (diff < 0L) {
                    return -1;
                }
                return cf1.columnFamily.compareTo(cf2.columnFamily);
            }
        });
        for (ColumnFamilyStore cfs : sortedColumnFamilies) {
            cfs.forceCleanup(renewer);
        }
    }

    public void snapshot(String snapshotName) {
        for (ColumnFamilyStore cfStore : this.columnFamilyStores.values()) {
            cfStore.snapshot(snapshotName);
        }
    }

    public static String getTimestampedSnapshotName(String clientSuppliedName) {
        String snapshotName = Long.toString(System.currentTimeMillis());
        if (clientSuppliedName != null && !clientSuppliedName.equals("")) {
            snapshotName = snapshotName + "-" + clientSuppliedName;
        }
        return snapshotName;
    }

    public boolean snapshotExists(String snapshotName) {
        for (String dataDirPath : DatabaseDescriptor.getAllDataFileLocations()) {
            String snapshotPath = dataDirPath + File.separator + this.name + File.separator + SNAPSHOT_SUBDIR_NAME + File.separator + snapshotName;
            File snapshot = new File(snapshotPath);
            if (!snapshot.exists()) continue;
            return true;
        }
        return false;
    }

    public void clearSnapshot(String tag) throws IOException {
        for (String dataDirPath : DatabaseDescriptor.getAllDataFileLocations()) {
            String snapshotPath = dataDirPath + File.separator + this.name + File.separator + SNAPSHOT_SUBDIR_NAME + File.separator + tag;
            File snapshotDir = new File(snapshotPath);
            if (!snapshotDir.exists()) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("Removing snapshot directory " + snapshotPath);
            }
            FileUtils.deleteRecursive(snapshotDir);
        }
    }

    public List<SSTableReader> getAllSSTables() {
        ArrayList<SSTableReader> list = new ArrayList<SSTableReader>();
        for (ColumnFamilyStore cfStore : this.columnFamilyStores.values()) {
            list.addAll(cfStore.getSSTables());
        }
        return list;
    }

    private Table(String table) {
        this.name = table;
        KSMetaData ksm = DatabaseDescriptor.getKSMetaData(table);
        assert (ksm != null) : "Unknown keyspace " + table;
        try {
            this.createReplicationStrategy(ksm);
        }
        catch (ConfigurationException e) {
            throw new RuntimeException(e);
        }
        this.indexLocks = new Object[DatabaseDescriptor.getConcurrentWriters() * 128];
        for (int i = 0; i < this.indexLocks.length; ++i) {
            this.indexLocks[i] = new Object();
        }
        for (String dataDir : DatabaseDescriptor.getAllDataFileLocations()) {
            try {
                File streamingDir;
                String keyspaceDir = dataDir + File.separator + table;
                if (!StorageService.instance.isClientMode()) {
                    FileUtils.createDirectory(keyspaceDir);
                }
                if (!(streamingDir = new File(keyspaceDir, "stream")).exists()) continue;
                FileUtils.deleteRecursive(streamingDir);
            }
            catch (IOException ex) {
                throw new IOError(ex);
            }
        }
        for (CFMetaData cfm : new ArrayList<CFMetaData>(DatabaseDescriptor.getTableDefinition(table).cfMetaData().values())) {
            logger.debug("Initializing {}.{}", (Object)this.name, (Object)cfm.cfName);
            this.initCf(cfm.cfId, cfm.cfName);
        }
        Runnable runnable = new Runnable(){

            @Override
            public void run() {
                for (ColumnFamilyStore cfs : Table.this.columnFamilyStores.values()) {
                    cfs.forceFlushIfExpired();
                }
            }
        };
        this.flushTask = StorageService.scheduledTasks.scheduleWithFixedDelay(runnable, 10L, 10L, TimeUnit.SECONDS);
    }

    public void createReplicationStrategy(KSMetaData ksm) throws ConfigurationException {
        if (this.replicationStrategy != null) {
            StorageService.instance.getTokenMetadata().unregister(this.replicationStrategy);
        }
        this.replicationStrategy = AbstractReplicationStrategy.createReplicationStrategy(ksm.name, ksm.strategyClass, StorageService.instance.getTokenMetadata(), DatabaseDescriptor.getEndpointSnitch(), ksm.strategyOptions);
    }

    public void dropCf(Integer cfId) throws IOException {
        assert (this.columnFamilyStores.containsKey(cfId));
        ColumnFamilyStore cfs = this.columnFamilyStores.remove(cfId);
        if (cfs == null) {
            return;
        }
        this.unloadCf(cfs);
        cfs.removeAllSSTables();
    }

    private void unloadCf(ColumnFamilyStore cfs) throws IOException {
        try {
            cfs.forceBlockingFlush();
        }
        catch (ExecutionException e) {
            throw new IOException(e);
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
        cfs.unregisterMBean();
    }

    public void initCf(Integer cfId, String cfName) {
        assert (!this.columnFamilyStores.containsKey(cfId)) : String.format("tried to init %s as %s, but already used by %s", cfName, cfId, this.columnFamilyStores.get(cfId));
        this.columnFamilyStores.put(cfId, ColumnFamilyStore.createColumnFamilyStore(this, cfName));
    }

    public void renameCf(Integer cfId, String newName) throws IOException {
        assert (this.columnFamilyStores.containsKey(cfId));
        ColumnFamilyStore cfs = this.columnFamilyStores.remove(cfId);
        this.unloadCf(cfs);
        cfs.renameSSTables(newName);
        this.initCf(cfId, newName);
    }

    public Row getRow(QueryFilter filter) throws IOException {
        ColumnFamilyStore cfStore = this.getColumnFamilyStore(filter.getColumnFamilyName());
        ColumnFamily columnFamily = cfStore.getColumnFamily(filter);
        return new Row(filter.key, columnFamily);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void apply(RowMutation mutation, boolean writeCommitLog) throws IOException {
        List<Memtable> memtablesToFlush = Collections.emptyList();
        if (logger.isDebugEnabled()) {
            logger.debug("applying mutation of row {}", (Object)ByteBufferUtil.bytesToHex(mutation.key()));
        }
        switchLock.readLock().lock();
        try {
            if (writeCommitLog) {
                CommitLog.instance.add(mutation);
            }
            DecoratedKey key = StorageService.getPartitioner().decorateKey(mutation.key());
            for (ColumnFamily cf : mutation.getColumnFamilies()) {
                ColumnFamilyStore cfs = this.columnFamilyStores.get(cf.id());
                if (cfs == null) {
                    logger.error("Attempting to mutate non-existant column family " + cf.id());
                    continue;
                }
                TreeSet<ByteBuffer> mutatedIndexedColumns = null;
                for (ByteBuffer column : cfs.getIndexedColumns()) {
                    if (!cf.getColumnNames().contains(column) && !cf.isMarkedForDelete()) continue;
                    if (mutatedIndexedColumns == null) {
                        mutatedIndexedColumns = new TreeSet<ByteBuffer>();
                    }
                    mutatedIndexedColumns.add(column);
                    if (!logger.isDebugEnabled()) continue;
                    ByteBuffer value = cf.getColumn(column) == null ? null : cf.getColumn(column).value();
                    logger.debug(String.format("mutating indexed column %s value %s", cf.getComparator().getString(column), value == null ? "null" : ByteBufferUtil.bytesToHex(value)));
                }
                Object object = this.indexLockFor(mutation.key());
                synchronized (object) {
                    Memtable fullMemtable;
                    ColumnFamily oldIndexedColumns = null;
                    if (mutatedIndexedColumns != null) {
                        oldIndexedColumns = Table.readCurrentIndexedColumns(key, cfs, mutatedIndexedColumns);
                        logger.debug("Pre-mutation index row is {}", (Object)oldIndexedColumns);
                        Table.ignoreObsoleteMutations(cf, mutatedIndexedColumns, oldIndexedColumns);
                    }
                    if ((fullMemtable = cfs.apply(key, cf)) != null) {
                        memtablesToFlush = Table.addFullMemtable(memtablesToFlush, fullMemtable);
                    }
                    if (mutatedIndexedColumns != null) {
                        Table.applyIndexUpdates(mutation.key(), cf, cfs, mutatedIndexedColumns, oldIndexedColumns);
                    }
                }
            }
        }
        finally {
            switchLock.readLock().unlock();
        }
        Iterator i$ = memtablesToFlush.iterator();
        while (i$.hasNext()) {
            Memtable memtable = (Memtable)i$.next();
            memtable.cfs.maybeSwitchMemtable(memtable, writeCommitLog);
        }
        return;
    }

    private static List<Memtable> addFullMemtable(List<Memtable> memtablesToFlush, Memtable fullMemtable) {
        if (memtablesToFlush.isEmpty()) {
            memtablesToFlush = new ArrayList<Memtable>(2);
        }
        memtablesToFlush.add(fullMemtable);
        return memtablesToFlush;
    }

    private static void ignoreObsoleteMutations(ColumnFamily cf, SortedSet<ByteBuffer> mutatedIndexedColumns, ColumnFamily oldIndexedColumns) {
        if (oldIndexedColumns == null) {
            return;
        }
        ColumnFamily cf2 = cf.cloneMe();
        for (IColumn oldColumn : oldIndexedColumns) {
            cf2.addColumn(oldColumn);
        }
        ColumnFamily resolved = ColumnFamilyStore.removeDeleted(cf2, Integer.MAX_VALUE);
        for (IColumn oldColumn : oldIndexedColumns) {
            IColumn resolvedColumn = resolved == null ? null : resolved.getColumn(oldColumn.name());
            if (resolvedColumn == null || !resolvedColumn.equals(oldColumn)) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("ignoring obsolete mutation of " + cf.getComparator().getString(oldColumn.name()));
            }
            cf.remove(oldColumn.name());
            mutatedIndexedColumns.remove(oldColumn.name());
            oldIndexedColumns.remove(oldColumn.name());
        }
    }

    private static ColumnFamily readCurrentIndexedColumns(DecoratedKey<?> key, ColumnFamilyStore cfs, SortedSet<ByteBuffer> mutatedIndexedColumns) {
        QueryFilter filter = QueryFilter.getNamesFilter(key, new QueryPath(cfs.getColumnFamilyName()), mutatedIndexedColumns);
        return cfs.getColumnFamily(filter);
    }

    private static List<Memtable> applyIndexUpdates(ByteBuffer key, ColumnFamily cf, ColumnFamilyStore cfs, SortedSet<ByteBuffer> mutatedIndexedColumns, ColumnFamily oldIndexedColumns) {
        List<Memtable> fullMemtables = Collections.emptyList();
        for (ByteBuffer columnName : mutatedIndexedColumns) {
            Memtable fullMemtable;
            IColumn column = cf.getColumn(columnName);
            if (column == null || column.isMarkedForDelete()) continue;
            DecoratedKey<LocalToken> valueKey = cfs.getIndexKeyFor(columnName, column.value());
            ColumnFamily cfi = cfs.newIndexedColumnFamily(columnName);
            if (column instanceof ExpiringColumn) {
                ExpiringColumn ec = (ExpiringColumn)column;
                cfi.addColumn(new ExpiringColumn(key, ByteBufferUtil.EMPTY_BYTE_BUFFER, ec.timestamp, ec.getTimeToLive(), ec.getLocalDeletionTime()));
            } else {
                cfi.addColumn(new Column(key, ByteBufferUtil.EMPTY_BYTE_BUFFER, column.timestamp()));
            }
            if (logger.isDebugEnabled()) {
                logger.debug("applying index row {}:{}", valueKey, (Object)cfi);
            }
            if ((fullMemtable = cfs.getIndexedColumnFamilyStore(columnName).apply(valueKey, cfi)) == null) continue;
            fullMemtables = Table.addFullMemtable(fullMemtables, fullMemtable);
        }
        if (oldIndexedColumns != null) {
            int localDeletionTime = (int)(System.currentTimeMillis() / 1000L);
            for (Map.Entry<ByteBuffer, IColumn> entry : oldIndexedColumns.getColumnsMap().entrySet()) {
                ByteBuffer columnName = entry.getKey();
                IColumn column = entry.getValue();
                if (column.isMarkedForDelete()) continue;
                DecoratedKey<LocalToken> valueKey = cfs.getIndexKeyFor(columnName, column.value());
                ColumnFamily cfi = cfs.newIndexedColumnFamily(columnName);
                cfi.addTombstone(key, localDeletionTime, column.timestamp());
                Memtable fullMemtable = cfs.getIndexedColumnFamilyStore(columnName).apply(valueKey, cfi);
                if (logger.isDebugEnabled()) {
                    logger.debug("applying index tombstones {}:{}", valueKey, (Object)cfi);
                }
                if (fullMemtable == null) continue;
                fullMemtables = Table.addFullMemtable(fullMemtables, fullMemtable);
            }
        }
        return fullMemtables;
    }

    public static void cleanupIndexEntry(ColumnFamilyStore cfs, ByteBuffer key, IColumn column) {
        if (column.isMarkedForDelete()) {
            return;
        }
        int localDeletionTime = (int)(System.currentTimeMillis() / 1000L);
        DecoratedKey<LocalToken> valueKey = cfs.getIndexKeyFor(column.name(), column.value());
        ColumnFamily cfi = cfs.newIndexedColumnFamily(column.name());
        cfi.addTombstone(key, localDeletionTime, column.timestamp());
        Memtable fullMemtable = cfs.getIndexedColumnFamilyStore(column.name()).apply(valueKey, cfi);
        if (logger.isDebugEnabled()) {
            logger.debug("removed index entry for cleaned-up value {}:{}", valueKey, (Object)cfi);
        }
        if (fullMemtable != null) {
            fullMemtable.cfs.maybeSwitchMemtable(fullMemtable, false);
        }
    }

    public IndexBuilder createIndexBuilder(ColumnFamilyStore cfs, SortedSet<ByteBuffer> columns, ReducingKeyIterator iter) {
        return new IndexBuilder(cfs, columns, iter);
    }

    public AbstractReplicationStrategy getReplicationStrategy() {
        return this.replicationStrategy;
    }

    private Object indexLockFor(ByteBuffer key) {
        return this.indexLocks[Math.abs(key.hashCode() % this.indexLocks.length)];
    }

    public List<Future<?>> flush() throws IOException {
        ArrayList futures = new ArrayList();
        for (Integer cfId : this.columnFamilyStores.keySet()) {
            Object future = this.columnFamilyStores.get(cfId).forceFlush();
            if (future == null) continue;
            futures.add((Future<?>)future);
        }
        return futures;
    }

    void load(RowMutation rowMutation) throws IOException {
        DecoratedKey key = StorageService.getPartitioner().decorateKey(rowMutation.key());
        for (ColumnFamily columnFamily : rowMutation.getColumnFamilies()) {
            Collection<IColumn> columns = columnFamily.getSortedColumns();
            for (IColumn column : columns) {
                ColumnFamilyStore cfStore = this.columnFamilyStores.get(ByteBufferUtil.toInt(column.name()));
                cfStore.applyBinary(key, column.value());
            }
        }
    }

    public String getDataFileLocation(long expectedCompactedFileSize) {
        String path = DatabaseDescriptor.getDataFileLocationForTable(this.name, expectedCompactedFileSize);
        if (path == null) {
            StorageService.instance.requestGC();
            try {
                Thread.sleep(20000L);
            }
            catch (InterruptedException e) {
                throw new AssertionError((Object)e);
            }
            path = DatabaseDescriptor.getDataFileLocationForTable(this.name, expectedCompactedFileSize);
        }
        return path;
    }

    public static String getSnapshotPath(String dataDirPath, String tableName, String snapshotName) {
        return dataDirPath + File.separator + tableName + File.separator + SNAPSHOT_SUBDIR_NAME + File.separator + snapshotName;
    }

    public static Iterable<Table> all() {
        Function<String, Table> transformer = new Function<String, Table>(){

            public Table apply(String tableName) {
                return Table.open(tableName);
            }
        };
        return Iterables.transform(DatabaseDescriptor.getTables(), (Function)transformer);
    }

    public void truncate(String cfname) throws InterruptedException, ExecutionException, IOException {
        logger.debug("Truncating...");
        ColumnFamilyStore cfs = this.getColumnFamilyStore(cfname);
        cfs.truncate().get();
        logger.debug("Truncation done.");
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(name='" + this.name + "')";
    }

    static {
        if (!StorageService.instance.isClientMode()) {
            try {
                DatabaseDescriptor.createAllDirectories();
            }
            catch (IOException ex) {
                throw new IOError(ex);
            }
        }
        instances = new NonBlockingHashMap();
    }

    public class IndexBuilder
    implements CompactionInfo.Holder {
        private final ColumnFamilyStore cfs;
        private final SortedSet<ByteBuffer> columns;
        private final ReducingKeyIterator iter;

        public IndexBuilder(ColumnFamilyStore cfs, SortedSet<ByteBuffer> columns, ReducingKeyIterator iter) {
            this.cfs = cfs;
            this.columns = columns;
            this.iter = iter;
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            return new CompactionInfo(this.cfs.table.name, this.cfs.columnFamily, CompactionType.INDEX_BUILD, this.iter.getTotalBytes(), this.iter.getBytesRead());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void build() {
            while (this.iter.hasNext()) {
                DecoratedKey key = this.iter.next();
                logger.debug("Indexing row {} ", (Object)key);
                List memtablesToFlush = Collections.emptyList();
                switchLock.readLock().lock();
                try {
                    Object object = Table.this.indexLockFor(key.key);
                    synchronized (object) {
                        ColumnFamily cf = Table.readCurrentIndexedColumns(key, this.cfs, this.columns);
                        if (cf != null) {
                            memtablesToFlush = Table.applyIndexUpdates(key.key, cf, this.cfs, cf.getColumnNames(), null);
                        }
                    }
                }
                finally {
                    switchLock.readLock().unlock();
                }
                for (Memtable memtable : memtablesToFlush) {
                    memtable.cfs.maybeSwitchMemtable(memtable, false);
                }
            }
            try {
                this.iter.close();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }
}

