/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ConfigurationException;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.KSMetaData;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ArrayBackedSortedColumns;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.IColumn;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.filter.QueryFilter;
import org.apache.cassandra.db.filter.QueryPath;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.locator.AbstractReplicationStrategy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.NodeId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Table {
    public static final String SYSTEM_TABLE = "system";
    private static final Logger logger = LoggerFactory.getLogger(Table.class);
    public static final ReentrantReadWriteLock switchLock = new ReentrantReadWriteLock();
    public final String name;
    private final Map<Integer, ColumnFamilyStore> columnFamilyStores = new ConcurrentHashMap<Integer, ColumnFamilyStore>();
    private final Object[] indexLocks;
    private volatile AbstractReplicationStrategy replicationStrategy;

    public static Table open(String table) {
        return Table.open(table, Schema.instance);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static Table open(String table, Schema schema) {
        Table tableInstance = schema.getTableInstance(table);
        if (tableInstance != null) return tableInstance;
        Class<Table> clazz = Table.class;
        synchronized (Table.class) {
            tableInstance = schema.getTableInstance(table);
            if (tableInstance != null) return tableInstance;
            tableInstance = new Table(table);
            schema.storeTableInstance(tableInstance);
            for (ColumnFamilyStore cfs : tableInstance.getColumnFamilyStores()) {
                cfs.initRowCache();
            }
            // ** MonitorExit[var3_3] (shouldn't be in output)
            return tableInstance;
        }
    }

    public static Table clear(String table) throws IOException {
        return Table.clear(table, Schema.instance);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Table clear(String table, Schema schema) throws IOException {
        Class<Table> clazz = Table.class;
        synchronized (Table.class) {
            Table t = schema.removeTableInstance(table);
            if (t != null) {
                for (ColumnFamilyStore cfs : t.getColumnFamilyStores()) {
                    t.unloadCf(cfs);
                }
            }
            // ** MonitorExit[var2_2] (shouldn't be in output)
            return t;
        }
    }

    public Collection<ColumnFamilyStore> getColumnFamilyStores() {
        return Collections.unmodifiableCollection(this.columnFamilyStores.values());
    }

    public ColumnFamilyStore getColumnFamilyStore(String cfName) {
        Integer id = Schema.instance.getId(this.name, cfName);
        if (id == null) {
            throw new IllegalArgumentException(String.format("Unknown table/cf pair (%s.%s)", this.name, cfName));
        }
        return this.getColumnFamilyStore(id);
    }

    public ColumnFamilyStore getColumnFamilyStore(Integer id) {
        ColumnFamilyStore cfs = this.columnFamilyStores.get(id);
        if (cfs == null) {
            throw new IllegalArgumentException("Unknown CF " + id);
        }
        return cfs;
    }

    public void forceCleanup(NodeId.OneShotRenewer renewer) throws IOException, ExecutionException, InterruptedException {
        if (this.name.equals(SYSTEM_TABLE)) {
            throw new UnsupportedOperationException("Cleanup of the system table is neither necessary nor wise");
        }
        ArrayList<ColumnFamilyStore> sortedColumnFamilies = new ArrayList<ColumnFamilyStore>(this.columnFamilyStores.values());
        Collections.sort(sortedColumnFamilies, new Comparator<ColumnFamilyStore>(){

            @Override
            public int compare(ColumnFamilyStore cf1, ColumnFamilyStore cf2) {
                long diff = cf1.getTotalDiskSpaceUsed() - cf2.getTotalDiskSpaceUsed();
                if (diff > 0L) {
                    return 1;
                }
                if (diff < 0L) {
                    return -1;
                }
                return cf1.columnFamily.compareTo(cf2.columnFamily);
            }
        });
        for (ColumnFamilyStore cfs : sortedColumnFamilies) {
            cfs.forceCleanup(renewer);
        }
    }

    public void snapshot(String snapshotName) {
        assert (snapshotName != null);
        for (ColumnFamilyStore cfStore : this.columnFamilyStores.values()) {
            cfStore.snapshot(snapshotName);
        }
    }

    public static String getTimestampedSnapshotName(String clientSuppliedName) {
        String snapshotName = Long.toString(System.currentTimeMillis());
        if (clientSuppliedName != null && !clientSuppliedName.equals("")) {
            snapshotName = snapshotName + "-" + clientSuppliedName;
        }
        return snapshotName;
    }

    public boolean snapshotExists(String snapshotName) {
        assert (snapshotName != null);
        for (ColumnFamilyStore cfStore : this.columnFamilyStores.values()) {
            if (!cfStore.snapshotExists(snapshotName)) continue;
            return true;
        }
        return false;
    }

    public void clearSnapshot(String snapshotName) throws IOException {
        for (ColumnFamilyStore cfStore : this.columnFamilyStores.values()) {
            cfStore.clearSnapshot(snapshotName);
        }
    }

    public List<SSTableReader> getAllSSTables() {
        ArrayList<SSTableReader> list = new ArrayList<SSTableReader>();
        for (ColumnFamilyStore cfStore : this.columnFamilyStores.values()) {
            list.addAll(cfStore.getSSTables());
        }
        return list;
    }

    private Table(String table) {
        this.name = table;
        KSMetaData ksm = Schema.instance.getKSMetaData(table);
        assert (ksm != null) : "Unknown keyspace " + table;
        try {
            this.createReplicationStrategy(ksm);
        }
        catch (ConfigurationException e) {
            throw new RuntimeException(e);
        }
        this.indexLocks = new Object[DatabaseDescriptor.getConcurrentWriters() * 128];
        for (int i = 0; i < this.indexLocks.length; ++i) {
            this.indexLocks[i] = new Object();
        }
        for (CFMetaData cfm : new ArrayList<CFMetaData>(Schema.instance.getTableDefinition(table).cfMetaData().values())) {
            logger.debug("Initializing {}.{}", (Object)this.name, (Object)cfm.cfName);
            this.initCf(cfm.cfId, cfm.cfName);
        }
    }

    public void createReplicationStrategy(KSMetaData ksm) throws ConfigurationException {
        if (this.replicationStrategy != null) {
            StorageService.instance.getTokenMetadata().unregister(this.replicationStrategy);
        }
        this.replicationStrategy = AbstractReplicationStrategy.createReplicationStrategy(ksm.name, ksm.strategyClass, StorageService.instance.getTokenMetadata(), DatabaseDescriptor.getEndpointSnitch(), ksm.strategyOptions);
    }

    public void dropCf(Integer cfId) throws IOException {
        assert (this.columnFamilyStores.containsKey(cfId));
        ColumnFamilyStore cfs = this.columnFamilyStores.remove(cfId);
        if (cfs == null) {
            return;
        }
        this.unloadCf(cfs);
    }

    private void unloadCf(ColumnFamilyStore cfs) throws IOException {
        try {
            cfs.forceBlockingFlush();
        }
        catch (ExecutionException e) {
            throw new IOException(e);
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
        cfs.invalidate();
    }

    public void initCf(Integer cfId, String cfName) {
        assert (!this.columnFamilyStores.containsKey(cfId)) : String.format("tried to init %s as %s, but already used by %s", cfName, cfId, this.columnFamilyStores.get(cfId));
        this.columnFamilyStores.put(cfId, ColumnFamilyStore.createColumnFamilyStore(this, cfName));
    }

    public Row getRow(QueryFilter filter) throws IOException {
        ColumnFamilyStore cfStore = this.getColumnFamilyStore(filter.getColumnFamilyName());
        ColumnFamily columnFamily = cfStore.getColumnFamily(filter, ArrayBackedSortedColumns.factory());
        return new Row(filter.key, columnFamily);
    }

    public void apply(RowMutation mutation, boolean writeCommitLog) throws IOException {
        this.apply(mutation, writeCommitLog, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void apply(RowMutation mutation, boolean writeCommitLog, boolean updateIndexes) throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("applying mutation of row {}", (Object)ByteBufferUtil.bytesToHex(mutation.key()));
        }
        switchLock.readLock().lock();
        try {
            if (writeCommitLog) {
                CommitLog.instance.add(mutation);
            }
            DecoratedKey key = StorageService.getPartitioner().decorateKey(mutation.key());
            for (ColumnFamily cf : mutation.getColumnFamilies()) {
                ColumnFamilyStore cfs = this.columnFamilyStores.get(cf.id());
                if (cfs == null) {
                    logger.error("Attempting to mutate non-existant column family " + cf.id());
                    continue;
                }
                TreeSet<ByteBuffer> mutatedIndexedColumns = null;
                if (updateIndexes) {
                    for (ByteBuffer column : cfs.indexManager.getIndexedColumns()) {
                        if (!cf.getColumnNames().contains(column) && !cf.isMarkedForDelete()) continue;
                        if (mutatedIndexedColumns == null) {
                            mutatedIndexedColumns = new TreeSet<ByteBuffer>();
                        }
                        mutatedIndexedColumns.add(column);
                        if (!logger.isDebugEnabled()) continue;
                        ByteBuffer value = cf.getColumn(column) == null ? null : cf.getColumn(column).value();
                        logger.debug(String.format("mutating indexed column %s value %s", cf.getComparator().getString(column), value == null ? "null" : ByteBufferUtil.bytesToHex(value)));
                    }
                }
                if (mutatedIndexedColumns == null) {
                    cfs.apply(key, cf);
                    continue;
                }
                Object object = this.indexLockFor(mutation.key());
                synchronized (object) {
                    ColumnFamily oldIndexedColumns = Table.readCurrentIndexedColumns(key, cfs, mutatedIndexedColumns);
                    logger.debug("Pre-mutation index row is {}", (Object)oldIndexedColumns);
                    Table.ignoreObsoleteMutations(cf, (SortedSet<ByteBuffer>)mutatedIndexedColumns, oldIndexedColumns);
                    cfs.apply(key, cf);
                    cfs.indexManager.applyIndexUpdates(mutation.key(), cf, mutatedIndexedColumns, oldIndexedColumns);
                }
            }
            return;
        }
        finally {
            switchLock.readLock().unlock();
        }
    }

    private static void ignoreObsoleteMutations(ColumnFamily cf, SortedSet<ByteBuffer> mutatedIndexedColumns, ColumnFamily oldIndexedColumns) {
        if (oldIndexedColumns == null) {
            return;
        }
        Iterator iter = mutatedIndexedColumns.iterator();
        while (iter.hasNext()) {
            boolean obsoleteColumn;
            ByteBuffer name = (ByteBuffer)iter.next();
            IColumn newColumn = cf.getColumn(name);
            if (newColumn != null && cf.isMarkedForDelete() && newColumn.timestamp() <= cf.getMarkedForDeleteAt()) {
                newColumn = null;
            }
            IColumn oldColumn = oldIndexedColumns.getColumn(name);
            boolean bothDeleted = !(newColumn != null && !newColumn.isMarkedForDelete() || oldColumn != null && !oldColumn.isMarkedForDelete());
            boolean obsoleteRowTombstone = newColumn == null && oldColumn != null && cf.getMarkedForDeleteAt() < oldColumn.timestamp();
            boolean bl = obsoleteColumn = newColumn != null && (newColumn.timestamp() <= oldIndexedColumns.getMarkedForDeleteAt() || oldColumn != null && oldColumn.reconcile(newColumn) == oldColumn);
            if (!bothDeleted && !obsoleteRowTombstone && !obsoleteColumn) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("skipping index update for obsolete mutation of " + cf.getComparator().getString(name));
            }
            iter.remove();
            oldIndexedColumns.remove(name);
        }
    }

    private static ColumnFamily readCurrentIndexedColumns(DecoratedKey<?> key, ColumnFamilyStore cfs, SortedSet<ByteBuffer> mutatedIndexedColumns) {
        QueryFilter filter = QueryFilter.getNamesFilter(key, new QueryPath(cfs.getColumnFamilyName()), mutatedIndexedColumns);
        return cfs.getColumnFamily(filter);
    }

    public AbstractReplicationStrategy getReplicationStrategy() {
        return this.replicationStrategy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void indexRow(DecoratedKey<?> key, ColumnFamilyStore cfs, SortedSet<ByteBuffer> indexedColumns) {
        if (logger.isDebugEnabled()) {
            logger.debug("Indexing row {} ", (Object)cfs.metadata.getKeyValidator().getString(key.key));
        }
        switchLock.readLock().lock();
        try {
            Object object = cfs.table.indexLockFor(key.key);
            synchronized (object) {
                ColumnFamily cf = Table.readCurrentIndexedColumns(key, cfs, indexedColumns);
                if (cf != null) {
                    try {
                        cfs.indexManager.applyIndexUpdates(key.key, cf, cf.getColumnNames(), null);
                    }
                    catch (IOException e) {
                        throw new IOError(e);
                    }
                }
            }
        }
        finally {
            switchLock.readLock().unlock();
        }
    }

    private Object indexLockFor(ByteBuffer key) {
        return this.indexLocks[Math.abs(key.hashCode() % this.indexLocks.length)];
    }

    public List<Future<?>> flush() throws IOException {
        ArrayList futures = new ArrayList();
        for (Integer cfId : this.columnFamilyStores.keySet()) {
            Future<?> future = this.columnFamilyStores.get(cfId).forceFlush();
            if (future == null) continue;
            futures.add(future);
        }
        return futures;
    }

    public static Iterable<Table> all() {
        Function<String, Table> transformer = new Function<String, Table>(){

            public Table apply(String tableName) {
                return Table.open(tableName);
            }
        };
        return Iterables.transform(Schema.instance.getTables(), (Function)transformer);
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(name='" + this.name + "')";
    }

    static {
        if (!StorageService.instance.isClientMode()) {
            try {
                DatabaseDescriptor.createAllDirectories();
            }
            catch (IOException ex) {
                throw new IOError(ex);
            }
        }
    }
}

