/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.collect.ImmutableMap;
import java.io.File;
import java.io.FileFilter;
import java.io.IOError;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.KSMetaData;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableDeletingTask;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.CLibrary;
import org.apache.cassandra.utils.Pair;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Directories {
    private static Logger logger = LoggerFactory.getLogger(Directories.class);
    public static final String BACKUPS_SUBDIR = "backups";
    public static final String SNAPSHOT_SUBDIR = "snapshots";
    public static final char SECONDARY_INDEX_NAME_SEPARATOR = '.';
    public static final File[] dataFileLocations;
    private final String tablename;
    private final String cfname;
    private final File[] sstableDirectories;

    public static Directories create(String tablename, String cfname) {
        int idx = cfname.indexOf(46);
        if (idx > 0) {
            return new Directories(tablename, cfname, cfname.substring(0, idx));
        }
        return new Directories(tablename, cfname, cfname);
    }

    private Directories(String tablename, String cfname, String directoryName) {
        this.tablename = tablename;
        this.cfname = cfname;
        this.sstableDirectories = new File[dataFileLocations.length];
        for (int i = 0; i < dataFileLocations.length; ++i) {
            this.sstableDirectories[i] = new File(dataFileLocations[i], Directories.join(tablename, directoryName));
        }
        if (!StorageService.instance.isClientMode()) {
            try {
                for (File dir : this.sstableDirectories) {
                    FileUtils.createDirectory(dir);
                }
            }
            catch (IOException e) {
                throw new IOError(e);
            }
        }
    }

    public File getDirectoryForNewSSTables(long estimatedSize) {
        File path = this.getLocationWithMaximumAvailableSpace(estimatedSize);
        if (!(path != null || DatabaseDescriptor.getDiskAccessMode() != Config.DiskAccessMode.mmap && DatabaseDescriptor.getIndexAccessMode() != Config.DiskAccessMode.mmap || FileUtils.isCleanerAvailable())) {
            logger.info("Forcing GC to free up disk space.  Upgrade to the Oracle JVM to avoid this");
            StorageService.instance.requestGC();
            SSTableDeletingTask.rescheduleFailedTasks();
            try {
                Thread.sleep(10000L);
            }
            catch (InterruptedException e) {
                throw new AssertionError((Object)e);
            }
            path = this.getLocationWithMaximumAvailableSpace(estimatedSize);
        }
        return path;
    }

    public File getLocationWithMaximumAvailableSpace(long estimatedSize) {
        long maxFreeDisk = 0L;
        File maxLocation = null;
        for (File dir : this.sstableDirectories) {
            if (maxFreeDisk >= dir.getUsableSpace()) continue;
            maxFreeDisk = dir.getUsableSpace();
            maxLocation = dir;
        }
        maxFreeDisk = (long)(0.9 * (double)maxFreeDisk);
        logger.debug(String.format("expected data files size is %d; largest free partition (%s) has %d bytes free", estimatedSize, maxLocation, maxFreeDisk));
        if (estimatedSize < maxFreeDisk) {
            return maxLocation;
        }
        return null;
    }

    public static File getSnapshotDirectory(Descriptor desc, String snapshotName) {
        return Directories.getOrCreate(desc.directory, SNAPSHOT_SUBDIR, snapshotName);
    }

    public static File getBackupsDirectory(Descriptor desc) {
        return Directories.getOrCreate(desc.directory, BACKUPS_SUBDIR);
    }

    public SSTableLister sstableLister() {
        return new SSTableLister();
    }

    public File tryGetLeveledManifest() {
        for (File dir : this.sstableDirectories) {
            File manifestFile = new File(dir, this.cfname + ".json");
            if (!manifestFile.exists()) continue;
            logger.debug("Found manifest at {}", (Object)manifestFile);
            return manifestFile;
        }
        logger.debug("No level manifest found");
        return null;
    }

    public File getOrCreateLeveledManifest() {
        File manifestFile = this.tryGetLeveledManifest();
        if (manifestFile == null) {
            manifestFile = new File(this.sstableDirectories[0], this.cfname + ".json");
        }
        return manifestFile;
    }

    public void snapshotLeveledManifest(String snapshotName) throws IOException {
        File manifest = this.tryGetLeveledManifest();
        if (manifest != null) {
            File snapshotDirectory = Directories.getOrCreate(manifest.getParentFile(), SNAPSHOT_SUBDIR, snapshotName);
            CLibrary.createHardLink(manifest, new File(snapshotDirectory, manifest.getName()));
        }
    }

    public boolean snapshotExists(String snapshotName) {
        for (File dir : this.sstableDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, snapshotName));
            if (!snapshotDir.exists()) continue;
            return true;
        }
        return false;
    }

    public void clearSnapshot(String snapshotName) throws IOException {
        String tag = snapshotName == null ? "" : snapshotName;
        for (File dir : this.sstableDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, tag));
            if (!snapshotDir.exists()) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("Removing snapshot directory " + snapshotDir);
            }
            FileUtils.deleteRecursive(snapshotDir);
        }
    }

    private static File getOrCreate(File base, String ... subdirs) {
        File dir;
        File file = dir = subdirs == null || subdirs.length == 0 ? base : new File(base, Directories.join(subdirs));
        if (dir.exists()) {
            if (!dir.isDirectory()) {
                throw new IOError(new IOException(String.format("Invalid directory path %s: path exists but is not a directory", dir)));
            }
        } else if (!dir.mkdirs()) {
            throw new IOError(new IOException("Unable to create directory " + dir));
        }
        return dir;
    }

    private static String join(String ... s) {
        return StringUtils.join((Object[])s, (String)File.separator);
    }

    public static boolean sstablesNeedsMigration() {
        if (StorageService.instance.isClientMode()) {
            return false;
        }
        boolean hasSystemKeyspace = false;
        for (File location : dataFileLocations) {
            File systemDir = new File(location, "system");
            hasSystemKeyspace |= systemDir.exists() && systemDir.isDirectory();
            File statusCFDir = new File(systemDir, "LocationInfo");
            if (!statusCFDir.exists()) continue;
            return false;
        }
        if (!hasSystemKeyspace) {
            return false;
        }
        int longestLocation = -1;
        try {
            for (File loc : dataFileLocations) {
                longestLocation = Math.max(longestLocation, loc.getCanonicalPath().length());
            }
        }
        catch (IOException e) {
            throw new IOError(e);
        }
        for (KSMetaData ksm : Schema.instance.getTableDefinitions()) {
            String ksname = ksm.name;
            for (Map.Entry<String, CFMetaData> entry : ksm.cfMetaData().entrySet()) {
                String cfname = entry.getKey();
                if (System.getProperty("os.name").startsWith("Windows") && longestLocation + (ksname.length() + cfname.length()) * 2 + 63 > 255) {
                    throw new RuntimeException(String.format("Starting with 1.1, keyspace names and column family names must be less than %s characters long. %s/%s doesn't respect that restriction. Please rename your keyspace/column families to respect that restriction before updating.", 48, ksname, cfname));
                }
                if (ksm.name.length() + cfname.length() + 28 <= 255) continue;
                throw new RuntimeException("Starting with 1.1, the keyspace name is included in data filenames.  For " + ksm.name + "/" + cfname + ", this puts you over the largest possible filename of 255 characters");
            }
        }
        return true;
    }

    public static void migrateSSTables() {
        logger.info("Upgrade from pre-1.1 version detected: migrating sstables to new directory layout");
        for (File location : dataFileLocations) {
            File[] ksDirs;
            if (!location.exists() || !location.isDirectory() || (ksDirs = location.listFiles()) == null) continue;
            for (File ksDir : ksDirs) {
                if (!ksDir.isDirectory()) continue;
                File[] files = ksDir.listFiles();
                if (files != null) {
                    for (File file : files) {
                        Directories.migrateFile(file, ksDir, null);
                    }
                }
                Directories.migrateSnapshots(ksDir);
                Directories.migrateBackups(ksDir);
            }
        }
    }

    private static void migrateSnapshots(File ksDir) {
        File snapshotDir = new File(ksDir, SNAPSHOT_SUBDIR);
        if (!snapshotDir.exists()) {
            return;
        }
        File[] snapshots = snapshotDir.listFiles();
        if (snapshots != null) {
            for (File snapshot : snapshots) {
                if (!snapshot.isDirectory()) continue;
                File[] files = snapshot.listFiles();
                if (files != null) {
                    for (File f : files) {
                        Directories.migrateFile(f, ksDir, Directories.join(SNAPSHOT_SUBDIR, snapshot.getName()));
                    }
                }
                if (snapshot.delete()) continue;
                logger.info("Old snapsot directory {} not deleted by migraation as it is not empty", (Object)snapshot);
            }
        }
        if (!snapshotDir.delete()) {
            logger.info("Old directory {} not deleted by migration as it is not empty", (Object)snapshotDir);
        }
    }

    private static void migrateBackups(File ksDir) {
        File backupDir = new File(ksDir, BACKUPS_SUBDIR);
        if (!backupDir.exists()) {
            return;
        }
        File[] files = backupDir.listFiles();
        if (files != null) {
            for (File f : files) {
                Directories.migrateFile(f, ksDir, BACKUPS_SUBDIR);
            }
        }
        if (!backupDir.delete()) {
            logger.info("Old directory {} not deleted by migration as it is not empty", (Object)backupDir);
        }
    }

    private static void migrateFile(File file, File ksDir, String additionalPath) {
        try {
            if (file.isDirectory()) {
                return;
            }
            String name = file.getName();
            boolean isManifest = name.endsWith(".json");
            String cfname = isManifest ? Directories.getCfNameFromManifest(name) : name.substring(0, name.indexOf(45));
            int idx = cfname.indexOf(46);
            String dirname = idx > 0 ? cfname.substring(0, idx) : cfname;
            File destDir = Directories.getOrCreate(ksDir, dirname, additionalPath);
            File destFile = new File(destDir, isManifest ? name : ksDir.getName() + '-' + name);
            logger.debug(String.format("[upgrade to 1.1] Moving %s to %s", file, destFile));
            FileUtils.renameWithConfirm(file, destFile);
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    private static String getCfNameFromManifest(String name) {
        String withoutExt = name.substring(0, name.length() - ".json".length());
        return withoutExt.endsWith("-old") || withoutExt.endsWith("-tmp") ? withoutExt.substring(0, withoutExt.length() - 4) : withoutExt;
    }

    static void overrideDataDirectoriesForTest(String loc) {
        for (int i = 0; i < dataFileLocations.length; ++i) {
            Directories.dataFileLocations[i] = new File(loc);
        }
    }

    static void resetDataDirectoriesAfterTest() {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataFileLocations[i] = new File(locations[i]);
        }
    }

    static {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        dataFileLocations = new File[locations.length];
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataFileLocations[i] = new File(locations[i]);
        }
    }

    public class SSTableLister {
        private boolean skipCompacted;
        private boolean skipTemporary;
        private boolean includeBackups;
        private boolean onlyBackups;
        private int nbFiles;
        private final Map<Descriptor, Set<Component>> components = new HashMap<Descriptor, Set<Component>>();
        private boolean filtered;
        private String snapshotName;

        public SSTableLister skipCompacted(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.skipCompacted = b;
            return this;
        }

        public SSTableLister skipTemporary(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.skipTemporary = b;
            return this;
        }

        public SSTableLister includeBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.includeBackups = b;
            return this;
        }

        public SSTableLister onlyBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.onlyBackups = b;
            this.includeBackups = b;
            return this;
        }

        public SSTableLister snapshots(String sn) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.snapshotName = sn;
            return this;
        }

        public Map<Descriptor, Set<Component>> list() {
            this.filter();
            return ImmutableMap.copyOf(this.components);
        }

        public List<File> listFiles() {
            this.filter();
            ArrayList<File> l = new ArrayList<File>(this.nbFiles);
            for (Map.Entry<Descriptor, Set<Component>> entry : this.components.entrySet()) {
                for (Component c : entry.getValue()) {
                    l.add(new File(entry.getKey().filenameFor(c)));
                }
            }
            return l;
        }

        private void filter() {
            if (this.filtered) {
                return;
            }
            for (File location : Directories.this.sstableDirectories) {
                if (this.snapshotName != null) {
                    new File(location, Directories.join(new String[]{Directories.SNAPSHOT_SUBDIR, this.snapshotName})).listFiles(this.getFilter());
                    continue;
                }
                if (!this.onlyBackups) {
                    location.listFiles(this.getFilter());
                }
                if (!this.includeBackups) continue;
                new File(location, Directories.BACKUPS_SUBDIR).listFiles(this.getFilter());
            }
            this.filtered = true;
        }

        private FileFilter getFilter() {
            final String sstablePrefix = Directories.this.tablename + '-' + Directories.this.cfname + '-';
            return new FileFilter(){

                @Override
                public boolean accept(File file) {
                    if (file.isDirectory() || !file.getName().startsWith(sstablePrefix)) {
                        return false;
                    }
                    Pair<Descriptor, Component> pair = SSTable.tryComponentFromFilename(file.getParentFile(), file.getName());
                    if (pair == null) {
                        return false;
                    }
                    if (SSTableLister.this.skipCompacted && new File(((Descriptor)pair.left).filenameFor(Component.COMPACTED_MARKER)).exists()) {
                        return false;
                    }
                    if (SSTableLister.this.skipTemporary && ((Descriptor)pair.left).temporary) {
                        return false;
                    }
                    HashSet previous = (HashSet)SSTableLister.this.components.get(pair.left);
                    if (previous == null) {
                        previous = new HashSet();
                        SSTableLister.this.components.put(pair.left, previous);
                    }
                    previous.add(pair.right);
                    SSTableLister.this.nbFiles++;
                    return false;
                }
            };
        }
    }
}

