/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.base.Predicate;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.google.common.collect.Ordering;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.Checksum;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.Column;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ColumnSerializer;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.UnknownColumnFamilyException;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.util.FastByteArrayInputStream;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.PureJavaCrc32;
import org.apache.cassandra.utils.WrappedRunnable;
import org.apache.commons.lang3.StringUtils;
import org.cliffc.high_scale_lib.NonBlockingHashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogReplayer {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogReplayer.class);
    private static final int MAX_OUTSTANDING_REPLAY_COUNT = Integer.getInteger("cassandra.commitlog_max_outstanding_replay_count", 1024);
    private final Set<Keyspace> keyspacesRecovered = new NonBlockingHashSet();
    private final List<Future<?>> futures = new ArrayList();
    private final Map<UUID, AtomicInteger> invalidMutations;
    private final AtomicInteger replayedCount;
    private final Map<UUID, ReplayPosition> cfPositions;
    private final ReplayPosition globalPosition;
    private final Checksum checksum;
    private byte[] buffer = new byte[4096];

    public CommitLogReplayer() {
        this.invalidMutations = new HashMap<UUID, AtomicInteger>();
        this.replayedCount = new AtomicInteger();
        this.checksum = new PureJavaCrc32();
        this.cfPositions = new HashMap<UUID, ReplayPosition>();
        Ordering replayPositionOrdering = Ordering.from(ReplayPosition.comparator);
        for (ColumnFamilyStore cfs : ColumnFamilyStore.all()) {
            ReplayPosition rp = ReplayPosition.getReplayPosition(cfs.getSSTables());
            ReplayPosition truncatedAt = SystemKeyspace.getTruncatedPosition(cfs.metadata.cfId);
            if (truncatedAt != null) {
                rp = (ReplayPosition)replayPositionOrdering.max(Arrays.asList(rp, truncatedAt));
            }
            this.cfPositions.put(cfs.metadata.cfId, rp);
        }
        this.globalPosition = (ReplayPosition)replayPositionOrdering.min(this.cfPositions.values());
        logger.debug("Global replay position is {} from columnfamilies {}", (Object)this.globalPosition, (Object)FBUtilities.toString(this.cfPositions));
    }

    public void recover(File[] clogs) throws IOException {
        for (File file : clogs) {
            this.recover(file);
        }
    }

    public int blockForWrites() {
        for (Map.Entry<UUID, AtomicInteger> entry : this.invalidMutations.entrySet()) {
            logger.info(String.format("Skipped %d mutations from unknown (probably removed) CF with id %s", entry.getValue().intValue(), entry.getKey()));
        }
        FBUtilities.waitOnFutures(this.futures);
        logger.debug("Finished waiting on mutations from recovery");
        this.futures.clear();
        for (Keyspace keyspace : this.keyspacesRecovered) {
            this.futures.addAll(keyspace.flush());
        }
        FBUtilities.waitOnFutures(this.futures);
        return this.replayedCount.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void recover(File file) throws IOException {
        final ReplayFilter replayFilter = ReplayFilter.create();
        CommitLogDescriptor desc = CommitLogDescriptor.fromFileName(file.getName());
        final long segment = desc.id;
        logger.info("Replaying {} (CL version {}, messaging version {})", new Object[]{file.getPath(), desc.getVersion(), desc.getMessagingVersion()});
        RandomAccessReader reader = RandomAccessReader.open(new File(file.getAbsolutePath()));
        try {
            int replayPosition;
            assert (reader.length() <= Integer.MAX_VALUE);
            if (this.globalPosition.segment < segment) {
                replayPosition = 0;
            } else if (this.globalPosition.segment == segment) {
                replayPosition = this.globalPosition.position;
            } else {
                logger.debug("skipping replay of fully-flushed {}", (Object)file);
                return;
            }
            if (logger.isDebugEnabled()) {
                logger.debug("Replaying " + file + " starting at " + replayPosition);
            }
            reader.seek(replayPosition);
            while (!reader.isEOF()) {
                RowMutation rm;
                long claimedCRC32;
                int serializedSize;
                if (logger.isDebugEnabled()) {
                    logger.debug("Reading mutation at " + reader.getFilePointer());
                }
                try {
                    serializedSize = reader.readInt();
                    if (serializedSize == 0) {
                        logger.debug("Encountered end of segment marker at " + reader.getFilePointer());
                        break;
                    }
                    if (serializedSize < 10) {
                        break;
                    }
                    long claimedSizeChecksum = reader.readLong();
                    this.checksum.reset();
                    if (desc.getVersion() < 3) {
                        this.checksum.update(serializedSize);
                    } else {
                        FBUtilities.updateChecksumInt(this.checksum, serializedSize);
                    }
                    if (this.checksum.getValue() != claimedSizeChecksum) {
                        break;
                    }
                    if (serializedSize > this.buffer.length) {
                        this.buffer = new byte[(int)(1.2 * (double)serializedSize)];
                    }
                    reader.readFully(this.buffer, 0, serializedSize);
                    claimedCRC32 = reader.readLong();
                }
                catch (EOFException eof) {
                    break;
                }
                this.checksum.update(this.buffer, 0, serializedSize);
                if (claimedCRC32 != this.checksum.getValue()) continue;
                FastByteArrayInputStream bufIn = new FastByteArrayInputStream(this.buffer, 0, serializedSize);
                try {
                    rm = RowMutation.serializer.deserialize(new DataInputStream(bufIn), desc.getMessagingVersion(), ColumnSerializer.Flag.LOCAL);
                    for (ColumnFamily cf : rm.getColumnFamilies()) {
                        for (Column cell : cf) {
                            cf.getComparator().validate(cell.name());
                        }
                    }
                }
                catch (UnknownColumnFamilyException ex) {
                    if (ex.cfId == null) continue;
                    AtomicInteger i = this.invalidMutations.get(ex.cfId);
                    if (i == null) {
                        i = new AtomicInteger(1);
                        this.invalidMutations.put(ex.cfId, i);
                        continue;
                    }
                    i.incrementAndGet();
                    continue;
                }
                catch (Throwable t) {
                    File f = File.createTempFile("mutation", "dat");
                    try (DataOutputStream out = new DataOutputStream(new FileOutputStream(f));){
                        out.write(this.buffer, 0, serializedSize);
                    }
                    String st = String.format("Unexpected error deserializing mutation; saved to %s and ignored.  This may be caused by replaying a mutation against a table with the same name but incompatible schema.  Exception follows: ", f.getAbsolutePath());
                    logger.error(st, t);
                    continue;
                }
                if (logger.isDebugEnabled()) {
                    logger.debug(String.format("replaying mutation for %s.%s: %s", rm.getKeyspaceName(), ByteBufferUtil.bytesToHex(rm.key()), "{" + StringUtils.join(rm.getColumnFamilies().iterator(), (String)", ") + "}"));
                }
                final long entryLocation = reader.getFilePointer();
                WrappedRunnable runnable = new WrappedRunnable(){

                    @Override
                    public void runMayThrow() throws IOException {
                        if (Schema.instance.getKSMetaData(rm.getKeyspaceName()) == null) {
                            return;
                        }
                        if (CommitLogReplayer.this.pointInTimeExceeded(rm)) {
                            return;
                        }
                        Keyspace keyspace = Keyspace.open(rm.getKeyspaceName());
                        RowMutation newRm = null;
                        for (ColumnFamily columnFamily : replayFilter.filter(rm)) {
                            if (Schema.instance.getCF(columnFamily.id()) == null) continue;
                            ReplayPosition rp = (ReplayPosition)CommitLogReplayer.this.cfPositions.get(columnFamily.id());
                            if (segment <= rp.segment && (segment != rp.segment || entryLocation <= (long)rp.position)) continue;
                            if (newRm == null) {
                                newRm = new RowMutation(rm.getKeyspaceName(), rm.key());
                            }
                            newRm.add(columnFamily);
                            CommitLogReplayer.this.replayedCount.incrementAndGet();
                        }
                        if (newRm != null) {
                            assert (!newRm.isEmpty());
                            Keyspace.open(newRm.getKeyspaceName()).apply(newRm, false);
                            CommitLogReplayer.this.keyspacesRecovered.add(keyspace);
                        }
                    }
                };
                this.futures.add(StageManager.getStage(Stage.MUTATION).submit(runnable));
                if (this.futures.size() <= MAX_OUTSTANDING_REPLAY_COUNT) continue;
                FBUtilities.waitOnFutures(this.futures);
                this.futures.clear();
            }
        }
        finally {
            FileUtils.closeQuietly(reader);
            logger.info("Finished reading " + file);
        }
    }

    protected boolean pointInTimeExceeded(RowMutation frm) {
        long restoreTarget = CommitLog.instance.archiver.restorePointInTime;
        for (ColumnFamily families : frm.getColumnFamilies()) {
            if (CommitLog.instance.archiver.precision.toMillis(families.maxTimestamp()) <= restoreTarget) continue;
            return true;
        }
        return false;
    }

    private static class CustomReplayFilter
    extends ReplayFilter {
        private Multimap<String, String> toReplay;

        public CustomReplayFilter(Multimap<String, String> toReplay) {
            this.toReplay = toReplay;
        }

        @Override
        public Iterable<ColumnFamily> filter(RowMutation rm) {
            final Collection cfNames = this.toReplay.get((Object)rm.getKeyspaceName());
            if (cfNames == null) {
                return Collections.emptySet();
            }
            return Iterables.filter(rm.getColumnFamilies(), (Predicate)new Predicate<ColumnFamily>(){

                public boolean apply(ColumnFamily cf) {
                    return cfNames.contains(cf.metadata().cfName);
                }
            });
        }
    }

    private static class AlwaysReplayFilter
    extends ReplayFilter {
        private AlwaysReplayFilter() {
        }

        @Override
        public Iterable<ColumnFamily> filter(RowMutation rm) {
            return rm.getColumnFamilies();
        }
    }

    private static abstract class ReplayFilter {
        private ReplayFilter() {
        }

        public abstract Iterable<ColumnFamily> filter(RowMutation var1);

        public static ReplayFilter create() {
            if (System.getProperty("cassandra.replayList") == null) {
                return new AlwaysReplayFilter();
            }
            HashMultimap toReplay = HashMultimap.create();
            for (String rawPair : System.getProperty("cassandra.replayList").split(",")) {
                String[] pair = rawPair.trim().split("\\.");
                if (pair.length != 2) {
                    throw new IllegalArgumentException("Each table to be replayed must be fully qualified with keyspace name, e.g., 'system.peers'");
                }
                Keyspace ks = Schema.instance.getKeyspaceInstance(pair[0]);
                if (ks == null) {
                    throw new IllegalArgumentException("Unknown keyspace " + pair[0]);
                }
                if (ks.getColumnFamilyStore(pair[1]) == null) {
                    throw new IllegalArgumentException(String.format("Unknown table %s.%s", pair[0], pair[1]));
                }
                toReplay.put((Object)pair[0], (Object)pair[1]);
            }
            return new CustomReplayFilter((Multimap<String, String>)toReplay);
        }
    }
}

