/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.base.Throwables;
import java.io.Closeable;
import java.io.DataInput;
import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import org.apache.cassandra.db.ArrayBackedSortedColumns;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.OnDiskAtom;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.compaction.CompactionController;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionInterruptedException;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.compaction.LazilyCompactedRow;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableIdentityIterator;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.sstable.SSTableRewriter;
import org.apache.cassandra.io.sstable.SSTableWriter;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.OutputHandler;

public class Scrubber
implements Closeable {
    private final ColumnFamilyStore cfs;
    private final SSTableReader sstable;
    private final File destination;
    private final boolean skipCorrupted;
    private final CompactionController controller;
    private final boolean isCommutative;
    private final int expectedBloomFilterSize;
    private final RandomAccessReader dataFile;
    private final RandomAccessReader indexFile;
    private final ScrubInfo scrubInfo;
    private final boolean isOffline;
    private SSTableReader newSstable;
    private SSTableReader newInOrderSstable;
    private int goodRows;
    private int badRows;
    private int emptyRows;
    private final OutputHandler outputHandler;
    private static final Comparator<Row> rowComparator = new Comparator<Row>(){

        @Override
        public int compare(Row r1, Row r2) {
            return r1.key.compareTo(r2.key);
        }
    };
    private final SortedSet<Row> outOfOrderRows = new TreeSet<Row>(rowComparator);

    public Scrubber(ColumnFamilyStore cfs, SSTableReader sstable, boolean skipCorrupted, boolean isOffline) throws IOException {
        this(cfs, sstable, skipCorrupted, new OutputHandler.LogOutput(), isOffline);
    }

    public Scrubber(ColumnFamilyStore cfs, SSTableReader sstable, boolean skipCorrupted, OutputHandler outputHandler, boolean isOffline) throws IOException {
        this.cfs = cfs;
        this.sstable = sstable;
        this.outputHandler = outputHandler;
        this.skipCorrupted = skipCorrupted;
        this.isOffline = isOffline;
        this.destination = cfs.directories.getDirectoryForNewSSTables();
        if (this.destination == null) {
            throw new IOException("disk full");
        }
        List<SSTableReader> toScrub = Collections.singletonList(sstable);
        this.controller = isOffline ? new ScrubController(cfs) : new CompactionController(cfs, Collections.singleton(sstable), CompactionManager.getDefaultGcBefore(cfs));
        this.isCommutative = cfs.metadata.isCounter();
        this.expectedBloomFilterSize = Math.max(cfs.metadata.getMinIndexInterval(), (int)SSTableReader.getApproximateKeyCount(toScrub));
        this.dataFile = isOffline ? sstable.openDataReader() : sstable.openDataReader(CompactionManager.instance.getRateLimiter());
        this.indexFile = RandomAccessReader.open(new File(sstable.descriptor.filenameFor(Component.PRIMARY_INDEX)));
        this.scrubInfo = new ScrubInfo(this.dataFile, sstable);
    }

    public void scrub() {
        this.outputHandler.output(String.format("Scrubbing %s (%s bytes)", this.sstable, this.dataFile.length()));
        SSTableRewriter writer = new SSTableRewriter(this.cfs, new HashSet<SSTableReader>(Collections.singleton(this.sstable)), this.sstable.maxDataAge, OperationType.SCRUB, this.isOffline);
        try {
            ByteBuffer nextIndexKey = ByteBufferUtil.readWithShortLength(this.indexFile);
            long firstRowPositionFromIndex = this.sstable.metadata.comparator.rowIndexEntrySerializer().deserialize((DataInput)this.indexFile, (Descriptor.Version)this.sstable.descriptor.version).position;
            assert (firstRowPositionFromIndex == 0L) : firstRowPositionFromIndex;
            writer.switchWriter(CompactionManager.createWriter(this.cfs, this.destination, this.expectedBloomFilterSize, this.sstable.getSSTableMetadata().repairedAt, this.sstable));
            DecoratedKey prevKey = null;
            while (!this.dataFile.isEOF()) {
                long dataSizeFromIndex;
                long nextRowPositionFromIndex;
                if (this.scrubInfo.isStopRequested()) {
                    throw new CompactionInterruptedException(this.scrubInfo.getCompactionInfo());
                }
                long rowStart = this.dataFile.getFilePointer();
                this.outputHandler.debug("Reading row at " + rowStart);
                DecoratedKey key = null;
                long dataSize = -1L;
                try {
                    key = this.sstable.partitioner.decorateKey(ByteBufferUtil.readWithShortLength(this.dataFile));
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                }
                ByteBuffer currentIndexKey = nextIndexKey;
                try {
                    nextIndexKey = this.indexFile.isEOF() ? null : ByteBufferUtil.readWithShortLength(this.indexFile);
                    nextRowPositionFromIndex = this.indexFile.isEOF() ? this.dataFile.length() : this.sstable.metadata.comparator.rowIndexEntrySerializer().deserialize((DataInput)this.indexFile, (Descriptor.Version)this.sstable.descriptor.version).position;
                }
                catch (Throwable th) {
                    this.outputHandler.warn("Error reading index file", th);
                    nextIndexKey = null;
                    nextRowPositionFromIndex = this.dataFile.length();
                }
                long dataStart = this.dataFile.getFilePointer();
                long dataStartFromIndex = currentIndexKey == null ? -1L : rowStart + 2L + (long)currentIndexKey.remaining();
                dataSize = dataSizeFromIndex = nextRowPositionFromIndex - dataStartFromIndex;
                String keyName = key == null ? "(unreadable key)" : ByteBufferUtil.bytesToHex(key.getKey());
                this.outputHandler.debug(String.format("row %s is %s bytes", keyName, dataSize));
                assert (currentIndexKey != null || this.indexFile.isEOF());
                try {
                    if (key == null) {
                        throw new IOError(new IOException("Unable to read row key from data file"));
                    }
                    if (dataSize > this.dataFile.length()) {
                        throw new IOError(new IOException("Impossible row size " + dataSize));
                    }
                    SSTableIdentityIterator atoms = new SSTableIdentityIterator(this.sstable, this.dataFile, key, dataSize, true);
                    if (prevKey != null && prevKey.compareTo(key) > 0) {
                        this.saveOutOfOrderRow(prevKey, key, atoms);
                        continue;
                    }
                    LazilyCompactedRow compactedRow = new LazilyCompactedRow(this.controller, Collections.singletonList(atoms));
                    if (writer.tryAppend(compactedRow) == null) {
                        ++this.emptyRows;
                    } else {
                        ++this.goodRows;
                    }
                    prevKey = key;
                    if (key.getKey().equals(currentIndexKey) && dataStart == dataStartFromIndex) continue;
                    this.outputHandler.warn("Index file contained a different key or row size; using key from data file");
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                    this.outputHandler.warn("Error reading row (stacktrace follows):", th);
                    if (!(currentIndexKey == null || key != null && key.getKey().equals(currentIndexKey) && dataStart == dataStartFromIndex && dataSize == dataSizeFromIndex)) {
                        this.outputHandler.output(String.format("Retrying from row index; data is %s bytes starting at %s", dataSizeFromIndex, dataStartFromIndex));
                        key = this.sstable.partitioner.decorateKey(currentIndexKey);
                        try {
                            SSTableIdentityIterator atoms = new SSTableIdentityIterator(this.sstable, this.dataFile, key, dataSize, true);
                            if (prevKey != null && prevKey.compareTo(key) > 0) {
                                this.saveOutOfOrderRow(prevKey, key, atoms);
                                continue;
                            }
                            LazilyCompactedRow compactedRow = new LazilyCompactedRow(this.controller, Collections.singletonList(atoms));
                            if (writer.tryAppend(compactedRow) == null) {
                                ++this.emptyRows;
                            } else {
                                ++this.goodRows;
                            }
                            prevKey = key;
                        }
                        catch (Throwable th2) {
                            this.throwIfFatal(th2);
                            this.throwIfCommutative(key, th2);
                            this.outputHandler.warn("Retry failed too. Skipping to next row (retry's stacktrace follows)", th2);
                            this.dataFile.seek(nextRowPositionFromIndex);
                            ++this.badRows;
                        }
                        continue;
                    }
                    this.throwIfCommutative(key, th);
                    this.outputHandler.warn("Row starting at position " + dataStart + " is unreadable; skipping to next");
                    if (currentIndexKey != null) {
                        this.dataFile.seek(nextRowPositionFromIndex);
                    }
                    ++this.badRows;
                }
            }
            if (!this.outOfOrderRows.isEmpty()) {
                long repairedAt = this.badRows > 0 ? 0L : this.sstable.getSSTableMetadata().repairedAt;
                SSTableWriter inOrderWriter = CompactionManager.createWriter(this.cfs, this.destination, this.expectedBloomFilterSize, repairedAt, this.sstable);
                for (Row row : this.outOfOrderRows) {
                    inOrderWriter.append(row.key, row.cf);
                }
                this.newInOrderSstable = inOrderWriter.closeAndOpenReader(this.sstable.maxDataAge);
                if (!this.isOffline) {
                    this.cfs.getDataTracker().addSSTables(Collections.singleton(this.newInOrderSstable));
                }
                this.outputHandler.warn(String.format("%d out of order rows found while scrubbing %s; Those have been written (in order) to a new sstable (%s)", this.outOfOrderRows.size(), this.sstable, this.newInOrderSstable));
            }
            writer.finish(!this.isOffline, this.badRows > 0 ? 0L : this.sstable.getSSTableMetadata().repairedAt);
            if (!writer.finished().isEmpty()) {
                this.newSstable = writer.finished().get(0);
            }
        }
        catch (Throwable t) {
            writer.abort();
            throw Throwables.propagate((Throwable)t);
        }
        finally {
            this.controller.close();
        }
        if (this.newSstable == null) {
            if (this.badRows > 0) {
                this.outputHandler.warn("No valid rows found while scrubbing " + this.sstable + "; it is marked for deletion now. If you want to attempt manual recovery, you can find a copy in the pre-scrub snapshot");
            } else {
                this.outputHandler.output("Scrub of " + this.sstable + " complete; looks like all " + this.emptyRows + " rows were tombstoned");
            }
        } else {
            this.outputHandler.output("Scrub of " + this.sstable + " complete: " + this.goodRows + " rows in new sstable and " + this.emptyRows + " empty (tombstoned) rows dropped");
            if (this.badRows > 0) {
                this.outputHandler.warn("Unable to recover " + this.badRows + " rows that were skipped.  You can attempt manual recovery from the pre-scrub snapshot.  You can also run nodetool repair to transfer the data from a healthy replica, if any");
            }
        }
    }

    private void saveOutOfOrderRow(DecoratedKey prevKey, DecoratedKey key, SSTableIdentityIterator atoms) {
        this.outputHandler.warn(String.format("Out of order row detected (%s found after %s)", key, prevKey));
        ArrayBackedSortedColumns cf = atoms.getColumnFamily().cloneMeShallow(ArrayBackedSortedColumns.factory, false);
        while (atoms.hasNext()) {
            OnDiskAtom atom = atoms.next();
            cf.addAtom(atom);
        }
        this.outOfOrderRows.add(new Row(key, (ColumnFamily)cf));
    }

    public SSTableReader getNewSSTable() {
        return this.newSstable;
    }

    public SSTableReader getNewInOrderSSTable() {
        return this.newInOrderSstable;
    }

    private void throwIfFatal(Throwable th) {
        if (th instanceof Error && !(th instanceof AssertionError) && !(th instanceof IOError)) {
            throw (Error)th;
        }
    }

    private void throwIfCommutative(DecoratedKey key, Throwable th) {
        if (this.isCommutative && !this.skipCorrupted) {
            this.outputHandler.warn(String.format("An error occurred while scrubbing the row with key '%s'.  Skipping corrupt rows in counter tables will result in undercounts for the affected counters (see CASSANDRA-2759 for more details), so by default the scrub will stop at this point.  If you would like to skip the row anyway and continue scrubbing, re-run the scrub with the --skip-corrupted option.", key));
            throw new IOError(th);
        }
    }

    @Override
    public void close() {
        FileUtils.closeQuietly(this.dataFile);
        FileUtils.closeQuietly(this.indexFile);
    }

    public CompactionInfo.Holder getScrubInfo() {
        return this.scrubInfo;
    }

    private static class ScrubController
    extends CompactionController {
        public ScrubController(ColumnFamilyStore cfs) {
            super(cfs, Integer.MAX_VALUE);
        }

        @Override
        public long maxPurgeableTimestamp(DecoratedKey key) {
            return Long.MIN_VALUE;
        }
    }

    private static class ScrubInfo
    extends CompactionInfo.Holder {
        private final RandomAccessReader dataFile;
        private final SSTableReader sstable;

        public ScrubInfo(RandomAccessReader dataFile, SSTableReader sstable) {
            this.dataFile = dataFile;
            this.sstable = sstable;
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            try {
                return new CompactionInfo(this.sstable.metadata, OperationType.SCRUB, this.dataFile.getFilePointer(), this.dataFile.length());
            }
            catch (Exception e) {
                throw new RuntimeException();
            }
        }
    }
}

