/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.Memtable;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.dht.AbstractBounds;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.metrics.StorageMetrics;
import org.apache.cassandra.notifications.INotificationConsumer;
import org.apache.cassandra.notifications.MemtableRenewedNotification;
import org.apache.cassandra.notifications.SSTableAddedNotification;
import org.apache.cassandra.notifications.SSTableDeletingNotification;
import org.apache.cassandra.notifications.SSTableListChangedNotification;
import org.apache.cassandra.notifications.SSTableRepairStatusChanged;
import org.apache.cassandra.notifications.TruncationNotification;
import org.apache.cassandra.utils.Interval;
import org.apache.cassandra.utils.IntervalTree;
import org.apache.cassandra.utils.concurrent.OpOrder;
import org.apache.cassandra.utils.concurrent.Refs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataTracker {
    private static final Logger logger = LoggerFactory.getLogger(DataTracker.class);
    public final Collection<INotificationConsumer> subscribers = new CopyOnWriteArrayList<INotificationConsumer>();
    public final ColumnFamilyStore cfstore;
    private final AtomicReference<View> view;
    public final boolean loadsstables;

    public DataTracker(ColumnFamilyStore cfstore, boolean loadsstables) {
        this.cfstore = cfstore;
        this.view = new AtomicReference();
        this.loadsstables = loadsstables;
        this.init();
    }

    public Memtable getMemtableFor(OpOrder.Group opGroup, ReplayPosition replayPosition) {
        for (Memtable memtable : this.view.get().liveMemtables) {
            if (!memtable.accepts(opGroup, replayPosition)) continue;
            return memtable;
        }
        throw new AssertionError((Object)this.view.get().liveMemtables.toString());
    }

    public Set<SSTableReader> getSSTables() {
        return this.view.get().sstables;
    }

    public Set<SSTableReader> getUncompactingSSTables() {
        return this.view.get().nonCompactingSStables();
    }

    public Iterable<SSTableReader> getUncompactingSSTables(Iterable<SSTableReader> candidates) {
        final View v = this.view.get();
        return Iterables.filter(candidates, (Predicate)new Predicate<SSTableReader>(){

            public boolean apply(SSTableReader sstable) {
                return !v.compacting.contains(sstable);
            }
        });
    }

    public View getView() {
        return this.view.get();
    }

    public Memtable switchMemtable(boolean truncating) {
        Memtable toFlushMemtable;
        View newView;
        View currentView;
        Memtable newMemtable = new Memtable(this.cfstore);
        do {
            currentView = this.view.get();
            toFlushMemtable = currentView.getCurrentMemtable();
        } while (!this.view.compareAndSet(currentView, newView = currentView.switchMemtable(newMemtable)));
        if (truncating) {
            this.notifyRenewed(newMemtable);
        }
        return toFlushMemtable;
    }

    public void markFlushing(Memtable memtable) {
        View newView;
        View currentView;
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.markFlushing(memtable))) {
        }
    }

    public void replaceFlushed(Memtable memtable, SSTableReader sstable) {
        View newView;
        View currentView;
        if (!this.cfstore.isValid()) {
            View newView2;
            View currentView2;
            do {
                currentView2 = this.view.get();
                newView2 = currentView2.replaceFlushed(memtable, sstable);
                if (sstable == null) continue;
                newView2 = newView2.replace(Arrays.asList(sstable), Collections.emptyList());
            } while (!this.view.compareAndSet(currentView2, newView2));
            return;
        }
        if (sstable != null) {
            this.maybeIncrementallyBackup(sstable);
        }
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.replaceFlushed(memtable, sstable))) {
        }
        if (sstable != null) {
            this.addNewSSTablesSize(Arrays.asList(sstable));
            this.notifyAdded(sstable);
        }
    }

    public void maybeIncrementallyBackup(SSTableReader sstable) {
        if (!DatabaseDescriptor.isIncrementalBackupsEnabled()) {
            return;
        }
        File backupsDir = Directories.getBackupsDirectory(sstable.descriptor);
        sstable.createLinks(FileUtils.getCanonicalPath(backupsDir));
    }

    public boolean markCompacting(Iterable<SSTableReader> sstables) {
        return this.markCompacting(sstables, false, false);
    }

    public boolean markCompacting(Iterable<SSTableReader> sstables, boolean newTables, boolean offline) {
        View newView;
        View currentView;
        assert (sstables != null && !Iterables.isEmpty(sstables));
        do {
            currentView = this.view.get();
            if (Iterables.any(sstables, (Predicate)Predicates.in(currentView.compacting))) {
                return false;
            }
            Predicate<SSTableReader> live = new Predicate<SSTableReader>(){

                public boolean apply(SSTableReader sstable) {
                    return currentView.sstablesMap.get(sstable) == sstable && !sstable.isMarkedCompacted();
                }
            };
            if (newTables) {
                assert (!Iterables.any(sstables, (Predicate)Predicates.in(currentView.sstables)));
                continue;
            }
            if (offline || Iterables.all(sstables, (Predicate)live)) continue;
            return false;
        } while (!this.view.compareAndSet(currentView, newView = currentView.markCompacting(sstables)));
        return true;
    }

    public void unmarkCompacting(Iterable<SSTableReader> unmark) {
        View newView;
        View currentView;
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.unmarkCompacting(unmark))) {
        }
        if (!this.cfstore.isValid()) {
            this.unreferenceSSTables();
        }
    }

    public void markObsolete(Collection<SSTableReader> sstables, OperationType compactionType) {
        this.removeSSTablesFromTracker(sstables);
        this.releaseReferences(sstables, false);
        this.notifySSTablesChanged(sstables, Collections.emptyList(), compactionType);
    }

    public void markCompactedSSTablesReplaced(Collection<SSTableReader> oldSSTables, Collection<SSTableReader> allReplacements, OperationType compactionType) {
        this.removeSSTablesFromTracker(oldSSTables);
        this.releaseReferences(oldSSTables, false);
        this.notifySSTablesChanged(oldSSTables, allReplacements, compactionType);
        this.addNewSSTablesSize(allReplacements);
    }

    public void addInitialSSTables(Collection<SSTableReader> sstables) {
        this.addSSTablesToTracker(sstables);
    }

    public void addSSTables(Collection<SSTableReader> sstables) {
        this.addSSTablesToTracker(sstables);
        for (SSTableReader sstable : sstables) {
            this.maybeIncrementallyBackup(sstable);
            this.notifyAdded(sstable);
        }
    }

    public void replaceWithNewInstances(Collection<SSTableReader> toReplace, Collection<SSTableReader> replaceWith) {
        this.replaceReaders(toReplace, replaceWith, true);
    }

    public void replaceEarlyOpenedFiles(Collection<SSTableReader> toReplace, Collection<SSTableReader> replaceWith) {
        for (SSTableReader s : toReplace) {
            assert (s.openReason == SSTableReader.OpenReason.EARLY);
        }
        this.replaceReaders(toReplace, replaceWith, false);
    }

    public void unreferenceSSTables() {
        Sets.SetView<SSTableReader> notCompacting;
        View newView;
        View currentView;
        do {
            currentView = this.view.get();
            if (currentView.compacting.isEmpty()) continue;
            logger.error("Set of compacting sstables is non-empty when invalidating sstables {}", currentView.compacting);
        } while (!this.view.compareAndSet(currentView, newView = currentView.replace((Collection<SSTableReader>)(notCompacting = currentView.nonCompactingSStables()), (Iterable<SSTableReader>)Collections.emptySet())));
        if (notCompacting.isEmpty()) {
            return;
        }
        this.notifySSTablesChanged((Collection<SSTableReader>)notCompacting, (Collection<SSTableReader>)Collections.emptySet(), OperationType.UNKNOWN);
        this.removeOldSSTablesSize((Iterable<SSTableReader>)notCompacting);
        this.releaseReferences((Iterable<SSTableReader>)notCompacting, true);
    }

    void removeUnreadableSSTables(File directory) {
        View newView;
        View currentView;
        HashSet<SSTableReader> remaining = new HashSet<SSTableReader>();
        do {
            currentView = this.view.get();
            for (SSTableReader r : currentView.nonCompactingSStables()) {
                if (r.descriptor.directory.equals(directory)) continue;
                remaining.add(r);
            }
            if (remaining.size() != currentView.nonCompactingSStables().size()) continue;
            return;
        } while (!this.view.compareAndSet(currentView, newView = currentView.replace(currentView.sstables, remaining)));
        for (SSTableReader sstable : currentView.sstables) {
            if (remaining.contains(sstable)) continue;
            sstable.selfRef().release();
        }
        this.notifySSTablesChanged(remaining, Collections.emptySet(), OperationType.UNKNOWN);
    }

    void init() {
        this.view.set(new View((List<Memtable>)ImmutableList.of((Object)new Memtable(this.cfstore)), (List<Memtable>)ImmutableList.of(), Collections.emptyMap(), Collections.emptySet(), Collections.emptySet(), SSTableIntervalTree.empty()));
    }

    private void replaceReaders(Collection<SSTableReader> oldSSTables, Collection<SSTableReader> newSSTables, boolean notify) {
        View newView;
        View currentView;
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.replace(oldSSTables, newSSTables))) {
        }
        if (!oldSSTables.isEmpty() && notify) {
            this.notifySSTablesChanged(oldSSTables, newSSTables, OperationType.UNKNOWN);
        }
        for (SSTableReader sstable : newSSTables) {
            sstable.setTrackedBy(this);
        }
        Refs.release(Refs.selfRefs(oldSSTables));
    }

    private void removeSSTablesFromTracker(Collection<SSTableReader> oldSSTables) {
        View newView;
        View currentView;
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.replace(oldSSTables, Collections.emptyList()))) {
        }
        this.removeOldSSTablesSize(oldSSTables);
    }

    private void addSSTablesToTracker(Collection<SSTableReader> sstables) {
        View newView;
        View currentView;
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.replace(Collections.emptyList(), sstables))) {
        }
        this.addNewSSTablesSize(sstables);
    }

    private void addNewSSTablesSize(Iterable<SSTableReader> newSSTables) {
        for (SSTableReader sstable : newSSTables) {
            if (logger.isDebugEnabled()) {
                logger.debug(String.format("adding %s to list of files tracked for %s.%s", sstable.descriptor, this.cfstore.keyspace.getName(), this.cfstore.name));
            }
            long size = sstable.bytesOnDisk();
            StorageMetrics.load.inc(size);
            this.cfstore.metric.liveDiskSpaceUsed.inc(size);
            this.cfstore.metric.totalDiskSpaceUsed.inc(size);
            sstable.setTrackedBy(this);
        }
    }

    private void removeOldSSTablesSize(Iterable<SSTableReader> oldSSTables) {
        for (SSTableReader sstable : oldSSTables) {
            if (logger.isDebugEnabled()) {
                logger.debug(String.format("removing %s from list of files tracked for %s.%s", sstable.descriptor, this.cfstore.keyspace.getName(), this.cfstore.name));
            }
            long size = sstable.bytesOnDisk();
            StorageMetrics.load.dec(size);
            this.cfstore.metric.liveDiskSpaceUsed.dec(size);
        }
    }

    private void releaseReferences(Iterable<SSTableReader> oldSSTables, boolean tolerateCompacted) {
        for (SSTableReader sstable : oldSSTables) {
            boolean firstToCompact = sstable.markObsolete();
            assert (tolerateCompacted || firstToCompact) : sstable + " was already marked compacted";
            sstable.selfRef().release();
        }
    }

    public void spaceReclaimed(long size) {
        this.cfstore.metric.totalDiskSpaceUsed.dec(size);
    }

    public long estimatedKeys() {
        long n = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            n += sstable.estimatedKeys();
        }
        return n;
    }

    public int getMeanColumns() {
        long sum = 0L;
        long count = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            long n = sstable.getEstimatedColumnCount().count();
            sum += sstable.getEstimatedColumnCount().mean() * n;
            count += n;
        }
        return count > 0L ? (int)(sum / count) : 0;
    }

    public double getDroppableTombstoneRatio() {
        double allDroppable = 0.0;
        long allColumns = 0L;
        int localTime = (int)(System.currentTimeMillis() / 1000L);
        for (SSTableReader sstable : this.getSSTables()) {
            allDroppable += sstable.getDroppableTombstonesBefore(localTime - sstable.metadata.getGcGraceSeconds());
            allColumns += sstable.getEstimatedColumnCount().mean() * sstable.getEstimatedColumnCount().count();
        }
        return allColumns > 0L ? allDroppable / (double)allColumns : 0.0;
    }

    public void notifySSTablesChanged(Collection<SSTableReader> removed, Collection<SSTableReader> added, OperationType compactionType) {
        SSTableListChangedNotification notification = new SSTableListChangedNotification(added, removed, compactionType);
        for (INotificationConsumer subscriber : this.subscribers) {
            subscriber.handleNotification(notification, this);
        }
    }

    public void notifyAdded(SSTableReader added) {
        SSTableAddedNotification notification = new SSTableAddedNotification(added);
        for (INotificationConsumer subscriber : this.subscribers) {
            subscriber.handleNotification(notification, this);
        }
    }

    public void notifySSTableRepairedStatusChanged(Collection<SSTableReader> repairStatusesChanged) {
        SSTableRepairStatusChanged notification = new SSTableRepairStatusChanged(repairStatusesChanged);
        for (INotificationConsumer subscriber : this.subscribers) {
            subscriber.handleNotification(notification, this);
        }
    }

    public void notifyDeleting(SSTableReader deleting) {
        SSTableDeletingNotification notification = new SSTableDeletingNotification(deleting);
        for (INotificationConsumer subscriber : this.subscribers) {
            subscriber.handleNotification(notification, this);
        }
    }

    public void notifyRenewed(Memtable renewed) {
        MemtableRenewedNotification notification = new MemtableRenewedNotification(renewed);
        for (INotificationConsumer subscriber : this.subscribers) {
            subscriber.handleNotification(notification, this);
        }
    }

    public void notifyTruncated(long truncatedAt) {
        TruncationNotification notification = new TruncationNotification(truncatedAt);
        for (INotificationConsumer subscriber : this.subscribers) {
            subscriber.handleNotification(notification, this);
        }
    }

    public void subscribe(INotificationConsumer consumer) {
        this.subscribers.add(consumer);
    }

    public void unsubscribe(INotificationConsumer consumer) {
        this.subscribers.remove(consumer);
    }

    public static SSTableIntervalTree buildIntervalTree(Iterable<SSTableReader> sstables) {
        return new SSTableIntervalTree(DataTracker.buildIntervals(sstables));
    }

    public static List<Interval<RowPosition, SSTableReader>> buildIntervals(Iterable<SSTableReader> sstables) {
        ArrayList<Interval<RowPosition, SSTableReader>> intervals = new ArrayList<Interval<RowPosition, SSTableReader>>(Iterables.size(sstables));
        for (SSTableReader sstable : sstables) {
            intervals.add(Interval.create(sstable.first, sstable.last, sstable));
        }
        return intervals;
    }

    public Set<SSTableReader> getCompacting() {
        return this.getView().compacting;
    }

    public static class View {
        private final List<Memtable> liveMemtables;
        private final List<Memtable> flushingMemtables;
        public final Set<SSTableReader> compacting;
        public final Set<SSTableReader> sstables;
        public final Map<SSTableReader, SSTableReader> sstablesMap;
        public final Set<SSTableReader> shadowed;
        public final SSTableIntervalTree intervalTree;

        View(List<Memtable> liveMemtables, List<Memtable> flushingMemtables, Map<SSTableReader, SSTableReader> sstables, Set<SSTableReader> compacting, Set<SSTableReader> shadowed, SSTableIntervalTree intervalTree) {
            this.shadowed = shadowed;
            assert (liveMemtables != null);
            assert (flushingMemtables != null);
            assert (sstables != null);
            assert (compacting != null);
            assert (intervalTree != null);
            this.liveMemtables = liveMemtables;
            this.flushingMemtables = flushingMemtables;
            this.sstablesMap = sstables;
            this.sstables = this.sstablesMap.keySet();
            this.compacting = compacting;
            this.intervalTree = intervalTree;
        }

        public Memtable getOldestMemtable() {
            if (!this.flushingMemtables.isEmpty()) {
                return this.flushingMemtables.get(0);
            }
            return this.liveMemtables.get(0);
        }

        public Memtable getCurrentMemtable() {
            return this.liveMemtables.get(this.liveMemtables.size() - 1);
        }

        public Iterable<Memtable> getMemtablesPendingFlush() {
            if (this.liveMemtables.size() == 1) {
                return this.flushingMemtables;
            }
            return Iterables.concat(this.liveMemtables.subList(0, 1), this.flushingMemtables);
        }

        public Iterable<Memtable> getAllMemtables() {
            return Iterables.concat(this.flushingMemtables, this.liveMemtables);
        }

        public Sets.SetView<SSTableReader> nonCompactingSStables() {
            return Sets.difference((Set)ImmutableSet.copyOf(this.sstables), this.compacting);
        }

        View switchMemtable(Memtable newMemtable) {
            ImmutableList newLiveMemtables = ImmutableList.builder().addAll(this.liveMemtables).add((Object)newMemtable).build();
            return new View((List<Memtable>)newLiveMemtables, this.flushingMemtables, this.sstablesMap, this.compacting, this.shadowed, this.intervalTree);
        }

        View markFlushing(Memtable toFlushMemtable) {
            List<Memtable> live = this.liveMemtables;
            List<Memtable> flushing = this.flushingMemtables;
            int i = live.indexOf(toFlushMemtable);
            assert (i < live.size() - 1);
            ImmutableList newLive = ImmutableList.builder().addAll(live.subList(0, i)).addAll(live.subList(i + 1, live.size())).build();
            for (i = flushing.size(); i > 0 && flushing.get(i - 1).creationTime() > toFlushMemtable.creationTime(); --i) {
            }
            ImmutableList newFlushing = ImmutableList.builder().addAll(flushing.subList(0, i)).add((Object)toFlushMemtable).addAll(flushing.subList(i, flushing.size())).build();
            return new View((List<Memtable>)newLive, (List<Memtable>)newFlushing, this.sstablesMap, this.compacting, this.shadowed, this.intervalTree);
        }

        View replaceFlushed(Memtable flushedMemtable, SSTableReader newSSTable) {
            int index = this.flushingMemtables.indexOf(flushedMemtable);
            ImmutableList newQueuedMemtables = ImmutableList.builder().addAll(this.flushingMemtables.subList(0, index)).addAll(this.flushingMemtables.subList(index + 1, this.flushingMemtables.size())).build();
            ImmutableMap newSSTables = this.sstablesMap;
            SSTableIntervalTree intervalTree = this.intervalTree;
            if (newSSTable != null) {
                assert (!this.sstables.contains(newSSTable));
                assert (!this.shadowed.contains(newSSTable));
                newSSTables = ImmutableMap.builder().putAll(this.sstablesMap).put((Object)newSSTable, (Object)newSSTable).build();
                intervalTree = DataTracker.buildIntervalTree(newSSTables.keySet());
            }
            return new View(this.liveMemtables, (List<Memtable>)newQueuedMemtables, (Map<SSTableReader, SSTableReader>)newSSTables, this.compacting, this.shadowed, intervalTree);
        }

        View replace(Collection<SSTableReader> oldSSTables, Iterable<SSTableReader> replacements) {
            ImmutableSet oldSet = ImmutableSet.copyOf(oldSSTables);
            int newSSTablesSize = this.shadowed.size() + this.sstables.size() - oldSSTables.size() + Iterables.size(replacements);
            assert (newSSTablesSize >= Iterables.size(replacements)) : String.format("Incoherent new size %d replacing %s by %s in %s", newSSTablesSize, oldSSTables, replacements, this);
            ImmutableMap newSSTables = new HashMap(newSSTablesSize);
            ImmutableSet newShadowed = new HashSet(this.shadowed.size());
            for (SSTableReader sstable : this.sstables) {
                if (oldSet.contains((Object)sstable)) continue;
                newSSTables.put(sstable, sstable);
            }
            for (SSTableReader sstable : this.shadowed) {
                if (oldSet.contains((Object)sstable)) continue;
                newShadowed.add(sstable);
            }
            for (SSTableReader replacement : replacements) {
                if (replacement.openReason == SSTableReader.OpenReason.SHADOWED) {
                    newShadowed.add(replacement);
                    continue;
                }
                newSSTables.put(replacement, replacement);
            }
            assert (newSSTables.size() + newShadowed.size() == newSSTablesSize) : String.format("Expecting new size of %d, got %d while replacing %s by %s in %s", newSSTablesSize, newSSTables.size() + newShadowed.size(), oldSSTables, replacements, this);
            newShadowed = ImmutableSet.copyOf(newShadowed);
            newSSTables = ImmutableMap.copyOf(newSSTables);
            SSTableIntervalTree intervalTree = DataTracker.buildIntervalTree(newSSTables.keySet());
            return new View(this.liveMemtables, this.flushingMemtables, (Map<SSTableReader, SSTableReader>)newSSTables, this.compacting, (Set<SSTableReader>)newShadowed, intervalTree);
        }

        View markCompacting(Iterable<SSTableReader> tomark) {
            ImmutableSet compactingNew = ImmutableSet.builder().addAll(this.compacting).addAll(tomark).build();
            return new View(this.liveMemtables, this.flushingMemtables, this.sstablesMap, (Set<SSTableReader>)compactingNew, this.shadowed, this.intervalTree);
        }

        View unmarkCompacting(Iterable<SSTableReader> tounmark) {
            ImmutableSet compactingNew = ImmutableSet.copyOf((Collection)Sets.difference(this.compacting, (Set)ImmutableSet.copyOf(tounmark)));
            return new View(this.liveMemtables, this.flushingMemtables, this.sstablesMap, (Set<SSTableReader>)compactingNew, this.shadowed, this.intervalTree);
        }

        public String toString() {
            return String.format("View(pending_count=%d, sstables=%s, compacting=%s)", this.liveMemtables.size() + this.flushingMemtables.size() - 1, this.sstables, this.compacting);
        }

        public List<SSTableReader> sstablesInBounds(AbstractBounds<RowPosition> rowBounds) {
            if (this.intervalTree.isEmpty()) {
                return Collections.emptyList();
            }
            RowPosition stopInTree = ((RowPosition)rowBounds.right).isMinimum() ? (RowPosition)this.intervalTree.max() : (RowPosition)rowBounds.right;
            return this.intervalTree.search(Interval.create(rowBounds.left, stopInTree));
        }
    }

    public static class SSTableIntervalTree
    extends IntervalTree<RowPosition, SSTableReader, Interval<RowPosition, SSTableReader>> {
        private static final SSTableIntervalTree EMPTY = new SSTableIntervalTree(null);

        private SSTableIntervalTree(Collection<Interval<RowPosition, SSTableReader>> intervals) {
            super(intervals);
        }

        public static SSTableIntervalTree empty() {
            return EMPTY;
        }
    }
}

